/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.internal;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.GuardMaintenanceService;
import com.openexchange.guard.UserTestResult;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guestreset.GuestResetService;
import com.openexchange.guard.guestupgrade.GuestUpgradeService;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.MailResolver;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.support.SupportService;
import com.openexchange.guard.support.SupportServiceImpl;

/**
 * {@link GuardMaintenanceServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardMaintenanceServiceImpl implements GuardMaintenanceService {

    private static final Logger LOG = LoggerFactory.getLogger(GuardMaintenanceServiceImpl.class);

    /**
     * Initialises a new {@link GuardMaintenanceServiceImpl}.
     */
    public GuardMaintenanceServiceImpl() {
        super();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.GuardMaintenanceService#test(java.lang.String)
     */
    @Override
    public JsonObject test(String mail) throws OXException {
        final MailResolver mailResolver = new MailResolver();
        JsonObject userdat;
        try {
            userdat = mailResolver.resolveUser(mail);
            if (userdat.has(mail)) {
                return userdat;
            } else {
                throw GuardMaintenanceExceptionCodes.UNABLE_TO_RESOLVE_MAIL_ADDRESS.create(mail);
            }
        } catch (final Exception e) {
            LOG.error("Error getting email from REST API", e);
            throw GuardMaintenanceExceptionCodes.UNABLE_TO_RESOLVE_MAIL_ADDRESS.create(e, mail);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.GuardMaintenanceService#reset(java.lang.String)
     */
    @Override
    public void reset(String mail) throws OXException {
        GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
        validatorService.validate(mail);
        try {
            this.resetPass(mail);
        } catch (final Exception e) {
            throw GuardMaintenanceExceptionCodes.FAILED_RESETTING_PASSWORD.create(e, mail);
        }
    }


    /*
     * Reset password. Used from command line tools only
     */
    private void resetPass(String email) throws Exception {
        try {
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys keys = keyService.getKeys(email);
            if (keys == null) {
                throw GuardMaintenanceExceptionCodes.UNABLE_TO_RESOLVE_MAIL_ADDRESS.create(email);
            }
            if (keys.getContextid() < 0) {// Guest
                int templid = 0;
                try {
                    GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
                    templid = guardConfigService.getIntProperty(GuardProperty.templateID, keys.getUserid(), keys.getContextid());
                } catch (Exception e) {
                    LOG.error("problem getting template id for reset password email");
                }
                GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
                String defaultLanguage = configService.getProperty(GuardProperty.defaultLanguage);

                reset(email, keyService, keys, templid, defaultLanguage);
                LOG.info("Reset sent for email {}", email);
            } else {
                new Auth().resetPass(keys.getUserid(), keys.getContextid(), null, null, null, false);
            }
        } catch (Exception e) {
            LOG.error("Error resetting password for " + email, e);
            throw e;
        }
    }

    protected void reset(String email, GuardKeyService keyService, GuardKeys keys, int templid, String defaultLanguage) throws OXException {
        boolean recoveryEnabled = keyService.isRecoveryEnabled(keys.getUserid(), keys.getContextid());
        if (!recoveryEnabled) {
            LOG.info("Failed to reset password for email address {} as recovery is not enabled/available.", email);
            throw GuardCoreExceptionCodes.DISABLED_ERROR.create("reset password");
        }

        String newpass = keyService.generatePassword();
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        JsonObject mail = mailCreatorService.getResetEmail(email, mailCreatorService.getFromAddress(email, email, keys.getUserid(), keys.getContextid()), newpass, defaultLanguage, templid, "", 0, 0);

        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        guardNotificationService.send(mail);

        keyService.resetPassword(email, newpass);
    }


    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.GuardMaintenanceService#removePIN(java.lang.String)
     */
    @Override
    public void removePIN(String mail) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        keyService.removePin(mail);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.GuardMaintenanceService#upgrade(java.lang.String)
     */
    @Override
    public void upgrade(String mail) throws OXException {
        GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
        validatorService.validate(mail);

        try {
            GuestUpgradeService upgradeService = Services.getService(GuestUpgradeService.class);
            upgradeService.upgrade(mail);
        } catch (final Exception e) {
            throw GuardMaintenanceExceptionCodes.FAILED_UPGRADING_TO_OX_ACCOUNT.create(e, mail);
        }
    }

    @Override
    public void delete(String email) throws OXException {
    	GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
        validatorService.validate(email);
        	SupportService svc = new SupportServiceImpl();
        	svc.deleteUser(email);
    }

    @Override
    public void resetGuest(String email, boolean force) throws OXException {
        GuestResetService resetService = Services.getService(GuestResetService.class);
        resetService.reset(email, 0, "", force, null);
    }

}
