/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.internal;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.context.ContextService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contexts.impl.ContextExceptionCodes;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.common.util.LocaleUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProductName;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guestupgrade.GuestUpgradeService;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.logging.GuardLogProperties;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.GuardUserService;

public class Auth {

    private static Logger LOG = LoggerFactory.getLogger(Auth.class);

    public void createKeys(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {
            LOG.error("Create keys but not logged into UI");
            response.setStatus(550);
            return;
        }
        LOG.debug("Starting to create master key");
        EmailStorage emailStorage = Services.getService(EmailStorage.class);
        String language = JsonUtil.getStringFromJson(json, "language");
        Locale locale = LocaleUtil.getLocalFor(language);
        String password = JsonUtil.getStringFromJson(json, "password", true);

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        KeyCreationService keyCreationService = Services.getService(KeyCreationService.class);
        GuardKeys keys = keyService.getKeys(userid, cid);
        KeyRecoveryService recoveryService = Services.getService(KeyRecoveryService.class);
        if (keys == null) {// make sure already don'e exist
            String email = emailStorage.getEmail(userid, cid);
            // Check we don't already have keys for that email
            GuardKeys prevKey = keyService.getKeys(email);
            if (prevKey != null) {
                LOG.error("Attempting to create keys that exist in other context");
                ServletUtils.sendNotAcceptable(response, "Existing");
                return;
            }
            String name = email;
            if (json.get("name") != null) {
                name = json.get("name").getAsString();
            }
            name =  sanitize(name);


            keys = keyCreationService.create(userid, cid, name, email, password, locale, true, recoveryService.checkCreateRecovery(cid, userid));
            keys = keyService.storeKeys(keys, true);

            if (keys == null) {
                LOG.debug("Failed to create master key");
                response.getWriter().write(" {\"result\":\"fail \"} ");
            } else {
                if (json.has("email") && !json.get("email").getAsString().isEmpty()) {
                    String recEmail = new EmailValidator().assertInput(json.get("email").getAsString(), "email");
                    keyService.storeQuestion(userid, cid, "e", recEmail);
                }
                LOG.debug("Created master key for user");
                response.getWriter().write(" {\"result\":\"ok\"} ");
            }
        } else {// Keys already exist, adding additional
            boolean adding = ServletUtils.getBooleanParameter(request, "add", false);
            if (adding) {
                String email = emailStorage.getEmail(userid, cid);
                String name = email;
                if (json.get("name") != null) {
                    name = json.get("name").getAsString();
                }
                name = sanitize(name);
                GuardKeys newkey = keyCreationService.create(userid, cid, name, email, password, locale, true, recoveryService.checkCreateRecovery(cid, userid));
                keyService.addNewKey(newkey);
                keyService.updateSettingsForUser(keys);
                keyService.updateAnswerQuestionForUser(keys);
                LOG.info("Added additional key for user");
                ServletUtils.sendHtmlOK(response, " {\"result\":\"ok\"} ");
            } else {
                LOG.info("Fail creating key, already exists and add flag not set");
                ServletUtils.sendNotAcceptable(response, "Existing");
                return;
            }

        }
    }

    private String sanitize(String data) throws OXException {
        GuardRatifierService ratifierService = Services.getService(GuardRatifierService.class);
        return ratifierService.normalise(data);
    }

    /**
     * Change password
     *
     * @param request
     * @param response
     * @param cookie
     * @return
     * @throws OXException
     * @throws Exception
     */
    public void changepass(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws OXException {
            JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
            // if ox member, verify actually logged in to ui
            if (cid > 0) {
                Api ap = new Api(cookie, request);
                if (ap.verifyLogin() == false) {
                    response.setStatus(550);
                    return;
                }
            }
            String newpass = JsonUtil.getStringFromJson(json, "newpass", true).trim();
            String question = "";
            if (json.has("question")) {// Sent with new question and answer?
                question = json.get("question").getAsString();
            }
            String answer = "";
            if (json.has("answer")) {
                answer = json.get("answer").getAsString();
            }
            String email = "";
            if (json.has("email")) {
                if (!json.get("email").getAsString().equals("")) {
                    answer = new EmailValidator().assertInput(json.get("email").getAsString(), "email");
                }
                if (!answer.equals("")) {
                    question = "e";
                }
            }
            String oldpass = "";
            oldpass = JsonUtil.getStringFromJson(json, "oldpass", true);
            PasswordChangeService passService = Services.getService(PasswordChangeService.class);
            String result = passService.changePassword(userid, cid, email, oldpass, newpass, question, answer, cookie.getJSESSIONID());
            ServletUtils.sendOK(response, "applicaton/json", result);

    }

    /**
     * Update secondary email address after verifying OxGuard password
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void changeSecondaryEmail(HttpServletRequest request, HttpServletResponse response, int userId, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        int badPasswordCount = configService.getIntProperty(GuardProperty.badPasswordCount);
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            ServletUtils.sendNotAcceptable(response, "Not logged in");
            return;
        }
        String password = JsonUtil.getStringFromJson(json, "password", true);
        final String newemail = new EmailValidator().assertInput(JsonUtil.getStringFromJson(json, "email", true), "email");
        GuardKeyService keyService = Services.getService(GuardKeyService.class);

        if (antiabuse.blockLogin(userId, cid,  password, ServletUtils.getClientIP(request))) {
            ServletUtils.sendNotAcceptable(response, "Lockout");
            return;
        }
        GuardKeys keys = keyService.getKeys(userId, cid);
        if (PGPUtil.verifyPassword(keys.getPGPSecretKeyRing(), password, keys.getSalt())) {
            if (updateEmail(userId, cid, newemail)) {
                ServletUtils.sendHtmlOK(response, "OK");
                antiabuse.reportLogin(true, userId, cid, password, ServletUtils.getClientIP(request));
            } else {
                LOG.info("Problem updating secondary email");
                ServletUtils.sendNotAcceptable(response, "Fail");
            }
        } else {
            antiabuse.reportLogin(false, userId, cid, password, ServletUtils.getClientIP(request));
            LOG.info("Bad password for changing secondary email");
            ServletUtils.sendNotAcceptable(response, "Bad password");
            return;
        }

    }

    public String getSecondaryEmail(int userid, int cid) throws Exception {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        List<GuardKeys> userKeys = ogKeyTableStorage.getKeysForUser(userid, cid);
        if (userKeys.size() > 0) {
            String encr_answer = userKeys.get(0).getAnswer();
            if (encr_answer == null) {
                return null;
            }
            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
            String rc = Services.getService(GuardMasterKeyService.class).getMasterKey().getRC();
            return cipherService.decrypt(encr_answer, rc, "e");
        }
        return null;
    }

    private boolean updateEmail(int userid, int cid, String email) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        return (keyService.storeQuestion(userid, cid, "e", email));
    }

    /**
     * Delete password recovery for the user
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void deleteRecovery(HttpServletRequest request, HttpServletResponse response, int userId, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(cookie, request);
        // verify login
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
            return;
        }
        String password = JsonUtil.getStringFromJson(json, "password", true);

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys key = keyService.getKeys(userId, cid);

        // verify password
        if (!PGPUtil.verifyPassword(key.getPGPSecretKeyRing(), password, key.getSalt())) {
            ServletUtils.sendNotAcceptable(response, "Bad password");

            return;
        }

        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.deleteRecovery(key);
        ServletUtils.sendHtmlOK(response, "OK");
    }

    public void resetPass(int userid, int cid, HttpServletRequest request, HttpServletResponse response, OxCookie cookie, boolean web) throws Exception {
        int templid = 0;
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            templid = guardConfigService.getIntProperty(GuardProperty.templateID, userid, cid);
        } catch (Exception e) {
            LOG.error("problem getting template id for reset password email");
        }
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        String lang = request == null ? configService.getProperty(GuardProperty.defaultLanguage) : request.getParameter("lang");
        List<GuardKeys> keysForUser = ogKeyTableStorage.getKeysForUser(userid, cid);

        if (keysForUser.size() > 0) {
            GuardKeys key = keysForUser.get(0);

            String encr_answer = key.getAnswer();
            String userEmail = key.getEmail();
            boolean secondary = true;
            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
            // Mailing to secondary email address if web.  If from command line, then email to primary
            String rc = Services.getService(GuardMasterKeyService.class).getMasterKey().getRC();
            String secondEmail = web ? cipherService.decrypt(encr_answer, rc, "e") : userEmail;
            if ((secondEmail == null) || secondEmail.equals("") || secondEmail.equals(userEmail)) {
                secondEmail = userEmail;
                secondary = false;
            }
            GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
            validatorService.validate(secondEmail);
            String hostname = (request == null ? "" : request.getServerName());
            reset(userid, cid, response, web, templid, lang, userEmail, secondEmail, hostname);

            if (web) {
                GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
                antiabuse.removeBad(cookie.getJSESSIONID());
                antiabuse.removeBad(userid + "-" + cid);
                if (secondary) {
                    ServletUtils.sendHtmlOK(response, "ok");
                } else {
                    ServletUtils.sendHtmlOK(response, "primary");
                }
            } else {
                System.out.println("OK");
            }
        } else {// Bad secondary email
            if (web) {
                ServletUtils.sendHtmlOK(response, "NoSecondary");
            } else {
                System.out.println("No Secondary Email");
            }
            return;
        }
    }

    protected void reset(int userid, int cid, HttpServletResponse response, boolean web, int templid, String lang, String userEmail, String secondEmail, String host) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        boolean recoveryEnabled = keyService.isRecoveryEnabled(userid, cid);
        if (!recoveryEnabled) {
            if (web) {
                LOG.info("Unable to recover password due to no recovery");
                ServletUtils.sendHtmlOK(response, "NoRecovery");
            } else {
                System.out.println("No recovery available");
            }
            throw GuardCoreExceptionCodes.DISABLED_ERROR.create("reset password");
        }
        String newpass = keyService.generatePassword();

        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        JsonObject mail = mailCreatorService.getResetEmail(secondEmail, mailCreatorService.getFromAddress(userEmail, userEmail, userid, cid), newpass, lang, templid, host, userid, cid);
        guardNotificationService.send(mail);

        keyService.resetPassword(userEmail, newpass);
    }

    /**
     * Authorize login from guest account
     *
     * @param request
     * @param response
     * @throws OXException
     */
    public void guestlogin(HttpServletRequest request, HttpServletResponse response) throws OXException {
        JsonObject login_json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        response.setContentType("application/json");
        // response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String sessionID = CipherUtil.getUUID();
        String cookieID = CipherUtil.getUUID();
        Cookie cookie = new Cookie("OxReaderID", cookieID);
        //cookie.setHttpOnly(true);
        cookie.setPath("/");
        response.addCookie(cookie);
        String username = JsonUtil.getStringFromJson(login_json, "username", true);
        if(username.isEmpty()) {
            throw GuardCoreExceptionCodes.JSON_PARAMETER_MISSING.create(username);
        }
        String pin = login_json.get("pin") == null ? null : login_json.get("pin").getAsString();
        try {

            String password = JsonUtil.getStringFromJson(login_json, "password", true);
            GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
            GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
            if (antiabuse.blockLogin(username, password, ServletUtils.getClientIP(request))) {
                response.getWriter().write("{\"auth\":\"lockout\", \"time:\": " + configService.getIntProperty(GuardProperty.badMinuteLock) + "}");
                LOG.info("Guest auth lockout for " + username);
                return;
            }


            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys keys = keyService.getKeys(username);
            if (keys == null) {
                response.getWriter().write("{\"auth\":\"No Key\"}");
                LOG.info("Guest auth No Key for " + username);
                return;
            }
            GuardSessionService sessionService = Services.getService(GuardSessionService.class);
            String token = sessionService.newToken(cookieID, keys.getUserid(), keys.getContextid());
            if (token == null) {
                response.getWriter().write("{\"auth\":\"Unable to get token\"}");
                LOG.error("Guest auth Unable to get database token for " + username);
                return;
            }
            // This is the json we're going to encrypt and include in the auth
            JsonObject json = new JsonObject();
            json.addProperty("user_id", keys.getUserid());
            json.addProperty("encr_password", password);
            json.addProperty("sessionID", sessionID);
            json.addProperty("cid", keys.getContextid());
            json.addProperty("recoveryAvail", keys.isRecoveryAvail());

            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_GCM);
            String returndata = "Bad Password";

            // If Guest, and no password provided, possibly first time user
            if (password.equals("") && keys.getContextid() < 0) {
                // Only works if recoveryAvail and configured to allow no password on first use
                if (keys.isRecoveryAvail() && !configService.getBooleanProperty(GuardProperty.newGuestsRequirePassword)) {
                    if (keys.getLastup() == null) {  // Make sure password never changed
                        JsonObject sendjson = new JsonObject();
                        sendjson.addProperty("productName", GuardProductName.getProductName((request.getServerName())));
                        // If PIN
                        if ((keys.getLastup() == null) && (keys.getQuestion() == null ? false : keys.getQuestion().equals("PIN"))) {
                              sendjson.addProperty("pin", true);
                            }
                        sendjson.addProperty("auth", "new");
                        //Getting generic guest settings from configuration
                        GuardCapabilities settings = getGenericGuestPermissions();
                        //converting settings to JSON
                        sendjson.add("settings", getSettings(keys.getSettings().toString(), settings, keys.getUserid(), keys.getContextid()));
                        ServletUtils.sendJsonOK(response, sendjson);
                        return;
                    }
                }
            }

            //Check password
            boolean passwordVerified = false;
            if (!PGPUtil.verifyPassword(keys.getPGPSecretKeyRing(),password, keys.getSalt())) {
                //Check the other keys according to Bug 49675
                Collection<GuardKeys> otherKeys = keyService.getAllKeys(keys.getUserid(), keys.getContextid());
                for(GuardKeys otherKey : otherKeys) {
                    if(otherKey.getKeyid() == keys.getKeyid()) {
                        otherKeys.remove(otherKey);
                        break;
                    }
                }
                for(GuardKeys otherKey : otherKeys) {
                    if(PGPUtil.verifyPassword(otherKey.getPGPSecretKeyRing(),password, otherKey.getSalt())) {
                        passwordVerified = true;
                        keys = otherKey;
                        break;
                    }
                }
            }
            else {
                passwordVerified = true;
            }

            if (!passwordVerified) {
                antiabuse.reportLogin(false, username, password, ServletUtils.getClientIP(request));
                LOG.info("Bad password for " + username);
                JsonObject sendjson = new JsonObject();
                sendjson.addProperty("auth", returndata);
                sendjson.addProperty("recoveryAvail", keys.isRecoveryAvail());
                ServletUtils.sendJsonOK(response, sendjson);
                return;
            } else {
                // Check if PIN
                if ((keys.getLastup() == null) && (keys.getQuestion() == null ? false : keys.getQuestion().equals("PIN"))) {
                    if (pin != null) {
                        if (pin.equals(keys.getAnswer())) {
                            returndata = cipherService.encrypt(createAuthJson(json, keys.getUserid(), keys.getContextid(), keys.getEmail()).toString(), token);
                        } else {
                            returndata = "Bad";
                            antiabuse.addBad(username);
                        }
                    } else {
                        LOG.info("New Guest login, pin required");
                        returndata = "PIN";
                    }
                } else {
                    antiabuse.reportLogin(true, username, password, ServletUtils.getClientIP(request));
                    returndata = cipherService.encrypt(createAuthJson(json, keys.getUserid(), keys.getContextid(), keys.getEmail()).toString(), token);
                }
            }

            // The json returned in plain text.  no password here (encrypted within auth)  Include settings
            JsonObject sendjson = new JsonObject();
            sendjson.addProperty("productName", GuardProductName.getProductName((request.getServerName())));
            sendjson.addProperty("auth", returndata);
            sendjson.addProperty("userid", keys.getUserid());
            sendjson.addProperty("sessionID", sessionID);
            sendjson.addProperty("cid", keys.getContextid());
            sendjson.addProperty("recoveryAvail", keys.isRecoveryAvail());
            sendjson.addProperty("lastMod", keys.getLastup() == null ? "null" : keys.getLastup().toString());
            sendjson.addProperty("new", keys.getQuestion() == null ? true : false);
            try {
                //Getting generic guest settings from configuration
                GuardCapabilities settings = getGenericGuestPermissions();
                //converting settings to JSON
                sendjson.add("settings", getSettings(keys.getSettings().toString(), settings, keys.getUserid(), keys.getContextid()));
            } catch (Exception ex) {
                LOG.error("problem getting guest settings ", ex);
            }

            GuardLogProperties.put(GuardLogProperties.Name.GUEST_USER_ID, keys.getUserid());
            GuardLogProperties.put(GuardLogProperties.Name.GUEST_CONTEXT_ID, keys.getContextid());
            GuardLogProperties.put(GuardLogProperties.Name.GUEST_MAIL, username);
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_AUTH_ID, returndata);

            LOG.info("Guest auth good for " + username);
            ServletUtils.sendJsonOK(response, sendjson);

        } catch (Exception e) {
            LOG.error("Error during guest login  for user " + username, e);
        }
    }

    public void logout(OxCookie cookie) throws Exception {
        GuardSessionService sessionService = Services.getService(GuardSessionService.class);
        sessionService.deleteToken(cookie.getJSESSIONID());
        sessionService.deleteToken(cookie.getOxReaderID());
    }

    public void login(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws IOException {
        try {
            JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
            GuardCapabilities settings = Services.getService(GuardUserService.class).getGuardCapabilieties(cid, userid);
            login(json, request, response, userid, cid, cookie, settings);
        } catch (Exception ex) {
            LOG.error("Error reading login JSON information", ex);
            JsonObject ret = new JsonObject();
            ret.addProperty("auth", "BAD FORMAT");
            ServletUtils.sendJsonOK(response, ret);
            return;
        }
    }

    private AtomicReference<String> pubkey = null;

    private void loadPublic() {
        GuardKeys passwordKeys;
        try {
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            passwordKeys = keyService.getKeys(-2, 0);
            if (passwordKeys != null) {
                pubkey = new AtomicReference<String>();
                pubkey.set(passwordKeys.getEncodedPublic());
            }
        } catch (Exception e) {
            LOG.error("Error getting public RSA key ", e);
        }
    }

    /**
     * FIXME: Moved this from class Core to the only usage. Hope this is desired behavior.
     */
    private String getPublicKey() {
        if (pubkey == null) {
            loadPublic();
        }
        return pubkey.get();
    }

    public void login(JsonObject json, HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie, GuardCapabilities settings) throws Exception {
        response.setContentType("application/json");
        // response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String lang = JsonUtil.getStringFromJson(json, "lang");
        JsonObject returnJson = new JsonObject();
        returnJson.add("lang", getAvailableLanguagesJson(lang));
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        returnJson.addProperty("server", GuardVersion.getGuardVersion());
        // Get public RSA keys for password encryption
        String pubkey = getPublicKey();
        if (pubkey != null) {
            returnJson.addProperty("pubKey", pubkey);
        }
        try {
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            String email = ((json == null) || (json.get("email") == null) || json.get("email").isJsonNull()) ? null : json.get("email").getAsString();
            if(email != null) {
                email = new EmailValidator().assertInput(email, "email");
            }
            // Check lockout of bad attempts.  If excessive, don't give encrypted auth key
            if (antiabuse.blockLogin(userid, cid, "", ServletUtils.getClientIP(request))) {
                returnJson.addProperty("auth", "Lockout");
                returnJson.add("settings", getSettings(null, settings, userid, cid));
                ServletUtils.sendJsonOK(response, returnJson);
                LOG.info("Lockout auth due to bad attempts");
                return;
            }

            //Verify and UPDATE the context; i.e. triggering update tasks.
            try {
                ContextService contextService = Services.getService(ContextService.class);
                contextService.getContext(cid);
            } catch (OXException ex) {
                if (ex.similarTo(ContextExceptionCodes.UPDATE)) {
                    returnJson.addProperty("auth", "Updating");
                    returnJson.addProperty("error", ex.getDisplayMessage(new Locale(lang)));
                    ServletUtils.sendJsonOK(response, returnJson);
                    LOG.debug("The server is currently down for maintenance. cid = " + cid + " userid = " + userid);
                    return;
                } else {
                    throw ex;
                }
            }

            returnJson.addProperty("cid", cid);
            GuardKeys keys = null;
            String keyid = ServletUtils.getStringParameter(request, "keyid");
            if (keyid != null) {
                try {
                    Long kid = Long.parseLong(keyid);
                    keys = keyService.getKeys(userid, cid, kid);
                } catch (Exception ex) {
                    LOG.error("Problem with login for Key ID " + keyid, ex);
                }
            } else {
                keys = keyService.getKeys(userid, cid);
            }

            // Nothing found.  Previously a non-ox user and new to us?
            if (keys == null) {
                if (email != null) {
                    GuardKeys oldGuestKey = keyService.getKeys(email);
                    if(oldGuestKey != null){
                        if (oldGuestKey.getContextid() < 0) {
                            try {
                                // try to upgrade the account
                                GuestUpgradeService guestUpgradeService = Services.getService(GuestUpgradeService.class);
                                guestUpgradeService.upgrade(oldGuestKey.getUserid(), oldGuestKey.getContextid(), userid, cid);
                                // try again: reload the new key
                                keys = keyService.getKeys(userid, cid);
                            } catch (Exception ex) {
                                LOG.error("Unable to upgrade email " + email);
                            }

                        }
                    }
                }
            }

            if (keys == null) {
                returnJson.addProperty("auth", "No Key");
                returnJson.add("settings", getSettings(null, settings, userid, cid));
                ServletUtils.sendJsonOK(response, returnJson);
                LOG.debug("Auth sent with No Key for cid = " + cid + " userid = " + userid);
                return;
            }
            returnJson.addProperty("primaryEmail", keys.getEmail());
            returnJson.add("settings", getSettings(keys.getSettings().toString(), settings, userid, cid));
            returnJson.add("recoveryAvail", new JsonPrimitive(keys.isRecoveryAvail()));
            if (keys.isPasswordNeeded() == true) {
                returnJson.addProperty("auth", "Password Needed");
                if ((keys.getQuestion() == null) && (keys.isRecoveryAvail())) {  // New, first time use
                    returnJson.addProperty("new", true);
                }
                ServletUtils.sendJson(response, returnJson);
                LOG.debug("Auth sent with Password Needed for cid = " + cid + " userid = " + userid);
                return;
            }

            GuardSessionService sessionService = Services.getService(GuardSessionService.class);
            String token = sessionService.getToken(cookie.getJSESSIONID());
            if (token == null) {
                token = sessionService.newToken(cookie.getJSESSIONID(), userid, cid);
            }
            if (token == null) {
                returnJson.addProperty("auth", "Unable to get token");
                ServletUtils.sendJson(response, returnJson);
                LOG.error("Auth unable to get token from database for cid = " + cid + " userid = " + userid);
                return;
            }

            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_GCM);
            try {
                String oxpassword = ((json == null) || (json.get("ox_password") == null)) ? "" : json.get("ox_password").getAsString();
                if (oxpassword.equals("null")) {
                    oxpassword = "";
                }
                String encrpass = JsonUtil.getStringFromJson(json, "encr_password");
                if (encrpass == null) {
                    encrpass = "";
                }
                String returndata = "Bad Password";

                if (encrpass.length() > 1) {// If provided with encryption password, check
                    try {
                        if (verifyPassword(keys, userid, cid, encrpass, request)) {
                            returndata = cipherService.encrypt(createAuthJson(json, userid, cid, keys.getEmail()).toString(), token);
                            antiabuse.reportLogin(true, userid, cid, encrpass, ServletUtils.getClientIP(request));
                            LOG.debug("Auth with proper password for cid = " + cid + " userid = " + userid);
                        } else {
                            antiabuse.reportLogin(false, userid, cid, encrpass, ServletUtils.getClientIP(request));
                            LOG.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                        }
                    } catch (OXException e) {
                        if (e.similarTo(GuardAuthExceptionCodes.LOCKOUT)) {
                            returndata = "lockout";
                        }
                        if (e.similarTo(GuardAuthExceptionCodes.BAD_PASSWORD)) {
                            antiabuse.reportLogin(false, userid, cid, encrpass, ServletUtils.getClientIP(request));
                            LOG.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                        }
                    }
                } else {// If no encr password, try the oxpassword
                    if (oxpassword.length() > 1) {
                        try {
                            if (verifyPassword(keys, userid, cid, oxpassword, request)) {
                                antiabuse.reportLogin(true, userid, cid, oxpassword, ServletUtils.getClientIP(request));
                                json.remove("encr_password");
                                json.addProperty("encr_password", oxpassword); // The encryption password is our ox password
                                returndata = cipherService.encrypt(createAuthJson(json, userid, cid, keys.getEmail()).toString(), token);
                                LOG.debug("Auth with proper password for cid = " + cid + " userid = " + userid);
                                // returndata = encr.encryptAES(json.toString(), token, cookie.JSESSIONID, Config.AESKeyLen);
                            } else {
                                antiabuse.reportLogin(false, userid, cid, oxpassword, ServletUtils.getClientIP(request));
                                LOG.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                            }
                        } catch (OXException e) {
                            if (e.similarTo(GuardAuthExceptionCodes.LOCKOUT)) {
                                returndata = "lockout";
                            }
                            if (e.similarTo(GuardAuthExceptionCodes.BAD_PASSWORD)) {
                                antiabuse.reportLogin(false, userid, cid, oxpassword, ServletUtils.getClientIP(request));
                                LOG.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                            }
                        }
                    }
                }

                returnJson.addProperty("auth", returndata);
                ServletUtils.sendJson(response, returnJson);
            } catch (Exception e) {
                LOG.error("Error during login", e);
                returnJson.addProperty("auth", "BAD FORMAT");
                ServletUtils.sendJson(response, returnJson);
                return;
            }

        } catch (Exception e) {
            if ((e.getMessage() != null) && (e.getMessage().contains("doesn't exist"))) {
                returnJson.addProperty("auth", "No Key");
                returnJson.add("settings", getSettings(null, settings, 0, 0));
                ServletUtils.sendJson(response, returnJson);
            } else {
                JsonObject errorJson = new JsonObject();
                errorJson.addProperty("auth", "BAD FORMAT");
                errorJson.addProperty("lang", getAvailableLanguagesJson().toString());
                ServletUtils.sendJsonOK(response, errorJson);
                LOG.error("Error during login", e);
            }
        }
    }

    /**
     * Create JSON that gets encrypted and returned as the auth code
     *
     * @param json
     * @param userid
     * @param cid
     * @param email
     * @return
     */
    private JsonObject createAuthJson(JsonObject json, int userid, int cid, String email) {
        JsonObject auth = new JsonObject();
        auth.addProperty("ox_password", json.has("ox_password") ? json.get("ox_password").getAsString() : "");
        auth.addProperty("encr_password", json.has("encr_password") ? json.get("encr_password").getAsString() : "");
        auth.addProperty("email", email);
        auth.addProperty("user_id", userid);
        auth.addProperty("cid", cid);
        auth.addProperty("language", json.has("language") ? json.get("language").getAsString() : "");
        return (auth);
    }

    private JsonObject getAvailableLanguagesJson() throws OXException {
        return getAvailableLanguagesJson(null);
    }

    private JsonObject getAvailableLanguagesJson(String code) throws OXException {
        GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
        if (Strings.isEmpty(code)) {
            code = translationService.getAvailableCode("NON_EXISTENT");
        }
        Map<String, String> available = translationService.getAvailableLanguages(code);
        JsonObject jsonAvailable = new JsonObject();
        for (String key : available.keySet()) {
            jsonAvailable.addProperty(key, available.get(key));
        }
        return jsonAvailable;
    }

    /**
     * First, verify the current password. If fails, then check if one of the other keys
     *
     * @param key
     * @param userid
     * @param cid
     * @param password
     * @return
     * @throws Exception
     */
    private boolean verifyPassword(GuardKeys key, int userid, int cid, String password, HttpServletRequest request) throws Exception {

        if (PGPUtil.verifyPassword(key.getPGPSecretKeyRing(), password, key.getSalt())) {
            return (true);
        }
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        List<GuardKeys> userKeys = ogKeyTableStorage.getKeysForUser(userid, cid);
        if (userKeys.size() > 1) {
            for (GuardKeys userKey : userKeys) {
                if (!userKey.isCurrent()) {// Don't recheck the current key
                    if (PGPUtil.verifyPassword( userKey.getPGPSecretKeyRing(), password, userKey.getSalt())) {
                        return (true);
                    }
                }
            }
        }
        return (false);
    }

    /**
     * Create a json of settings, include the individuals settings, plus global
     *
     * @param settings
     * @return
     * @throws OXException
     */
    private JsonObject getSettings(String settings, GuardCapabilities setting, int id, int cid) throws OXException {
        StringBuilder allSettings = new StringBuilder();
        if (settings == null) {
            settings = "";
        }

        // If settings string loaded with key, convert to JSON
        if (!settings.startsWith("{")) {
            allSettings.append("{");
        }
        if (settings != null) {
            allSettings.append(settings);
        }
        if (!settings.endsWith("}")) {
            allSettings.append("}");
        }

        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);

        // Convert to Json, then check if global defaults present and add. Individual settings override
        Gson gson = new GsonBuilder().create();
        JsonObject json = gson.fromJson(allSettings.toString(), JsonObject.class);
        if (!json.has("Private")) {// If not defined at user level database, use global settings
            if (setting.isPrivatepass() == false) {// If config cascade value set as false, then use
                json.add("ChangePrivate", new JsonPrimitive("false"));
            } else {
                json.add("ChangePrivate", new JsonPrimitive(configService.getBooleanProperty(GuardProperty.okPrivate) ? "true" : "false"));
            }
        }

        if (!json.has("oxguard")) {
            json.add("oxguard", new JsonPrimitive(setting.hasPermission(GuardCapabilities.Permissions.MAIL)));
        }

        // If either the global setting is set for no delete keys, or the config cascade, then use
        json.add("noDeletePrivate", new JsonPrimitive(setting.isNoDeletePrivate()));
        json.add("noDeleteRecovery", new JsonPrimitive(setting.isNoDeleteRecovery()));

        json.add("noRecovery", new JsonPrimitive(setting.isNorecovery()));

        if (!configService.getBooleanProperty(GuardProperty.showStatus)) {
            json.addProperty("status", false);
        }
        json.addProperty("min_password_length", configService.getIntProperty(GuardProperty.minPasswordLength));
        json.addProperty("password_length", configService.getIntProperty(GuardProperty.newPassLength));

        json.addProperty("secureReply", secureReply(id, cid));

        return (json);
    }

    /**
     * Get the {@link GuardProperty.secureReply} 'com.openexchange.guard.SecureReply' property for the specified user in the specified context
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return The value of the {@link GuardProperty.secureReply} 'com.openexchange.guard.SecureReply' property
     */
    private boolean secureReply(int userId, int contextId) {
        try {
            GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
            return guardConfigurationService.getBooleanProperty(GuardProperty.secureReply, userId, contextId);
        } catch (OXException e) {
            LOG.error("{}", e);
        }
        return GuardProperty.secureReply.getDefaultValue(Boolean.class);
    }

    public GuardCapabilities getGenericGuestPermissions() throws OXException {
        GuardCapabilities guestSettings = new GuardCapabilities();
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);

        guestSettings.setNoDeleteRecovery(configService.getBooleanProperty(GuardProperty.noDeleteRecovery));
        guestSettings.setNorecovery(configService.getBooleanProperty(GuardProperty.noRecovery));
        guestSettings.setNoDeletePrivate(configService.getBooleanProperty(GuardProperty.noDeletePrivate));

        return guestSettings;
    }
}
