/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.user.internal;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.oxapi.MailResolver;
import com.openexchange.guard.oxapi.capabilities.Capabilities;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.GuardUser;
import com.openexchange.guard.user.GuardUserService;
import com.openexchange.guard.user.exceptions.GuardUserExceptionCodes;

/**
 * {@link GuardUserServiceImpl}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since v2.4.0
 */
public class GuardUserServiceImpl implements GuardUserService {

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(GuardUserServiceImpl.class);
    private final MailResolver mailResolver;
    private final GuardConfigurationService configurationService;

    /**
     * Initializes a new {@link GuardUserServiceImpl}.
     *
     * @param configurationService The configuration service
     */
    public GuardUserServiceImpl(GuardConfigurationService configurationService) {
        this.configurationService = configurationService;
        this.mailResolver = new MailResolver();
    }

    @Override
    public GuardUser getUser(String email) throws OXException {
        JsonObject userdat = mailResolver.resolveUser(email);
        if ((userdat != null) && (userdat.has(email))) {
            JsonObject user = userdat.get(email).getAsJsonObject();
            int userId = user.get("uid").getAsInt();
            int contextId = user.get("cid").getAsInt();
            String language = user.get("user").getAsJsonObject().get("language").getAsString();
            String name = "";
            if (user.get("user").getAsJsonObject().has("displayName")) {
                name = user.get("user").getAsJsonObject().get("displayName").getAsString();
            }
            return new GuardUser(email, userId, contextId, name, language);
        }
        return null;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.user.GuardUserService#getUserCapabilieties(int, int)
     */
    @Override
    public GuardCapabilities getGuardCapabilieties(int contextId, int userId) throws OXException {
        try {
            GuardCapabilities guardCapabilities = new GuardCapabilities();
            guardCapabilities.setPerm(GuardCapabilities.Permissions.NONE);
            JsonArray userCapabilities = new Capabilities().getUserCapabilities(userId, contextId);
            String jsonData = userCapabilities.toString();
            if (jsonData.contains("guard")) {
                guardCapabilities.setPerm(GuardCapabilities.Permissions.GUARD);
            }
            if (jsonData.contains("guard:mail") || jsonData.contains("guard-mail")) {
                guardCapabilities.setPerm(GuardCapabilities.Permissions.MAIL);
            }
            if (jsonData.contains("guard:drive") || jsonData.contains("guard-drive")) {
                if (guardCapabilities.getPerm().equals(GuardCapabilities.Permissions.MAIL)) {
                    guardCapabilities.setPerm(GuardCapabilities.Permissions.GUARD_MAIL_FILE);
                } else {
                    guardCapabilities.setPerm(GuardCapabilities.Permissions.FILE);
                }
            }

            if (jsonData.contains("guard-noprivate")) {
                guardCapabilities.setPrivatepass(false);
            }
            if (jsonData.contains("guard-nodeleterecovery")) {
                guardCapabilities.setNoDeleteRecovery(true);
            } else {
                guardCapabilities.setNoDeleteRecovery(configurationService.getBooleanProperty(GuardProperty.noDeleteRecovery));
            }

            if (jsonData.contains("guard-norecovery")) {
                guardCapabilities.setNorecovery(true);
            } else {
                guardCapabilities.setNorecovery(configurationService.getBooleanProperty(GuardProperty.noRecovery));
            }

            if (jsonData.contains("guard-nodeleteprivate")) {
                guardCapabilities.setNoDeletePrivate(true);
            } else {
                guardCapabilities.setNoDeletePrivate(configurationService.getBooleanProperty(GuardProperty.noDeletePrivate));
            }
            return (guardCapabilities);
        } catch (Exception e) {
            logger.error("Error while fetching user capabilities: " + e.getMessage());
            throw GuardUserExceptionCodes.CAPABILITY_ERROR.create(e, "guard");
        }
    }
}
