/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.session.internal;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.session.osgi.Services;

/**
 * {@link GuardSessionServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardSessionServiceImpl implements GuardSessionService {

    private static final Logger LOG = LoggerFactory.getLogger(GuardSessionServiceImpl.class);

    private final GuardSessionStorage guardSessionStorage;

    /**
     * Initialises a new {@link GuardSessionServiceImpl}.
     */
    public GuardSessionServiceImpl() {
        super();
        guardSessionStorage = new GuardSessionStorageImpl();
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.session.GuardSessionService#newToken(java.lang.String, int, int)
     */
    @Override
    public String newToken(String sessionId, int userId, int contextId) throws OXException {
        //Create a new token
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        String token = cipherService.getRandomKey();

        //Insert or update the token
        guardSessionStorage.InsertOrUpdate(sessionId, token, userId, contextId);

        return token;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.session.GuardSessionService#deleteToken(java.lang.String)
     */
    @Override
    public void deleteToken(String sessionId) throws OXException {
        if (sessionId != null) {
            guardSessionStorage.deleteById(sessionId);
        }
    }
    
    /* (non-Javadoc)
     * @see com.openexchange.guard.session.GuardSessionService#deleteTokens(long)
     */
    @Override
    public int deleteTokens(long milliSeconds) throws OXException {
        return guardSessionStorage.deleteOldSessions(milliSeconds);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.session.GuardSessionService#getToken(java.lang.String)
     */
    @Override
    public String getToken(String sessionId) throws OXException {
        try {
            GuardSession session = guardSessionStorage.getById(sessionId);
            if (session != null) {
                return session.getToken();
            }
        } catch (Exception e) {
            LOG.error("Error while retrieving token from session " + sessionId, e);
        }
        //TODO: maybe throw exception instead of returning null
        return null;
    }

}
