/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.slf4j.Logger;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;
import com.openexchange.guard.pgpcore.commons.PGPDecrypter;
import com.openexchange.guard.pgpcore.commons.PGPEncrypter;
import com.openexchange.guard.pgpcore.commons.PGPKeyRetrievalStrategy;
import com.openexchange.guard.pgpcore.commons.PGPSignatureVerificationResult;
import com.openexchange.guard.pgpcore.commons.GuardKeyRingRetrievalStrategy;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.pgpcore.services.UserIdentity;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;

/**
 * {@link PGPCryptoServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class PGPCryptoServiceImpl implements PGPCryptoService {

    private final PGPKeyRetrievalStrategy pgpKeyRetrievalStrategy;
    private final GuardKeyRingRetrievalStrategy recipientKeyRingRetrievalStrategy;
    private static final Logger logger = org.slf4j.LoggerFactory.getLogger(PGPCryptoServiceImpl.class);

    /**
     * Initializes a new {@link PGPCryptoServiceImpl}.
     *
     * @param pgpKeyRetrievalStrategy the strategy for retrieving keys for decryption.
     * @param recipientKeyRingRetrievalStrategy the strategy for retrieving keys for encryption
     */
    public PGPCryptoServiceImpl(GuardKeyRingRetrievalStrategy recipientKeyRingRetrievalStrategy,
                                PGPKeyRetrievalStrategy pgpKeyRetrievalStrategy) {
        this.recipientKeyRingRetrievalStrategy = recipientKeyRingRetrievalStrategy;
        this.pgpKeyRetrievalStrategy = pgpKeyRetrievalStrategy;
    }

    /**
     * Fetches the public keys for the given identities
     *
     * @param recipients The identities to fetch the public keys for
     * @return A set of public keys for the given identities
     * @throws OXException If a key was not found for a given identity
     */
    PGPPublicKey[] getRecipientKeys(List<String> recipients) throws OXException {

        ArrayList<PGPPublicKey> ret = new ArrayList<>();
        for (String recipient : recipients) {
            GuardKeys guardKeys = recipientKeyRingRetrievalStrategy.getKeyRingForRecipient(recipient);
            if(guardKeys != null) {
                PGPPublicKey key = guardKeys.getPGPPublicKey();
                if (key != null) {
                    ret.add(key);
                }
                else {
                    throw PGPCoreServicesExceptionCodes.RECIPIENT_KEY_NOT_FOUND.create(recipient);
                }
            }
            else {
                throw PGPCoreServicesExceptionCodes.RECIPIENT_KEY_NOT_FOUND.create(recipient);
            }
        }
        return ret.toArray(new PGPPublicKey[ret.size()]);
    }

    /**
     * Fetches the Guard key for the given identity
     *
     * @param identity The identity to get the signing key for
     * @return The signing key for the given identity
     * @throws OXException If the key was not found
     */
    GuardKeys getSignerGuardKey(String identity) throws OXException {
        GuardKeys guardKeys = recipientKeyRingRetrievalStrategy.getKeyRingForRecipient(identity);
        if (guardKeys != null) {
            return guardKeys;
        }
        else {
            throw PGPCoreServicesExceptionCodes.SIGNING_KEY_NOT_FOUND.create(identity);
        }
    }

    /**
     * Extracts the PGP signing key from the given Guard key
     *
     * @param guardKey The key to get the singing key for
     * @return The signing key for the given Guard key
     * @throws OXException If no signing key was found
     */
    PGPSecretKey getSignerKey(GuardKeys guardKey) throws OXException {
        PGPSecretKey signingKey = PGPUtil.getSigningKey(guardKey.getPGPSecretKeyRing());
        if (signingKey != null) {
            return signingKey;
        }
        else {
            throw PGPCoreServicesExceptionCodes.SIGNING_KEY_NOT_FOUND.create(guardKey.getKeyid());
        }
    }

    private String listToString(List l) {
        return l.toString().replace("[", "").replace("]", "").replace(", ", ",");
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.services.PGPCryptoService#encrypt(java.io.InputStream, java.io.OutputStream, boolean, com.openexchange.guard.pgpcore.services.PGPPublicKey[])
     */
    @Override
    public void encrypt(InputStream input, OutputStream output, boolean armored, List<String> recipients) throws OXException {
        try {
            logger.debug("Encrypting data for recipients: " + listToString(recipients) + ". ASCII-Armor = " + armored + ".");
            PGPPublicKey[] recipientKeys = getRecipientKeys(recipients);
            new PGPEncrypter().encrypt(input, output, armored, recipientKeys);
        } catch (OXException e) {
            throw e;
        } catch (Exception e) {
            throw PGPCoreServicesExceptionCodes.ENCRYPTION_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.services.PGPCryptoService#encryptSigned(java.io.InputStream, java.io.OutputStream, boolean, com.openexchange.guard.pgpcore.services.PGPSecretKey, char[],
     * com.openexchange.guard.pgpcore.services.PGPPublicKey[])
     */
    @Override
    public void encryptSigned(InputStream input, OutputStream output, boolean armored, UserIdentity signerIdentity, List<String> recipients) throws OXException {
        try {
            logger.debug("Encrypting data for recipients: " + listToString(recipients) + ". Signging as: " + signerIdentity.getIdentity() + ". ASCII-Armor = " + armored + ".");
            PGPPublicKey[] recipientKeys = getRecipientKeys(recipients);
            GuardKeys guardKey = getSignerGuardKey(signerIdentity.getIdentity());
            PGPSecretKey signingKey = getSignerKey(guardKey);
            char[] hashedPassword = CipherUtil.getSHA(new String(signerIdentity.getPassword()), guardKey.getSalt()).toCharArray();
            new PGPEncrypter().encryptSigned(input, output, armored, signingKey, hashedPassword, recipientKeys);
        } catch (OXException e) {
            throw e;
        } catch (Exception e) {
            PGPCoreServicesExceptionCodes.ENCRYPTION_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.services.PGPCryptoService#decrypt(java.io.InputStream, java.io.OutputStream, java.lang.String, char[])
     */
    @Override
    public List<PGPSignatureVerificationResult> decrypt(InputStream input, OutputStream output, UserIdentity recipient) throws OXException {
        try {
            logger.debug("Decrypting data for : " + recipient.getIdentity());
            PGPDecrypter decrypter = new PGPDecrypter(pgpKeyRetrievalStrategy);
            return decrypter.decrypt(input, output, recipient.getIdentity(), recipient.getPassword());
        } catch (OXException e) {
            throw e;
        } catch (Exception e) {
            throw PGPCoreServicesExceptionCodes.DECRYPTION_ERROR.create(e, e.getMessage());
        }
    }
}
