/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mailcreator.internal;

import java.util.ArrayList;
import java.util.List;
import javax.mail.Message.RecipientType;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProductName;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.translation.ConditionalVissibleTemplateTransformation;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.translation.TemplateTransformation;

/**
 * {@link MailCreatorServiceImpl}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since v2.4.0
 */
public class MailCreatorServiceImpl implements MailCreatorService {

    private static Logger LOG = LoggerFactory.getLogger(MailCreatorServiceImpl.class);

    public static final String OUTSITE_TEMPLATE = "";

    public static final String PASSWORD_EMAIL_TEMPLATE = "";

    @Override
    public JsonObject createBlankMail() {
        String blank = " {\"data\":{\"from\":\"\",\"to\":\"\",\"cc\":\"\",\"bcc\":\"\",\"reply_to\":\"\",\"subject\":\"\",\"priority\":\"3\",\"vcard\":\"0\",\"attachments\":[{\"content\":\"\",\"content_type\":\"ALTERNATIVE\"}],\"nested_msgs\":[],\"sendtype\":\"0\"},\"mode\":\"compose\",\"format\":\"html\",\"signature\":\"\",\"files\":[]} ";
        Gson g = new Gson();
        JsonObject json = g.fromJson(blank, JsonObject.class);
        return (json);
    }

    @Override
    public JsonObject createBlankOutMail(String lang, int templId) throws OXException {
        JsonObject email = createBlankMail();
        GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
        String content = translationService.getTranslation("oxtempl.html", lang, templId);
        if (content == null) {
            return (null);
        }
        return (setContent(email, content));
    }

    @Override
    public JsonObject createBlankGuestMail(String lang, int templId, String host, int userid, int cid, String guestMessage) throws OXException {
        return createBlankMail (lang, templId, host, userid, cid, "guesttempl.html", guestMessage);
    }

    @Override
    public JsonObject createBlankMail (String lang, int templId, String host, int userid, int cid, String template, String guestmessage) throws OXException {
        JsonObject email = createBlankMail();
        ArrayList<TemplateTransformation> transformations = new ArrayList<TemplateTransformation>();
        //"guestmessage" is a conditional part of the template file. We only show it, if the user entered a guest message
        transformations.add(new ConditionalVissibleTemplateTransformation("guestmessage", guestmessage == null || guestmessage.isEmpty()));
        transformations.add(new ConditionalVissibleTemplateTransformation("!guestmessage", guestmessage != null && !guestmessage.isEmpty()));
        GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
        String content = translationService.getTranslation(template, lang, templId, transformations);
        content = setProductName (content, host, userid, cid);
        if (content == null) {
            return (null);
        }
        return (setContent(email, content));
    }

    @Override
    public JsonObject createBlankGuestMail(String lang, int templId, String messageID, String host, int userid, int cid, String guestMessage) throws OXException {
        JsonObject mail = createBlankGuestMail(lang, templId, host, userid, cid, guestMessage);
        JsonObject header = new JsonObject();
        header.addProperty("Message-ID", messageID);
        mail.get("data").getAsJsonObject().add("headers", header);
        return (mail);
    }

    @Override
    public JsonObject addPlainText(JsonObject email, String data) {
        Gson gson = new Gson();
        String emailText = gson.toJson(email);
        if (data == null) {
            data = "";
        }
        emailText = emailText.replace("$plaintext", data.replace("\r", "<br>"));
        Gson g = new Gson();
        return (g.fromJson(emailText, JsonObject.class));
    }

    @Override
    public JsonObject addURL(JsonObject email, String data) {
        Gson gson = new Gson();
        String emailText = gson.toJson(email);
        emailText = emailText.replace("$url", data);
        Gson g = new Gson();
        return (g.fromJson(emailText, JsonObject.class));
    }

    @Override
    public JsonObject addTo(JsonObject mail, String name, String email, RecipientType type) {
        if (type == RecipientType.TO) {
            return (addTo(mail, name, email));
        }
        if (type == RecipientType.CC) {
            return (addCC(mail, name, email));
        }
        if (type == RecipientType.BCC) {
            return (addBCC(mail, name, email));
        }
        return (null);
    }

    // Add flattened to address to mail
    @Override
    public JsonObject addTo(JsonObject mail, String name, String email) {
        String to = mail.getAsJsonObject("data").get("to").getAsString();
        if (to.length() > 0) {
            to = to + ", ";
        }
        if (name.length() > 0) {
                to = to + name + "<" + email + ">";
        } else {
            to = to + email;
        }
        mail.getAsJsonObject("data").add("to", new JsonPrimitive(to));
        return (mail);
    }

    // Add flattened to address to mail
    @Override
    public JsonObject addCC(JsonObject mail, String name, String email) {
        String to = mail.getAsJsonObject("data").get("cc").getAsString();
        if (to.length() > 0) {
            to = to + ", ";
        }
        if (name.length() > 0) {
            to = to + name + "<" + email + ">";
        } else {
            to = to + email;
        }
        mail.getAsJsonObject("data").add("cc", new JsonPrimitive(to));
        return (mail);
    }

    // Add flattened to address to mail
    @Override
   public JsonObject addBCC(JsonObject mail, String name, String email) {
        String to = mail.getAsJsonObject("data").get("bcc").getAsString();
        if (to.length() > 0) {
            to = to + ", ";
        }
        if (name.length() > 0) {
            to = to + name + "<" + email + ">";
        } else {
            to = to + email;
        }
        mail.getAsJsonObject("data").add("bcc", new JsonPrimitive(to));
        return (mail);
    }

    // Add flattened from address to mail
    @Override
    public JsonObject addFrom(JsonObject mail, String name, String email) {
        String from = email;
        if (name != null && name.length() > 0) {
            from = name + "<" + email + ">";
        }
        mail.getAsJsonObject("data").add("from", new JsonPrimitive(from));
        return (mail);
    }

    @Override
    public JsonObject addSubject(JsonObject mail, String subject) {
        mail.getAsJsonObject("data").add("subject", new JsonPrimitive(subject));
        return (mail);
    }

    @Override
    public JsonObject setContent(JsonObject mail, String content) {
        mail.getAsJsonObject("data").getAsJsonArray("attachments").get(0).getAsJsonObject().add("content", new JsonPrimitive(content));
        return (mail);
    }

    @Override
    public JsonObject noSave(JsonObject email) {
        email.getAsJsonObject("data").add("copy2Sent", new JsonPrimitive("false"));
        return (email);
    }

    @Override
    public JsonObject getOxPasswordEmail(String to, String sendername, String from, String from2, String password, String lang, int templId, String host, int userid, int cid) throws OXException {
        return getPasswordEmail(to, sendername, from, from2, password, lang, templId, "", "oxpasswordtempl.html", host, userid, cid);
    }

    @Override
    public JsonObject getPasswordEmail(String to, String sendername, String from, String from2, String password, String lang, int templId, String guestmessage, String host, int userid, int cid) throws OXException {
        return getPasswordEmail(to, sendername, from, from2, password, lang, templId, guestmessage, "passwordtempl.html", host, userid, cid);
    }

    /**
     *
     * @param to email of recipient
     * @param sendername Full name of the sender
     * @param from  Full name of the sender
     * @param from2 Email of the sender
     * @param password password to send
     * @param lang
     * @param templId
     * @param guestmessage
     * @param templateName
     * @param host the senders hostname
     * @param userid the recipient's user id
     * @param cid the recipient's context id
     * @return
     * @throws OXException
     */
    private JsonObject getPasswordEmail(String to, String sendername, String from, String from2, String password, String lang, int templId, String guestmessage, String templateName, String host, int userid, int cid) throws OXException {
        JsonObject email = createBlankMail();
        email = addTo(email, to, to);
        email = addFrom(email, from, from2);
        email.getAsJsonObject("data").add("copy2Sent", new JsonPrimitive("false"));// Do not copy this to sent folder

        ArrayList<TemplateTransformation> transformations = new ArrayList<TemplateTransformation>();
        //"guestmessage" is a conditional part of the template file. We only show it, if the user entered a guest message
        transformations.add(new ConditionalVissibleTemplateTransformation("guestmessage", guestmessage == null || guestmessage.isEmpty()));
        transformations.add(new ConditionalVissibleTemplateTransformation("!guestmessage", guestmessage != null && !guestmessage.isEmpty()));

        GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
        String templ = translationService.getTranslation(templateName, lang, templId, transformations);
        templ = templ.replace("$from", sendername != null ? sendername : from2);
        templ = templ.replace("$password", password);
        templ = templ.replace("$plaintext", guestmessage);
        templ = setProductName (templ, host, userid, cid);
        if (templ.contains("Subject:")) {
            String subject = templ.substring(templ.indexOf("Subject:"), templ.indexOf("\n")).replace("Subject:", "");
            email = addSubject(email, subject);
        } else {
            LOG.error("Missing subject in password template! Make sure starts with 'Subject:'");
        }
        if (templ.contains("Body:")) {
            templ = templ.substring(templ.indexOf("Body:") + 5);
        } else {
            LOG.error("Missing body in password template! Make sure starts with 'Body:'");
        }
        email = setContent(email, templ);
        return (email);
    }

    @Override
    public JsonObject getResetEmail(String to, List<String> from, String password, String lang, int templId, String host, int userid, int cid) throws OXException {
        GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
        String templ = translationService.getTranslation("resettempl.html", lang, templId).replace("$from", from.get(0)).replace("$password", password);
        return (templToEmail (templ, to, from, host, userid, cid));
    }

    @Override
    public JsonObject getGuestResetEmail(String to, String resetId, String lang, int templId, String host, int userid, int cid) throws OXException {
        List<String> from = getFromAddress(to, to, userid, cid);
        GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
        String templ = translationService.getTranslation("guestresettempl.html", lang, templId).replace("$resetid", resetId);
        return (templToEmail (templ, to, from, host, userid, cid));
    }

    private JsonObject templToEmail (String templ, String to, List<String> from, String host, int userid, int cid) throws OXException {
        JsonObject email = createBlankMail();
        email = addTo(email, to, to);
        email = addFrom(email, from.get(0), from.get(1));
        email.getAsJsonObject("data").add("copy2Sent", new JsonPrimitive("false"));// Do not copy this to sent folder
        templ = setProductName (templ, host, userid, cid);
        if (templ.contains("Subject:")) {
            String subject = templ.substring(templ.indexOf("Subject:"), templ.indexOf("\n", templ.indexOf("Subject:"))).replace("Subject:", "");
            email = addSubject(email, subject);
        } else {
            LOG.error("Missing subject in password template! Make sure starts with 'Subject:'");
        }
        if (templ.contains("Body:")) {
            templ = templ.substring(templ.indexOf("Body:") + 5);
        } else {
            LOG.error("Missing body in password template! Make sure starts with 'Body:'");
        }
        email = setContent(email, templ);
        return (email);
    }

    /**
     * Gets sender address from configuration cascade
     *
     * @param name
     * @param email
     * @param userid
     * @param cid
     * @return Arraylist, index 0 is name, index 1 is email address
     */
    @Override
    public ArrayList<String> getFromAddress(String name, String email, int userid, int cid) throws OXException {
        ArrayList<String> answer = new ArrayList<String>();
        // Check if Backend has a from address for this person in config cascade
        String fromemail = null;
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        if (userid != 0) {
            fromemail = guardConfigService.getProperty(GuardProperty.fromEmail, userid, cid);
        }
        // If no config cascade
        if (fromemail == null) {
            // Check if we have global configuration
            String fromAddress = guardConfigService.getProperty(GuardProperty.passwordFromAddress);
            String fromName = guardConfigService.getProperty(GuardProperty.passwordFromName);
            if (Strings.isNotEmpty(fromAddress)) {
                if (Strings.isNotEmpty(fromName)) {
                    answer.add(fromName);
                } else {
                    answer.add(fromAddress);
                }
                answer.add(fromAddress);
                return (answer);
            }
            // If no global, just use the sender info
            answer.add(name);
            answer.add(email);
            if (name == null || email == null) {
                LOG.error("Problem resolving from email " + name + " " + email);
            }
            return (answer);
        }
        try {
            // If config cascade exists, validate and use
            InternetAddress addr = new InternetAddress(fromemail);
            addr.validate();
            answer.add(addr.getPersonal() == null ? addr.getAddress() : addr.getPersonal());
            answer.add(addr.getAddress());
            return (answer);
        } catch (AddressException e) {
            // Validation failed, use sender info
            answer.add(name);
            answer.add(email);
            return (answer);
        }

    }

    /**
     * Replace variable productName with configured Guard product name
     * @param content
     * @param host
     * @param id
     * @param cid
     * @return
     * @throws OXException
     */
    public String setProductName (String content, String host, int id, int cid) throws OXException {
        String name = GuardProductName.getProductName(host, cid, id);
        content = content.replace("$productName", name);
        return (content);
    }
}
