/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services;

import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.OGPGPKeyRing;

/**
 * {@link PublicExternalKeyService} handles external public keys uploaded by a user.
 * <p>
 * An OX Guard user is able to upload public keys for external recipients.
 * </p>
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public interface PublicExternalKeyService {

    /**
     * Gets a list of user uploaded public PGP key rings of external recipients.
     *
     * @param userId The ID of the user
     * @param contextId The context of the user
     * @return A list of uploaded PGP keys
     * @throws OXException
     */
    List<OGPGPKeyRing> get(int userId, int contextId) throws OXException;

    /**
     * Adds new public PGP key rings for an external recipient
     *
     * @param userId The ID who imports the keys
     * @param contextId The context ID of the user who imports the keys
     * @param publicKeyRings The keys to import.
     * @throws OXException
     */
    void importPublicKeyRing(int userId, int contextId, PGPPublicKeyRing... publicKeyRings) throws OXException;

    /**
     * Deletes a public PGP key ring for an external recipient
     *
     * @param userId The ID of the user who wants to delete the public key ring
     * @param contextId The context ID of the user who wants to delete the public key ring
     * @param ids The identifier of the key ring to delete
     * @throws OXException
     */
    void delete(int userId, int contextId, String ids) throws OXException;

    /**
     * Sets the share mode for a specified public key ring
     *
     * @param userId The user ID
     * @param contextId The context ID
     * @param ids The identifier of the key ring
     * @param share True, to mark the key ring as "shared", false otherwise.
     * @throws OXException
     */
    void share(int userId, int contextId, String ids, boolean share) throws OXException;

    /**
     * Sets the inline mode for a specified public key ring
     *
     * @param userId The user ID
     * @param contextId The context ID
     * @param ids The identifier of the key ring
     * @param inline True, to mark the key ring as "inline", False otherwise
     * @throws OXException
     */
    void setInline(int userId, int contextId, String ids, boolean inline) throws OXException;
}
