/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.GuardMasterKey;
import com.openexchange.guard.keymanagement.keysources.RealtimeKeyPairSource;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.impl.exceptions.GuardMasterKeyExceptionCodes;

/**
 * {@link GuardMasterKeyServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardMasterKeyServiceImpl implements GuardMasterKeyService {

    private static final Logger LOG = LoggerFactory.getLogger(GuardMasterKeyServiceImpl.class);

    private GuardMasterKey masterKey;
    private final GuardConfigurationService guardConfigurationService;
    private final GuardKeyService guardKeyService;
    private final KeyCreationService keyCreationService;


    /**
     * Initializes a new {@link GuardMasterKeyServiceImpl}.
     *
     * @throws OXException
     */
    public GuardMasterKeyServiceImpl(GuardConfigurationService guardConfigurationService,
                                     GuardKeyService guardKeyService,
                                     KeyCreationService keyCreationService) throws OXException {
        this.guardConfigurationService = guardConfigurationService;
        this.guardKeyService = guardKeyService;
        this.keyCreationService = keyCreationService;
    }

    /**
     * Reads the master password file 'oxguardpass'
     *
     * @return
     * @throws OXException
     */
    private Map<String, String> readMasterKey(File file) throws OXException {
        // Read lines from the 'oxguardpass'
        List<String> lines = readMasterKeyFile(file);

        // Get lines
        Map<String, String> tmp = new HashMap<String, String>(2);
        for (String line : lines) {
            String k = line.substring(0, 2);
            String v = line.substring(2);
            tmp.put(k, v);
        }

        if (tmp.isEmpty()) {
            throw GuardMasterKeyExceptionCodes.MALFORMED_MASTER_KEY_FILE.create(file.getAbsolutePath());
        }

        return tmp;
    }

    /**
     * Reads the master key from the database
     *
     * @param mc The MC salt
     * @param rc The RC salt
     * @throws OXException
     */
    private void readMasterKey(String mc, String rc) throws OXException {
        // Read private and public key from the db
        PrivateKey privateKey = null;
        try {
            GuardKeys passwordKeys = guardKeyService.getKeys(-2, 0);
            if (passwordKeys != null) {
                privateKey = guardKeyService.decryptPrivateKey(passwordKeys.getEncodedPrivate(), rc, passwordKeys.getSalt());
            }
        } catch (Exception ex) {
            LOG.error("Problem loading password RSA key", ex);
        }

        GuardKeys passwordKeys;
        String publicKey = null;
        try {
            passwordKeys = guardKeyService.getKeys(-2, 0);
            if (passwordKeys != null) {
                publicKey = passwordKeys.getEncodedPublic();
            }
        } catch (Exception e) {
            LOG.error("Error getting public RSA key ", e);
        }

        // Reference the master key
        masterKey = new GuardMasterKey(mc, rc, publicKey, privateKey);
    }

    /**
     * Reads the master key file from the specified file and returns the contents in a List where every entry represents a line
     *
     * @param file The file to read the master key from
     * @return A list with the contents of the file, where every entry represents a line
     * @throws OXException
     */
    private List<String> readMasterKeyFile(File file) throws OXException {
        List<String> lines;
        try {
            lines = Files.readAllLines(Paths.get(file.getAbsolutePath()), StandardCharsets.UTF_8);
            if (lines.size() != 2) {
                throw GuardMasterKeyExceptionCodes.MALFORMED_MASTER_KEY_FILE.create(file.getAbsolutePath());
            }
        } catch (IOException e) {
            throw GuardMasterKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
        return lines;
    }


    /**
     * Generates the master keys in database with the specified mc and rc salts
     *
     * @param mcSalt The MC salt
     * @param rcSalt The RC salt
     * @throws OXException
     */
    private void generateMasterKey(String mcSalt, String rcSalt) throws OXException {
        // Generate public and private keys
        PrivateKey privateKey = null;
        String publicKey = null;
        try {
            GuardKeys keys = keyCreationService.create(new RealtimeKeyPairSource(guardConfigurationService), -1, 0, "master", "dont@remove.com", mcSalt, null, true, false);

            if (keys == null) {
                LOG.error("Unable to create master keys");
            } else {
                guardKeyService.storeKeys(keys, false);
                privateKey = guardKeyService.decryptPrivateKey(keys.getEncodedPrivate(), mcSalt, keys.getSalt());
            }
            GuardKeys  keys2 = keyCreationService.create(new RealtimeKeyPairSource(guardConfigurationService), -2, 0, "master", "dont@remove.com", rcSalt, null, true, false);
            if (keys2 == null) {
                LOG.error("Unable to create master keys for client encryption");
            } else {
                guardKeyService.storeKeys(keys2, false);
                publicKey = keys2.getEncodedPublic();
            }
        } catch (Exception ex) {
            LOG.error("Unable to create master keys", ex);
            throw ex;
        }

        // Update reference
        masterKey = new GuardMasterKey(mcSalt, rcSalt, publicKey, privateKey);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.GuardMasterKeyService#initialise()
     */
    @Override
    public void initialise() throws OXException {
        File f = guardConfigurationService.getPropertyFile("oxguardpass");
        if (f == null || !f.exists()) {
            throw GuardMasterKeyExceptionCodes.MASTER_KEY_FILE_NOT_EXIST.create();
        } else {
            Map<String, String> salts = readMasterKey(f);
            final String mc = salts.get("MC");
            final String rc = salts.get("RC");

            GuardKeys passwordKeys = guardKeyService.getKeys(-2, 0);
            if (passwordKeys == null) {
                generateMasterKey(mc, rc);
            }
            readMasterKey(mc, rc);
        }
    }
    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.configuration.GuardMasterKeyService#getMasterKey()
     */
    @Override
    public GuardMasterKey getMasterKey() {
        return masterKey;
    }
}
