/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.commons.export;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Objects;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.exceptions.KeysExceptionCodes;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;

/**
 * {@link KeyExportUtil} provides several key exporting helper methods.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyExportUtil {

    /**
     * Exports a {@link PGPPublicKeyRing} object into ASCII-Armored string representation
     *
     * @param pgpKeyRing The {@link PGPPublicKeyRing} object to export
     * @return The ASCII-Armored string representation of the given {@link PGPPublicKeyRing} object
     * @throws OXException
     */
    public static String export(PGPPublicKeyRing pgpKeyRing) throws OXException {

        pgpKeyRing = Objects.requireNonNull(pgpKeyRing, "pgpKeyRing must not be null");
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ArmoredOutputStream arm = new ArmoredOutputStream(out);
            ByteArrayOutputStream bout = new ByteArrayOutputStream();

            Iterator<PGPPublicKey> keys = pgpKeyRing.getPublicKeys();
            while (keys.hasNext()) {
                PGPPublicKey k = keys.next();
                k.encode(bout);
            }
            arm.write(bout.toByteArray());

            arm.close();
            bout.close();
            out.close();

            return (new String(out.toByteArray()));
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    public static String exportPGPPrivateKey(PGPSecretKeyRing keys, String password, String salt) throws OXException {
        keys = Objects.requireNonNull(keys, "keys must not be null");
        password = Objects.requireNonNull(password, "password must not be null");
        salt = Objects.requireNonNull(salt, "salt must not be null");
        try {
            String oldpass = CipherUtil.getSHA(password, salt);
            PGPSecretKeyRing newkeyring = PGPUtil.duplicateSecretKeyRing(keys, oldpass, password, keys.getSecretKey().getKeyEncryptionAlgorithm());
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ArmoredOutputStream arm = new ArmoredOutputStream(out);
            arm.write(newkeyring.getEncoded());
            arm.close();
            out.close();
            return (new String(out.toByteArray()));
        } catch (PGPException e) {
            if(e.getMessage().contains("checksum mismatch")) {
                throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }
            throw KeysExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Export a given {@link GuardKeys} object as ASCII-Armored string representation
     *
     * @param guardKeys The {@link GuardKeys} object to export
     * @param password The password required for exporting the key
     * @return The ASCII-Armored string representation of the given {@link GuardKeys} object
     * @throws OXException
     */
    public static String exportPGPPrivateKey(GuardKeys guardKeys, String password) throws OXException {
        return exportPGPPrivateKey(guardKeys.getPGPSecretKeyRing(), password, guardKeys.getSalt());
    }

    public static String exportPGPPublicAndPrivateKey(GuardKeys guardKeys, String password) throws OXException{
        return export(guardKeys.getPGPPublicKeyRing()) +
               "\r\n" +
               exportPGPPrivateKey(guardKeys, password);
    }
}
