/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.commons.antiabuse;

import java.util.Objects;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;

/**
 * {@link AntiAbuseWrapper} is a helper around {@link GuardAntiAbuseService} which allows easier anti abuse handling.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class AntiAbuseWrapper {

    private final GuardAntiAbuseService antiAbuseService;
    private final PGPSecretKeyRing secretKeyRing;
    private final int contextId;
    private final int userId;
    private final String password;
    private final String salt;
    private final String remoteIp;

    /**
     * Initializes a new {@link AntiAbuseWrapper}.
     *
     * @param antiAbuseService The {@link GuardAntiAbuseService} to wrap.
     * @param key The user's {@link GuardKeys} to check the password against.
     * @param contextId The constext ID of the user
     * @param userId The user ID
     * @param password The password to check
     * @param remoteIp The remote IP of the request
     */
    public AntiAbuseWrapper(GuardAntiAbuseService antiAbuseService,
                            GuardKeys key,
                            int contextId,
                            int userId,
                            String password,
                            String remoteIp) {
        this(antiAbuseService,
             key.getPGPSecretKeyRing(),
             contextId,
             userId,
             password,
             key.getSalt(),
             remoteIp);
    }

    /**
     * Initializes a new {@link AntiAbuseWrapper}.
     *
     * @param antiAbuseService The {@link GuardAntiAbuseService} to wrap.
     * @param secretKeyRing The {@link PGPSecretKeyRing} to check the password against.
     * @param contextId The constext ID of the user
     * @param userId The user ID
     * @param password The password to check
     * @param salt The salt of the key
     * @param remoteIp The remote IP of the request
     */
    public AntiAbuseWrapper(GuardAntiAbuseService antiAbuseService,
                            PGPSecretKeyRing secretKeyRing,
                            int contextId,
                            int userId,
                            String password,
                            String salt,
                            String remoteIp) {
        this.antiAbuseService = Objects.requireNonNull(antiAbuseService, "antiAbuseService must not be null");
        this.secretKeyRing = Objects.requireNonNull(secretKeyRing, "secretKeyRing must not be null");
        this.contextId = contextId;
        this.userId = userId;
        this.password = password;
        this.salt = salt;
        this.remoteIp = remoteIp;
    }

    /**
     * Executes a given action only if the user provided the correct password for his key and was not locked out due anti abuse policy.
     *
     * @param action The action to execute
     * @return The result of the action
     * @throws OXException If the action could not be performed due a bad password or due an account lockout.
     */
    public <T> T doAction(AntiAbuseAction<T> action) throws OXException {
        if(antiAbuseService.allowLogin(userId, contextId, password, remoteIp)){
            if(PGPUtil.verifyPassword(secretKeyRing, password,salt)) {
                final boolean success = true;
                antiAbuseService.reportLogin(success, userId, contextId, password, remoteIp);
                return action.doAction();
            }
            else {
                final boolean success = false;
                antiAbuseService.reportLogin(success, userId, contextId, password, remoteIp);
                throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }
        }
        else {
            throw GuardAuthExceptionCodes.LOCKOUT.create();
        }
    }
}
