/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.commons;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keymanagement.commons.exceptions.KeysExceptionCodes;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;

/**
 * Class for a key that was deleted and saved in the deletedKeyTable
 *
 * @author greg
 *
 */
public class DeletedKey {

    private final int userid;
    private final int cid;
    private PGPSecretKeyRing secretKeyRing;
    private String pgpSecret;
    private String recovery;
    private String salt;
    private int version;
    private String email;
    private boolean exposed;

    /**
     * Creating a deleted key object based on a existing key
     *
     * @param key The key to create the deleted key from
     * @throws OXException
     */
    public DeletedKey(GuardKeys key) throws OXException {
        this.userid = key.getUserid();
        this.cid = key.getContextid();
        this.recovery = key.getRecovery();
        this.salt = key.getSalt();
        this.version = key.getVersion();
        this.email = key.getEmail();
        try {
            if (key.getEncodedPGPSecret() != null) {
                this.secretKeyRing = new PGPSecretKeyRing(Base64.decodeBase64(key.getEncodedPGPSecret()), new BcKeyFingerprintCalculator());
            } else {
                this.secretKeyRing = null;
            }
            this.pgpSecret = key.getEncodedPGPSecret();
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } catch (PGPException e) {
            throw KeysExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }
    }

    public DeletedKey(int userId, int cid, String pgpSecret, String recovery, String salt, int version, String email, boolean exposed) throws OXException {
        this.userid = userId;
        this.cid = cid;
        try {
            if (pgpSecret != null) {
                this.secretKeyRing = new PGPSecretKeyRing(Base64.decodeBase64(pgpSecret), new BcKeyFingerprintCalculator());
            } else {
                this.secretKeyRing = null;
            }
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } catch (PGPException e) {
            throw KeysExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }

        this.pgpSecret = pgpSecret;
        this.recovery = recovery;
        this.salt = salt;
        this.version = version;
        this.email = email;
        this.exposed = exposed;
    }

    public PGPSecretKeyRing getSecretKeyRing() {
        return secretKeyRing;
    }

    public void setSecretKeyRing(PGPSecretKeyRing secretKeyRing) {
        this.secretKeyRing = secretKeyRing;
    }

    public String getRecovery() {
        return recovery;
    }

    public void setRecovery(String recovery) {
        this.recovery = recovery;
    }

    public String getSalt() {
        return salt;
    }

    public void setSalt(String salt) {
        this.salt = salt;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    public String getEmail() {
        return email;
    }

    public void setEmail(String email) {
        this.email = email;
    }

    public String getPGPSecret() {
        return pgpSecret;
    }

    public void setPgpSecret(String pgpSecret) {
        this.pgpSecret = pgpSecret;
    }

    public boolean isExposed() {
        return exposed;
    }

    public void setExposed(boolean exposed) {
        this.exposed = exposed;
    }

    public int getUserId() {
        return userid;
    }

    public int getCid() {
        return cid;
    }

    /**
     * Verify the password is correct for this key
     *
     * @param password
     * @return
     */
    public boolean verifyPassword(String password) {
        char[] pass = CipherUtil.getSHA(password, salt).toCharArray();
        PGPSecretKey sec_key = secretKeyRing.getSecretKey();
        try {
            PGPPrivateKey decryptor = sec_key.extractPrivateKey(new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(pass));
            if (decryptor != null) {
                return (true);
            }
        } catch (PGPException ex) {
            return (false);
        }
        return (false);
    }

    /**
     * Get arraylist of userids for this keyring
     *
     * @return
     */
    public ArrayList<String> getUserIds() {
        ArrayList<String> ids = new ArrayList<String>();
        @SuppressWarnings("unchecked") Iterator<String> it = secretKeyRing.getSecretKey().getUserIDs();
        while (it.hasNext()) {
            ids.add(it.next());
        }
        return (ids);
    }

    /**
     * Get arraylist of fingerprints for this keyring
     *
     * @return
     */
    public ArrayList<String> getKeyFingerprints() {
        ArrayList<String> fps = new ArrayList<String>();
        Iterator<PGPSecretKey> keys = secretKeyRing.getSecretKeys();
        while (keys.hasNext()) {
            PGPSecretKey skey = keys.next();
            fps.add(PGPUtil.getFingerPrintInBlocks(skey.getPublicKey().getFingerprint()));
        }
        return (fps);
    }
}
