
package com.openexchange.guard.hkpclient.services;

import java.util.ArrayList;
import java.util.Collection;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.DefaultInterests;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadable;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.hkpclient.client.HKPClient;
import com.openexchange.guard.hkpclient.osgi.Services;

/**
 * {@link RemoteHKPClientService} - default implementation for querying public HKP servers
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class RemoteHKPClientService implements HKPClientService, Reloadable  {

    private static final Logger logger = LoggerFactory.getLogger(RemoteHKPClientService.class);
    private Collection<String> configuredRemoteHKPServers;
    private Collection<String> untrustedRemoteHKPServers;
    private final HKPClientService delegate;
    private final GuardConfigurationService guardConfigService;

    public RemoteHKPClientService() throws OXException {
        this(null);
    }

    public RemoteHKPClientService(HKPClientService delegate) throws OXException {
        this.delegate = delegate;
        this.guardConfigService = Services.getService(GuardConfigurationService.class);
        loadConfig();
    }

    /**
     * Internal method to load the required configuration settings
     *
     * @throws OXException
     */
    private void loadConfig() throws OXException {
        // Depreciating publicPGPDirectory.  Now using trustedPGPDirectory
        String urlStrings = guardConfigService.getProperty(GuardProperty.publicPGPDirectory) + "," + guardConfigService.getProperty(GuardProperty.trustedPGPDirectory);
        this.configuredRemoteHKPServers = getRemoteHKPServers(urlStrings);
        this.untrustedRemoteHKPServers = getRemoteHKPServers(guardConfigService.getProperty(GuardProperty.untrustedPGPDirectory));
    }

    /**
     * Internal method to read configured HKP servers from configuration
     *
     * @return A list of configured HKP servers
     * @throws OXException
     */
    private Collection<String> getRemoteHKPServers(String urlStrings) throws OXException {
        Collection<String> hkpServers = new ArrayList<String>();
        String[] urls = urlStrings.split(",");
        for (String url : urls) {
            url = url.trim();
            if (url.startsWith("htt") || url.startsWith("hkp")) {
                if (!url.endsWith("?")) {
                    if (!url.endsWith("/")) {
                        url += "/";
                    }
                }
                hkpServers.add(url);
            }
        }
        return hkpServers;
    }

    /**
     * Internal method to query a list HKP Servers
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param hkpServers a list of HKP servers to query
     * @return A list of found keys
     * @throws Exception
     */
    private ArrayList<PGPPublicKeyRing> findByEmail(String clientToken, Collection<String> hkpServers, String email) throws Exception {
        ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();
        for (String hkpServer : hkpServers) {
            Collection<PGPPublicKeyRing> result = new HKPClient(clientToken).findKeys(hkpServer, email);
            if (!result.isEmpty()) {
                ret.addAll(ret.size(), result);
            }
        }
        return ret;
    }

    private PGPPublicKeyRing findById (String clientToken, Collection<String> hkpServers, String id) throws Exception {
    	for (String hkpServer : hkpServers) {
            PGPPublicKeyRing result = new HKPClient(clientToken).getKeyById(hkpServer, id);
            if (result != null) {
                return result;
            }
        }
    	return null;
    }

    private RemoteKeyResult findInternal (String clientToken, Long id, String email) throws Exception {
    	String search = "0x" + LongUtil.longToHexString(id);
    	PGPPublicKeyRing ring = findById (clientToken, configuredRemoteHKPServers, search);
    	if (ring != null) {
    	    logger.debug("Remote key found in trusted HKP server");
    		return new RemoteKeyResult (ring, HKPKeySources.TRUSTED_HKP_REMOTE_SERVER);
    	}
    	else {
    	    ring = findById (clientToken, untrustedRemoteHKPServers, search);
    	    if (ring != null) {
    	        logger.debug("Remote keys found in public HKP server");
    	        return new RemoteKeyResult (ring, HKPKeySources.PUBLIC_HKP_REMOTE_SERVER);
    	    }
    	}
    	RemoteKeyResult result = null;
        if (delegate != null) {
            result = delegate.find(clientToken, email, id);
        }
        return result;
    }

    /**
     * Lookup public keys for email address. Checks SRV records first, then configured public servers
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email
     * @return
     * @throws Exception
     */
    private Collection<RemoteKeyResult> findAllInternal(String clientToken, String email) throws Exception {

        //Searching configured HKP servers
        try {
        	// First check the trusted configured HKP servers
            Collection<PGPPublicKeyRing> ret = findByEmail(clientToken, configuredRemoteHKPServers, email);
            if (ret != null && ret.size() > 0) {
                logger.debug("Remote keys found in trusted HKP server");
                return RemoteKeyResult.createCollectionFrom(ret, HKPKeySources.TRUSTED_HKP_REMOTE_SERVER);
            } else {
            	// If not found, then check the untrusted servers.
            	ret = findByEmail(clientToken, untrustedRemoteHKPServers, email);
                if (ret != null && ret.size() > 0) {
                    logger.debug("Remote keys found in public HKP server");
                    return RemoteKeyResult.createCollectionFrom(ret, HKPKeySources.PUBLIC_HKP_REMOTE_SERVER);
                }
            }
        }
        catch (Exception e) {
            logger.error("Error querying remote HKP server", e);
        }
        if (delegate != null) {
            return delegate.find(clientToken, email);
        }
        return null;
    }

    @Override
    public Collection<RemoteKeyResult> find(String clientToken, String email) throws OXException {
        try {
            return findAllInternal(clientToken, email);
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

	@Override
	public RemoteKeyResult find(String clientToken, String email, Long id) throws OXException {
        try {
            return findInternal(clientToken, id, email);
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
	}

    /* (non-Javadoc)
     * @see com.openexchange.config.Reloadable#reloadConfiguration(com.openexchange.config.ConfigurationService)
     */
    @Override
    public void reloadConfiguration(ConfigurationService configService) {
        try {
            loadConfig();
        }
        catch (OXException e){
            logger.error("Error reloading HKP client configuration.",e);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.config.Reloadable#getInterests()
     */
    @Override
    public Interests getInterests() {
        return DefaultInterests.builder().propertiesOfInterest("com.openexchange.guard.*").configFileNames("guard-core.properties").build();
    }

}
