/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.hkp.internal;

import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.hkp.osgi.Services;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.export.KeyExportUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.ratifier.GuardRatifierService;

/**
 *
 * {@link HKPGet} - The HKP "get" operation requests keys from the keyserver.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v.2.40
 */
public class HKPGet {

    /**
     * The "get" operation requests keys from the keyserver.
     *
     * @param search specifies which key to return
     * @return The key, or null, if no such key was found
     * @throws OXException
     */
    public String get(String search) throws OXException {

        if (search != null && !search.isEmpty()) {
            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            if (search.contains("@")) {  // If email search

                String email = search.trim();
                GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
                validatorService.validate(email);
                GuardKeyService keyService = Services.getService(GuardKeyService.class);
                GuardKeys keys = keyService.getKeys(email);
                if (keys != null) {
                    return KeyExportUtil.export(keys.getPGPPublicKeyRing());
                }

            } else { // Otherwise keyid search

                search = normalizeSearch(search);

                PGPPublicKeyRing resultKeyRing = null;
                if (search.length() > 8) {
                    Long req_id = LongUtil.stringToLong(search);
                    resultKeyRing = pgpKeyService.getPublicKeyRingById(req_id, 0, 0);
                } else {
                    resultKeyRing = pgpKeyService.getPublicKeyRingById(normalizeSearch(search), 0, 0);
                }

                if (resultKeyRing != null) {
                    return KeyExportUtil.export(resultKeyRing);
                }
            }
        }
        return null;
    }

    /**
     * Normalize search parameter. If more than the 16 character ID (Fingerprint), then shorten
     *
     * @param search The search parameter to normalize
     * @return The normalized search parameter
     */
    private String normalizeSearch(String search) {
        search = search.replace(" ", "");  // remove padding from a fingerprint
        search = search.replace("0x", "").toLowerCase();
        if (search.length() > 16) {  // If more than 16, we'll just look at the last 16
            search = search.substring(search.length() - 16);
        }
        return search;
    }
}
