/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.cipher.osgi;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.internal.BouncyCastleCryptoInitializer;
import com.openexchange.guard.cipher.internal.GuardCipherFactoryServiceImpl;
import com.openexchange.guard.cipher.internal.GuardCryptoInitializer;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.configuration.validation.ConfigurationValidator;
import com.openexchange.guard.configuration.validation.InvalidValueHandler;
import com.openexchange.guard.inputvalidation.RangeInputValidator;
import com.openexchange.guard.inputvalidation.WhiteListInputValidator;
import com.openexchange.osgi.HousekeepingActivator;

/**
 * {@link GuardCipherActivator}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardCipherActivator extends HousekeepingActivator {

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { GuardConfigurationService.class };
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        Logger logger = LoggerFactory.getLogger(GuardCipherActivator.class);
        logger.info("Starting bundle {}", context.getBundle().getSymbolicName());

        //Registering bouncy castle provider
        GuardCryptoInitializer cryptoInitializer = new BouncyCastleCryptoInitializer();
        cryptoInitializer.initialize();

        Services.setServiceLookup(this);
        registerService(GuardCipherFactoryService.class, new GuardCipherFactoryServiceImpl());

        //Validating OX Guard Key configuration
        GuardConfigurationService configurationService = this.getService(GuardConfigurationService.class);
        int aesKeyLength = configurationService.getIntProperty(GuardProperty.aesKeyLength);

        ConfigurationValidator validator = new ConfigurationValidator(this.getService(GuardConfigurationService.class));
        //Validate that the configured AES key length is a valid AES key length
        validator.assertValue(GuardProperty.aesKeyLength,
            Integer.class,
            new WhiteListInputValidator<>(new Integer[] { 128, 192, 256 }),
            new InvalidValueHandler<Integer>() {

                @Override
                public void handle(Integer invalidValue) {
                    LoggerFactory.getLogger(GuardCipherActivator.class).error(
                        "OX Guard is unable to operate with invalid configured AES key length of {} (From property {}). Valid values are 128, 192, 256.",
                        invalidValue,
                        GuardProperty.aesKeyLength.getFQPropertyName());
                }
            });

        //Validates that the configured RSA key length is at least 1024 bit. In theory shorter RSA keys are possible but RSA can only encrypt data which is
        //about up to the key length and OX Guard requires at least about 1024 bit.
        validator.assertValue(GuardProperty.rsaKeyLength,
            Integer.class,
            new RangeInputValidator<Integer>(1024, Integer.MAX_VALUE),
            new InvalidValueHandler<Integer>() {

                @Override
                public void handle(Integer invalidValue) {
                    LoggerFactory.getLogger(GuardCipherActivator.class).error(
                        "OX Guard is unable to operate with invalid configured RSA key length of {} (From property {}). The RSA key length must be at least 1024.",
                        invalidValue,
                        GuardProperty.rsaKeyLength.getFQPropertyName());
                }
            });

        logger.info("Guard Cipher Algorithms registered.");
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.HousekeepingActivator#stopBundle()
     */
    @Override
    protected void stopBundle() throws Exception {
        final Logger logger = LoggerFactory.getLogger(GuardCipherActivator.class);
        logger.info("Stopping bundle {}", context.getBundle().getSymbolicName());
        Services.setServiceLookup(null);

        unregisterService(GuardCipherFactoryService.class);

        logger.info("Unregistering Guard Cipher Algorithms");

        super.stopBundle();
    }
}
