/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.s3.internal;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.Region;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.dataobject.EncryptedObject;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.storage.AbstractStorage;
import com.openexchange.guard.storage.Storage;

/**
 * Class for saving encrypted objects to and from an S3 object store.
 * 
 * @author benjamin.otterbach@open-xchange.com
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class S3StorageImpl extends AbstractStorage {

    private static Logger logger = LoggerFactory.getLogger(S3StorageImpl.class);

    private AmazonS3Client s3Client;

    private final GuardConfigurationService guardConfigService;

    /**
     * Initialises a new {@link S3StorageImpl}.
     * 
     * @throws OXException
     */
    public S3StorageImpl(GuardConfigurationService configService) throws OXException {
        super();
        this.guardConfigService = configService;
        initialiseClient(configService);
    }

    /**
     * Initialises the S3 Client
     * 
     * @throws OXException
     */
    private void initialiseClient(GuardConfigurationService guardConfigService) throws OXException {
        String accessKey = guardConfigService.getProperty(GuardProperty.accessKey);
        String secret = guardConfigService.getProperty(GuardProperty.secretKey);
        String region = guardConfigService.getProperty(GuardProperty.region);
        String bucketName = guardConfigService.getProperty(GuardProperty.bucketName);
        String endpoint = guardConfigService.getProperty(GuardProperty.endPoint);

        BasicAWSCredentials credentials = new BasicAWSCredentials(accessKey, secret);

        /*
         * Instantiate client
         */
        s3Client = new AmazonS3Client(credentials);
        s3Client.setS3ClientOptions(new S3ClientOptions().withPathStyleAccess(true));

        /*
         * Configure client
         */
        if (Strings.isNotEmpty(endpoint)) {
            s3Client.setEndpoint(endpoint);
        } else {
            try {
                s3Client.setRegion(com.amazonaws.regions.Region.getRegion(Regions.fromName(region)));
            } catch (IllegalArgumentException e) {
                logger.error("Error at s3Client setRegion", e);
            }
        }

        /*
         * Check if configured S3 bucket exists, otherwise create.
         */
        if (!s3Client.doesBucketExist(bucketName)) {
            s3Client.createBucket(bucketName, Region.fromValue(region));
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#writeObj(java.lang.String, java.lang.String, byte[])
     */
    @Override
    public Boolean writeObj(String directoryPrefix, String ObjId, byte[] data) throws OXException {
        boolean uploadSucess = false;

        /*
         * prepare upload
         */
        try {
            //            if (data.length < UPLOAD_BUFFER_SIZE) {
            /*
             * whole file fits into buffer (this includes a zero byte file), upload directly
             */
            ByteArrayInputStream dataStream = null;
            try {
                dataStream = new ByteArrayInputStream(data);
                ObjectMetadata metadata = new ObjectMetadata();
                metadata.setContentLength(data.length);
                s3Client.putObject(guardConfigService.getProperty(GuardProperty.bucketName), directoryPrefix + Storage.DELIMITER + ObjId, dataStream, metadata);
                uploadSucess = true;
            } finally {
                if (dataStream != null) {
                    dataStream.close();
                }
            }
        } catch (Exception e) {
            logger.error("Error while uploading file to S3 store", e);
        }
        return uploadSucess;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#getEncrObj(java.lang.String, java.lang.String)
     */
    @Override
    public EncryptedObject getEncrObj(String directoryPrefix, String ObjId) throws OXException {
        String bucketName = "";
        try {
            bucketName = guardConfigService.getProperty(GuardProperty.bucketName);
            InputStream in = s3Client.getObject(bucketName, directoryPrefix + Storage.DELIMITER + ObjId).getObjectContent();
            EncryptedObject obj = new EncryptedObject(in);
            in.close();
            return (obj);
        } catch (FileNotFoundException e) {
            logger.error("File \"" + directoryPrefix + Storage.DELIMITER + ObjId + "\" not found in bucket \"" + bucketName + "\"");
            EncryptedObject obj = new EncryptedObject();
            obj.setItemID("not found");
            return (obj);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#getObjectStream(java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getObjectStream(String directoryPrefix, String ObjId) throws OXException {
        try {
            InputStream in = s3Client.getObject(guardConfigService.getProperty(GuardProperty.bucketName), directoryPrefix + Storage.DELIMITER + ObjId).getObjectContent();
            return (in);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#getObjectStream(java.lang.String)
     */
    @Override
    public InputStream getObjectStream(String location) throws OXException {
        try {
            InputStream in = s3Client.getObject(guardConfigService.getProperty(GuardProperty.bucketName), location).getObjectContent();
            return (in);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#deleteObj(java.lang.String)
     */
    @Override
    public void deleteObj(String location) throws OXException {
        s3Client.deleteObject(guardConfigService.getProperty(GuardProperty.bucketName), location);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#deleteEncrObj(java.lang.String)
     */
    @Override
    public void deleteEncrObj(String location) throws OXException {
        deleteObj(location);
    }
}
