/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpcore;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgpcore.services.TokenAuthenticationService;
import com.openexchange.guard.pgpcore.services.UserIdentity;

/**
 * {@link PGPCoreServletUtil}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class PGPCoreServletUtil {

    private static final Logger logger = LoggerFactory.getLogger(PGPCoreServletUtil.class);
    
    /**
     * Constructs an UserIdentity from a given authToken
     * 
     * @param sessionIdentifier The client identifier
     * @param authToken The authToken
     * @return The decrypted UserIdentifier
     * @throws OXException
     */
    private static UserIdentity getIdentityForToken(String sessionIdentifier, String authToken) throws OXException {
        TokenAuthenticationService service = Services.getService(TokenAuthenticationService.class);
        return service.decryptUserIdentity(sessionIdentifier, authToken);
    }
    
    /**
     * Constructs a ServletFileUpload object
     * 
     * @return The servlet ServletFileUpload object to use for parsing multipart data
     */
    public static ServletFileUpload createFileUpload() {
        return new ServletFileUpload(new DiskFileItemFactory());
    }

    /**
     * Parses the JSON part from the incoming input stream
     * 
     * @param jsonInputStream the input stream to parse the JSON from
     * @return The json object parse from the input stream
     * @throws IOException
     */
    public static JsonObject getJsonFromStream(InputStream jsonInputStream) throws IOException {
        ByteArrayOutputStream jsonData = new ByteArrayOutputStream();
        IOUtils.copy(jsonInputStream, jsonData);
        return JsonUtil.parseAsJsonObject(new String(jsonData.toByteArray()));
    }

    /**
     * Parses the signing identity from the given JSON object
     * 
     * @param json The JSON to parse the signer from
     * @return The signer identity, or null if no signer identity was provided in the given JSON
     * @throws Exception
     */
    public static UserIdentity getUserIdentityFromJson(JsonObject json) throws OXException {
        try {
            if (json.has("user")) {
                JsonObject signer = json.getAsJsonObject("user");
                if (signer.has("identity") && signer.has("password")) {
                    return new UserIdentity(signer.get("identity").getAsString(), signer.get("password").getAsString());
                }
                else if (signer.has("session") && signer.has("auth")) {
                    UserIdentity ret = getIdentityForToken(signer.get("session").getAsString(), signer.get("auth").getAsString());
                    if(ret == null) {
                        logger.error("Unable to decode user identity from auth-token");
                    }
                    return ret;
                }
            }
            return null;
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.JSON_ERROR.create(e.getMessage());
        }
    }
}
