/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpcore;

import java.util.ArrayList;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.streaming.FileItemInputStream;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.pgpcore.services.UserIdentity;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import static com.openexchange.guard.servlets.pgpcore.PGPCoreServletUtil.*;

/**
 * {@link EncryptAction}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class EncryptAction extends GuardServletAction {

    private static final Logger log = LoggerFactory.getLogger(GuardServletAction.class);
    private static final String CONTENT_TYPE_PGP_ENCRYPTED = "application/pgp-encrypted";

    /**
     * Parses the recipient list from the given JSON object
     * 
     * @param json the JSON object to parse the recipient list from
     * @return A list of recipients parsed from the given JSON object
     * @throws OXException
     */
    private List<String> getRecipientFromJson(JsonObject json) throws OXException {
        JsonArray recipients = JsonUtil.getArrayFromJson(json, "recipients", true);
        ArrayList<String> ret = new ArrayList<String>();
        for (JsonElement recipient : recipients) {
            ret.add(recipient.getAsString());
        }
        return ret;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {

        ServletFileUpload servletFileUpload = createFileUpload();
        FileItemIterator fileItemIterator = servletFileUpload.getItemIterator(request);
        try (FileItemInputStream jsonStream = new FileItemInputStream("json", fileItemIterator);
            FileItemInputStream dataStream = new FileItemInputStream("data", fileItemIterator);) {

            //Parsing Json metadata
            JsonObject json = getJsonFromStream(jsonStream);
            boolean armored = JsonUtil.getBooleanFromJson(json, "armored");
            List<String> recipients = getRecipientFromJson(json);
            UserIdentity signingIdentity = getUserIdentityFromJson(json);
            if (json.has("user") && signingIdentity == null) {
                throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }
            
            //Encrypting the data
            response.setContentType(CONTENT_TYPE_PGP_ENCRYPTED);
            PGPCryptoService cryptoService = Services.getService(PGPCryptoService.class);
            if (signingIdentity != null) {
                cryptoService.encryptSigned(dataStream, response.getOutputStream(), armored, signingIdentity, recipients);
            }
            else {
                cryptoService.encrypt(dataStream, response.getOutputStream(), armored, recipients);
            }
        }
    }
}
