/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.logging.GuardLogProperties;
import com.openexchange.guard.servlets.authentication.GuardAuthenticationHandler;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.authentication.GuestSessionValidator;
import com.openexchange.guard.servlets.authentication.RestSessionValidator;

/**
 * Represents an OX Guard HTTP action
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public abstract class GuardServletAction {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(GuardServletAction.class);

    private static final String WWW_AUTHENTICATE_HEADER = "WWW-Authenticate";
    private List<GuardAuthenticationHandler> authenticationHandlers;

    /**
     * Initializes a new {@link GuardServletAction}.
     */
    public GuardServletAction() {}

    /**
     * Initializes a new {@link GuardServletAction}.
     *
     * @param authenticationHandlers a set of authentication handlers
     */
    public GuardServletAction(GuardAuthenticationHandler... authenticationHandlers) {
        this.authenticationHandlers = new ArrayList<GuardAuthenticationHandler>(Arrays.asList(authenticationHandlers));
    }

    /**
     * Fluent like method to set the authentication handlers
     *
     * @param authenticationHandlers Authentication handlers to set
     * @return this, with the new authentication handlers set
     */
    public GuardServletAction setAuthenticationHandler(GuardAuthenticationHandler... authenticationHandlers) {
        this.authenticationHandlers = new ArrayList<GuardAuthenticationHandler>(Arrays.asList(authenticationHandlers));
        return this;
    }

    /**
     * Checks authentication using all known authentication handler
     *
     * @param request the request to check authentication for
     * @param oxUserSession The user session or null if the action is not performed in a user context
     * @param response the response, used to set WWW_AUTHENTICATE_HEADER in case the authentication failed
     * @return true if authentication has been granted by all known authentication handlers of the action, false otherwise
     * @throws Exception
     */
    private boolean doAuthentication(HttpServletRequest request, GuardUserSession oxUserSession, HttpServletResponse response) throws Exception {
        if (authenticationHandlers != null && authenticationHandlers.size() > 0) {
            for (GuardAuthenticationHandler handler : authenticationHandlers) {
                if (!handler.authenticate(this, oxUserSession, request)) {
                    //Authentication failed - Adding the www-authenticate-header telling the client how authentication should be done
                    response.addHeader(WWW_AUTHENTICATE_HEADER, handler.getWWWAuthenticateHeader());
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Performs the action after checking the authentication
     *
     * @param request the request related to the action
     * @param response the response related to the action
     * @throws Exception
     */
    public void doAction(HttpServletRequest request, HttpServletResponse response) throws Exception {
        //Validate session if provided by the client and get corresponding context-id and user-id
        GuardUserSession guardUserSession = new GuestSessionValidator().validateSession(request);
        if (guardUserSession == null) {
            guardUserSession = new RestSessionValidator().validateSession(request);
            logAction(request, guardUserSession);
        }
        else {
            logGuestAction(request, guardUserSession);
        }

        //authenticate
        if (doAuthentication(request, guardUserSession, response)) {
            doActionInternal(request, response, guardUserSession);
        }
        else {
            ServletUtils.sendResponse(response, HttpServletResponse.SC_UNAUTHORIZED, "Unauthorized");
        }
    }

    /**
     * Sets and logs current user and delegates to desired action
     *
     * @param request the request related to the action
     * @param userSession a user session or null, if invoked in a non-user context
     * @throws Exception
     */
    private void logGuestAction(HttpServletRequest request, GuardUserSession userSession) {
        if (userSession == null) {
            return;
        }
        GuardLogProperties.put(GuardLogProperties.Name.GUEST_CONTEXT_ID, userSession.getContextId());
        GuardLogProperties.put(GuardLogProperties.Name.GUEST_USER_ID, userSession.getUserId());
        GuardLogProperties.put(GuardLogProperties.Name.GUEST_SESSION_ID, userSession.getSessionId());

        LOG.info(String.format("Command sent \"%s\" from IP %s", ServletUtils.getStringParameter(request, "action"), ServletUtils.getClientIP(request)));
    }

    /**
     * Sets and logs current user and delegates to desired action
     *
     * @param request the request related to the action
     * @param userSession a user session or null, if invoked in a non-user context
     * @throws Exception
     */
    private void logAction(HttpServletRequest request, GuardUserSession userSession) {
        if (userSession != null) {
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_CONTEXT_ID, userSession.getContextId());
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_USER_ID, userSession.getUserId());
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_SESSION_ID, userSession.getSessionId());
        }

        LOG.info(String.format("Command sent \"%s\" from IP %s", ServletUtils.getStringParameter(request, "action"), ServletUtils.getClientIP(request)));
    }

    /**
     * Template method which allows implementation of the action execution by subclasses
     *
     * @param request the request related to the action
     * @param response the response related to the action
     * @param userSession a user session or null, if invoked in a non-user context
     * @throws Exception
     */
    protected abstract void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception;
}
