/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.handler.impl;

import java.io.BufferedReader;
import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.handler.exceptions.GuardResponseHandlerExceptionCodes;
import com.openexchange.guard.keys.GuardMasterKeyService;

/**
 * {@link ResponseHandlerImpl}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since v2.4.0
 */
public class ResponseHandlerImpl implements ResponseHandler {

    private static final Logger LOG = LoggerFactory.getLogger(ResponseHandlerImpl.class);
    private final GuardMasterKeyService masterKeyService;
    private final GuardCipherFactoryService cipherFactoryService;

    public ResponseHandlerImpl(GuardMasterKeyService masterKeyService, GuardCipherFactoryService cipherFactoryService) {
        this.masterKeyService = masterKeyService;
        this.cipherFactoryService = cipherFactoryService;
    }

     @Override
    public JsonObject getJsonAndDecodeEncryptedPasswords(HttpServletRequest request) throws OXException {
        return decodeEncryptedPassword(getJson(request));
    }
    
    @Override
    public JsonObject getJson(HttpServletRequest request) throws OXException{
        String postBody = getPostBody(request);
        if (postBody == null) {
            LOG.warn("Provided post body is empty. Cannot return decoded password.");
            return null;
        }
        return JsonUtil.parseAsJsonObject(postBody);
    }

    /**
     * Get the string contents of a request
     *
     * @param request The request to read from
     * @return the request's body as string
     */
    private String getPostBody(HttpServletRequest request) {
        StringBuffer sb = new StringBuffer();
        String line = null;
        try {
            BufferedReader reader = request.getReader();
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
            reader.close();
        } catch (Exception e) {
            LOG.error("Error reading from request", e);
        }
        if (sb.length() == 0) {
            return null;
        }
        return sb.toString();
    }

    /**
     * Parse JSOn and decode all encrypted passwords
     *
     * @param result
     * @return
     * @throws OXException
     */
    private JsonObject decodeEncryptedPassword(JsonObject result) throws OXException {
        if (result == null) {
            return (null);
        }
        result = decr(result, "e_password", "password");
        result = decr(result, "e_encr_password", "encr_password");
        result = decr(result, "e_ox_password", "ox_password");
        return result;
    }

    private JsonObject decr(JsonObject result, String key, String newkey) throws OXException {
        if (result.has(key)) {
            String encr = result.get(key).getAsString();
            GuardCipherService cipherService = this.cipherFactoryService.getCipherService(GuardCipherAlgorithm.RSA);
            String decoded = cipherService.decrypt(encr, masterKeyService.getMasterKey().getPrivateKey());
            if (decoded != null) {
                result.addProperty(newkey, decoded);
            } else {
                if (encr.length() > 2) {
                    throw GuardResponseHandlerExceptionCodes.BAD_RSA_ERROR.create();
                }
            }
        }
        return result;
    }
}
