/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.internal.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.sig.RevocationReasonTags;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.KeySignatureVerificationResult;
import com.openexchange.guard.keys.dao.OGPGPKey;
import com.openexchange.guard.keys.dao.PGPKeys;
import com.openexchange.guard.keys.internal.pgp.exceptions.PGPKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.OGPGPKeysStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.keys.util.PGPUtil;

/**
 * {@link PGPKeyServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class PGPKeyServiceImpl implements PGPKeyService {

    private static final Logger LOG = LoggerFactory.getLogger(PGPKeyServiceImpl.class);

    /**
     * Initialises a new {@link PGPKeyServiceImpl}.
     */
    public PGPKeyServiceImpl() {
        super();
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#updatePGPKeys(int, int, com.openexchange.guard.keys.dao.GuardKeys)
     */
    @Override
    public void updatePGPKeys(int id, int cid, GuardKeys key) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.updatePublicAndPrivateKey(key);
        PGPKeysStorage keysStorage = Services.getService(PGPKeysStorage.class);
        keysStorage.addPublicKeyIndex(key.getContextid(), 0L, key.getEmail(), key.getPublicKeyRing());
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#getPublicKeyRingById(long, int, int)
     */
    @Override
    public PGPPublicKeyRing getPublicKeyRingById(long reqId, int userid, int cid) throws OXException {
        try {
            PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
            PGPKeys keyMapping = pgpKeysStorage.getById(reqId);
            if (keyMapping != null) {
                if (keyMapping.isLocal()) {
                    GuardKeyService keyService = Services.getService(GuardKeyService.class);
                    GuardKeys key = keyService.getKeysFromEmail(keyMapping.getEmail(), keyMapping.getKeyId());
                    return key.getPublicKeyRing();
                }
            }

            String hexid = LongUtil.longToHexStringTruncated(reqId);
            return (getUploadedPublicKeyRingById(hexid, userid, cid));
        } catch (Exception ex) {
            LOG.error("Error while retrieving public key ring for " + reqId, ex);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#getPublicKeyRingById(java.lang.String, int, int)
     */
    @Override
    public PGPPublicKeyRing getPublicKeyRingById(String hexId, int userid, int cid) throws OXException {
        try {
            PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
            PGPKeys keyMapping = pgpKeysStorage.getByHexId(hexId);
            if (keyMapping != null) {
                if (keyMapping.isLocal()) {
                    GuardKeyService keyService = Services.getService(GuardKeyService.class);
                    GuardKeys key = keyService.getKeysFromEmail(keyMapping.getEmail(), keyMapping.getKeyId());
                    return key.getPublicKeyRing();
                }
            }
            return (getUploadedPublicKeyRingById(hexId, userid, cid));

        } catch (Exception ex) {
            LOG.error("Error while retrieving public key ring for " + hexId, ex);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#getPublicKeyRing(java.lang.String)
     */
    @Override
    public List<PGPPublicKeyRing> getPublicKeyRing(String email) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
        List<PGPKeys> internalKeyMappings = pgpKeysStorage.getInternalByEmail(email);

        //remove duplicate mappings
        Set<PGPKeys> duplicatesRemoved = new TreeSet<PGPKeys>(new Comparator<PGPKeys>() {

            @Override
            public int compare(PGPKeys key1, PGPKeys key2) {
                if (key1.getKeyId() == key2.getKeyId()) {
                    return 0;
                }
                return 1;
            }
        });
        duplicatesRemoved.addAll(internalKeyMappings);
        internalKeyMappings = new ArrayList<PGPKeys>(duplicatesRemoved);

        List<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();
        for (PGPKeys internalMapping : internalKeyMappings) {
            if (internalMapping.isLocal()) {
                GuardKeys key = keyService.getKeysFromEmail(email,internalMapping.getKeyId());
                if (key != null) {
                    ret.add(key.getPublicKeyRing());
                }
            }
        }
        return ret;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#updateDuplicatePGPKey(int, int, com.openexchange.guard.keys.dao.GuardKeys, java.lang.String, java.lang.Long)
     */
    @Override
    public void updateDuplicatePGPKey(int id, int cid, GuardKeys key, String recovery, Long duplicate) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.updateDuplicate(key, recovery, duplicate);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#export(org.bouncycastle.openpgp.PGPPublicKeyRing)
     */
    @Override
    public String export(PGPPublicKeyRing pgpKeyRing) throws OXException {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ArmoredOutputStream arm = new ArmoredOutputStream(out);
            ByteArrayOutputStream bout = new ByteArrayOutputStream();

            Iterator<PGPPublicKey> keys = pgpKeyRing.getPublicKeys();
            while (keys.hasNext()) {
                PGPPublicKey k = keys.next();
                k.encode(bout);
            }
            arm.write(bout.toByteArray());

            arm.close();
            bout.close();
            out.close();

            return (new String(out.toByteArray()));
        } catch (IOException e) {
            throw PGPKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#getPGPPrivateKey(java.lang.String, java.lang.String, org.bouncycastle.openpgp.PGPSecretKeyRing)
     */
    @Override
    public PGPPrivateKey getPGPPrivateKey(String password, String salt, PGPSecretKeyRing keyRing) throws OXException {
        PGPSecretKey sec_key = null;
        Iterator<PGPSecretKey> it = keyRing.getSecretKeys();
        PGPSecretKey master = null;
        while (sec_key == null && it.hasNext()) {
            PGPSecretKey key = it.next();
            if (!key.isMasterKey()) { // We prefer to not return the master.  Only return master if no other encr keys found
                return getDecodedPrivate(key, password, salt);
            } else {
                master = key;
            }
        }
        if (master != null) {
            return getDecodedPrivate(master, password, salt);
        }
        LOG.error("Unable to find PGP encrypting key");
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#getPGPPrivateKey(java.lang.String, java.lang.String, org.bouncycastle.openpgp.PGPSecretKeyRing, long)
     */
    @Override
    public PGPPrivateKey getPGPPrivateKey(String password, String salt, PGPSecretKeyRing keyRing, long id) throws OXException {
        PGPSecretKey secretKey = keyRing.getSecretKey(id);
        if (secretKey != null) {
            return getDecodedPrivate(secretKey, password, salt);
        } else {
            LOG.error("Unable to find PGP key for id {}", Long.toString(id));
            return null;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#getPGPSignKey(org.bouncycastle.openpgp.PGPSecretKeyRing, java.lang.String, java.lang.String)
     */
    @Override
    public PGPPrivateKey getPGPSignKey(PGPSecretKeyRing keyRing, String password, String salt) throws OXException {
        PGPPrivateKey signingKey = getDecodedPrivate(PGPUtil.getSigningKey(keyRing), password, salt);
        if (signingKey == null) {
            LOG.error("Unable to find PGP signing key");
        }
        return signingKey;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#checkSignatures(int, int, org.bouncycastle.openpgp.PGPPublicKeyRing)
     */
    @Override
    public List<KeySignatureVerificationResult> checkSignatures(int id, int cid, PGPPublicKeyRing keyRing) throws OXException {
        ArrayList<KeySignatureVerificationResult> results = new ArrayList<KeySignatureVerificationResult>();
        Iterator<PGPPublicKey> keys = keyRing.getPublicKeys();
        while (keys.hasNext()) {
            PGPPublicKey key = keys.next();
            results.addAll(checkSignatures(id, cid, key));
        }
        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#checkSignatures(int, int, org.bouncycastle.openpgp.PGPPublicKey)
     */
    @SuppressWarnings("unchecked")
    @Override
    public List<KeySignatureVerificationResult> checkSignatures(int id, int cid, PGPPublicKey key) throws OXException {
        Iterator<PGPSignature> sigs = key.getSignatures();
        ArrayList<KeySignatureVerificationResult> results = new ArrayList<KeySignatureVerificationResult>();
        while (sigs.hasNext()) {
            PGPSignature sig = sigs.next();
            results.addAll(verifySignature(id, cid, sig, key));
        }
        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#addUID(org.bouncycastle.openpgp.PGPPrivateKey, org.bouncycastle.openpgp.PGPPublicKeyRing, java.lang.String)
     */
    @Override
    public PGPPublicKeyRing addUID(PGPPrivateKey privateKey, PGPPublicKeyRing publicKeyRing, String userId) throws OXException {
        try {
            PGPPublicKey pub = publicKeyRing.getPublicKey();
            PGPSignatureGenerator generator = new PGPSignatureGenerator(new BcPGPContentSignerBuilder(PGPPublicKey.RSA_GENERAL, org.bouncycastle.openpgp.PGPUtil.SHA1));
            generator.init(PGPSignature.POSITIVE_CERTIFICATION, privateKey);
            PGPSignatureSubpacketGenerator signhashgen = new PGPSignatureSubpacketGenerator();
            generator.setHashedSubpackets(signhashgen.generate());
            PGPSignature certification = generator.generateCertification(userId, pub);
            PGPPublicKey newPubKey = PGPPublicKey.addCertification(pub, userId, certification);
            publicKeyRing = PGPPublicKeyRing.removePublicKey(publicKeyRing, pub);
            publicKeyRing = PGPPublicKeyRing.insertPublicKey(publicKeyRing, newPubKey);
            return publicKeyRing;
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#checkBadUID(com.openexchange.guard.keys.dao.GuardKeys, java.lang.String)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void checkBadUID(GuardKeys keys, String password) throws OXException {
        PGPPublicKey pubkey = keys.getPublicKeyRing().getPublicKey();
        Iterator<String> it = pubkey.getUserIDs();
        while (it.hasNext()) {
            String uid = it.next();
            if (uid.contains(" <")) {
                // already updated
                markUIDUpdated(keys);
            } else {
                int i = uid.indexOf("<");
                if (i < 0) {
                    // no email address found
                    return;
                }
                String newuid = uid.substring(0, i) + " " + uid.substring(i);
                PGPPrivateKey priv;
                try {
                    priv = PGPUtil.decodePrivate(keys.getPGPSecretKeyRing().getSecretKey(), password, keys.getSalt());
                } catch (PGPException e) {
                    throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
                }
                if (priv != null) {
                    keys.setPublicKeyRing(addUID(priv, keys.getPublicKeyRing(), newuid));
                    PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
                    pgpKeyService.updatePGPKeys(keys.getUserid(), keys.getContextid(), keys);
                    markUIDUpdated(keys);
                    LOG.info("Corrected missing space for userid {}", newuid);
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#createPublicKeyRingFromPrivateKey(org.bouncycastle.openpgp.PGPSecretKeyRing)
     */
    @Override
    public PGPPublicKeyRing createPublicKeyRingFromPrivateKey(PGPSecretKeyRing secretKeyRing) throws OXException {
        PGPPublicKeyRing publicKeyRing = null;

        for (Iterator<PGPPublicKey> it = secretKeyRing.getPublicKeys(); it.hasNext();) {
            PGPPublicKey key = it.next();
            if (publicKeyRing == null) {
                try {
                    publicKeyRing = new PGPPublicKeyRing(key.getEncoded(), new JcaKeyFingerprintCalculator());
                } catch (IOException e) {
                    throw PGPKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
                }
            } else {
                publicKeyRing = PGPPublicKeyRing.insertPublicKey(publicKeyRing, key);
            }
        }
        for (Iterator<PGPPublicKey> it2 = secretKeyRing.getExtraPublicKeys(); it2.hasNext();) {
            PGPPublicKey key = it2.next();
            PGPPublicKeyRing.insertPublicKey(publicKeyRing, key);
        }
        return publicKeyRing;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#revokePGPPublicKeys(com.openexchange.guard.keys.dao.GuardKeys, java.lang.String, java.lang.String)
     */
    @Override
    public GuardKeys revokePGPPublicKeys(GuardKeys keys, String password, String reason) throws OXException {
        try {
            PGPSecretKey secretKey = keys.getPGPSecretKey();
            Iterator<PGPPublicKey> pkeys = keys.getPublicKeyRing().getPublicKeys();
            PGPPublicKey master = secretKey.getPublicKey();
            while (pkeys.hasNext()) {
                PGPPublicKey pub = pkeys.next();
                keys.setPublicKeyRing(PGPPublicKeyRing.removePublicKey(keys.getPublicKeyRing(), pub));
                PGPSignatureSubpacketGenerator subHashGenerator = new PGPSignatureSubpacketGenerator();
                PGPSignatureSubpacketGenerator subUnHashGenerator = new PGPSignatureSubpacketGenerator();
                PGPSignatureGenerator generator = new PGPSignatureGenerator(new BcPGPContentSignerBuilder(pub.getAlgorithm(), org.bouncycastle.openpgp.PGPUtil.SHA1));
                PGPPrivateKey priv = PGPUtil.decodePrivate(secretKey, password, keys.getSalt());
                if (pub.isMasterKey()) {
                    generator.init(PGPSignature.KEY_REVOCATION, priv);
                    master = pub;
                } else {
                    generator.init(PGPSignature.SUBKEY_REVOCATION, priv);
                }
                subHashGenerator.setSignatureCreationTime(false, new Date());
                subHashGenerator.setRevocationReason(false, revokeReason(reason), reason);
                subUnHashGenerator.setRevocationKey(false, pub.getAlgorithm(), pub.getFingerprint());
                generator.setHashedSubpackets(subHashGenerator.generate());
                generator.setUnhashedSubpackets(subUnHashGenerator.generate());
                if (pub.isMasterKey()) {
                    PGPSignature signature = generator.generateCertification(pub);
                    pub = PGPPublicKey.addCertification(pub, signature);
                } else {
                    PGPSignature signature = generator.generateCertification(master, pub);
                    pub = PGPPublicKey.addCertification(pub, signature);
                }

                keys.setPublicKeyRing(PGPPublicKeyRing.insertPublicKey(keys.getPublicKeyRing(), pub));
            }

            return keys;
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#signPGPPublicKey(org.bouncycastle.openpgp.PGPPublicKey, com.openexchange.guard.keys.dao.GuardKeys, java.lang.String)
     */
    @Override
    public PGPPublicKey signPGPPublicKey(PGPPublicKey keyToSign, GuardKeys keys, String password) throws OXException {
        ////////FIXME: SIGN PUBLIC KEY.  NOT YET WORKING
        try {
            PGPPrivateKey priv = PGPUtil.decodePrivate(keys.getPGPSigningKey(), password, keys.getSalt());
            PGPSignatureGenerator signatureGenerator = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(keys.getPGPSecretKey().getPublicKey().getAlgorithm(), org.bouncycastle.openpgp.PGPUtil.SHA256));
            signatureGenerator.init(PGPSignature.DIRECT_KEY, priv);
            PGPSignature sig = signatureGenerator.generateCertification(keyToSign);

            return PGPPublicKey.addCertification(keyToSign, sig);
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#signPGPPublicKeyRing(org.bouncycastle.openpgp.PGPSecretKey, org.bouncycastle.openpgp.PGPPrivateKey, org.bouncycastle.openpgp.PGPPublicKey)
     */
    @Override
    public PGPPublicKeyRing signPGPPublicKeyRing(PGPSecretKey secretKey, PGPPrivateKey privateKey, PGPPublicKey keyToSign) throws OXException {
        try {
            PGPSignatureGenerator sGen = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(secretKey.getPublicKey().getAlgorithm(), org.bouncycastle.openpgp.PGPUtil.SHA1).setProvider("BC"));
            sGen.init(PGPSignature.DIRECT_KEY, privateKey);
            PGPSignatureSubpacketGenerator spGen = new PGPSignatureSubpacketGenerator();
            PGPSignatureSubpacketVector packetVector = spGen.generate();
            sGen.setHashedSubpackets(packetVector);
            byte[] signedPublicKey = PGPPublicKey.addCertification(keyToSign, sGen.generate()).getEncoded();

            return new PGPPublicKeyRing(new ByteArrayInputStream(signedPublicKey), new JcaKeyFingerprintCalculator());
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        } catch (IOException e) {
            throw PGPKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#exportPGPPrivateKey(com.openexchange.guard.keys.dao.GuardKeys, java.lang.String)
     */
    @Override
    public String exportPGPPrivateKey(GuardKeys keys, String password) throws OXException {
        try {
            String oldpass = CipherUtil.getSHA(password, keys.getSalt());
            PGPSecretKeyRing newkeyring = PGPUtil.duplicateSecretKeyRing(keys.getPGPSecretKeyRing(), oldpass, password, getPGPEncryptedData());
            if (newkeyring == null) {
                throw PGPKeyExceptionCodes.PGP_ERROR.create("Bad password");
            }
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ArmoredOutputStream arm = new ArmoredOutputStream(out);
            arm.write(newkeyring.getEncoded());
            arm.close();
            out.close();
            return (new String(out.toByteArray()));
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        } catch (IOException e) {
            throw PGPKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    ////////////////////////////////// HELPERS /////////////////////////////////////////

    /**
     * Get the upload public key ring with the specified identifier
     *
     * @param pgpId The public key ring identifier
     * @param userid The user identifier
     * @param cid The context identifier
     * @return The PGP public key ring
     * @throws Exception
     */
    private PGPPublicKeyRing getUploadedPublicKeyRingById(String pgpId, int userid, int cid) throws Exception {
        if (cid == 0) {
            return null;
        }
        OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
        List<OGPGPKey> pgpKeys = ogpgpKeysStorage.getForUserByIds(userid, cid, Arrays.asList(new String[] { pgpId }));
        if (pgpKeys.size() > 0) {
            GuardKeys key = new GuardKeys();
            key.setPGPKeyRingFromAsc(pgpKeys.get(0).getPublicPGPAscData());
            return key.getPublicKeyRing();
        }
        return null;
    }

    /**
     * Extracts the PGP private key
     *
     * @param secretKey The secret key
     * @param password The password
     * @param salt The salt
     * @return The PGP private key
     * @throws PGPException
     */
    private PGPPrivateKey getDecodedPrivate(PGPSecretKey secretKey, String password, String salt) throws OXException {
        char[] pass = CipherUtil.getSHA(password, salt).toCharArray();
        try {
            //org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor decryptor = new org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyDecryptorBuilder(new org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider()).build(pass);
            return secretKey.extractPrivateKey(new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(pass));
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Verify the specified signature in the specified public key for the specified user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param signature The signature to verify
     * @param publicKey The public key
     * @return A list with the verification results
     * @throws OXException
     */
    @SuppressWarnings("unchecked")
    private List<KeySignatureVerificationResult> verifySignature(int userid, int cid, PGPSignature signature, PGPPublicKey publicKey) throws OXException {
        ArrayList<KeySignatureVerificationResult> results = new ArrayList<KeySignatureVerificationResult>();
        if (signature.getKeyID() == 0) {
            results.add(new KeySignatureVerificationResult("", 0, signature.getKeyID()));
            return (results);
        }
        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        PGPPublicKeyRing ring = pgpKeyService.getPublicKeyRingById(signature.getKeyID(), userid, cid);
        if (ring == null) {
            results.add(new KeySignatureVerificationResult("", 0, signature.getKeyID()));
            return (results);
        }
        PGPPublicKey sigKey = ring.getPublicKey(signature.getKeyID());
        if (sigKey == null) {
            results.add(new KeySignatureVerificationResult("", 0, signature.getKeyID()));
            return (results);
        }
        String uid = "";
        Iterator<String> it = sigKey.getUserIDs();
        if (it.hasNext()) {
            uid = it.next();
        }
        try {
            signature.init(new JcaPGPContentVerifierBuilderProvider(), sigKey);
        } catch (PGPException e) {
            throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }
        if (signature.getSignatureType() == PGPSignature.KEY_REVOCATION) {
            try {
                if (signature.verifyCertification(publicKey)) {
                    KeySignatureVerificationResult result = new KeySignatureVerificationResult("", signature.getSignatureType(), uid);
                    results.add(result);
                    return (results);
                }
            } catch (PGPException e) {
                throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
            }
        }
        // Check if signed userid
        Iterator<String> userids = publicKey.getUserIDs();
        boolean verified = false;
        while (userids.hasNext()) {
            String id = userids.next();
            try {
                byte[] sig = signature.getSignature();
                if (sig.length == 255) {
                    LOG.debug("Bad signature length");
                    KeySignatureVerificationResult result = new KeySignatureVerificationResult("", -1, uid);
                    results.add(result);
                    return (results);
                }
                if (signature.verifyCertification(id, publicKey)) {
                    KeySignatureVerificationResult result = new KeySignatureVerificationResult(id, signature.getSignatureType(), uid);
                    results.add(result);
                    verified = true;
                }
            } catch (Exception e) {
                LOG.error("Problem verifying signature ", e);
            }
        }
        // If not, check if signed attribute (picture, etc)
        if (!verified) {
            Iterator<PGPUserAttributeSubpacketVector> userattr = publicKey.getUserAttributes();
            while (userattr.hasNext()) {
                PGPUserAttributeSubpacketVector id = userattr.next();
                try {
                    if (signature.verifyCertification(id, publicKey)) {
                        KeySignatureVerificationResult result = new KeySignatureVerificationResult("image", signature.getSignatureType(), uid);
                        result.setImageType(id.getImageAttribute().getType());
                        result.setImage(id.getImageAttribute().getImageData());
                        results.add(result);
                        verified = true;
                    }
                } catch (PGPException e) {
                    throw PGPKeyExceptionCodes.PGP_ERROR.create(e, e.getMessage());
                }
            }
        }
        if (!verified) {
            try {
                if (signature.verifyCertification(ring.getPublicKey(signature.getKeyID()), publicKey)) {
                    KeySignatureVerificationResult result = new KeySignatureVerificationResult("", signature.getSignatureType(), uid);
                    results.add(result);
                    verified = true;
                }
            } catch (Exception e) {

            }
        }
        if (!verified) {
            KeySignatureVerificationResult result = new KeySignatureVerificationResult("", -1, uid);
            results.add(result);
        }
        return results;
    }

    /**
     * Mark the user with the specified key as updated
     *
     * @param key The {@link GuardKeys} of the user
     */
    private void markUIDUpdated(GuardKeys key) {
        try {
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            ogKeyTableStorage.updateKeyVersion(key, 0);
        } catch (Exception e) {
            LOG.error("Problem updating version from -1 for PGP correction, User: {}, CID: {}", key.getUserid(), key.getContextid());
        }
    }

    /**
     * Convert the specified reason string a byte representation. See {@link RevocationReasonTags}
     *
     * @param reason The reason in string
     * @return The byte representation
     */
    private byte revokeReason(String reason) {
        switch (reason) {
            case "NO_REASON":
                return RevocationReasonTags.NO_REASON;
            case "KEY_SUPERSEDED":
                return RevocationReasonTags.KEY_SUPERSEDED;
            case "KEY_COMPROMISED":
                return RevocationReasonTags.KEY_COMPROMISED;
            case "KEY_RETIRED":
                return RevocationReasonTags.KEY_RETIRED;
            case "USER_NO_LONGER_VALID":
                return RevocationReasonTags.USER_NO_LONGER_VALID;
        }
        return RevocationReasonTags.NO_REASON;
    }

    private int getPGPEncryptedData() throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        int keyLength = guardConfigService.getIntProperty(GuardProperty.rsaKeyLength);
        if (keyLength == 256) {
            return (PGPEncryptedData.AES_256);
        }
        return (PGPEncryptedData.AES_128);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.PGPKeyService#importPGPPrivateKey(com.openexchange.guard.keys.dao.GuardKeys, java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public GuardKeys importPGPPrivateKey(GuardKeys keys, String pgpPrivateAsc, String pgpPassword, String guardPassword) throws OXException {
        // PGPUtil.getDecoderStream() will detect ASCII-armor automatically and decode it,
        // the PGPObject factory then knows how to read all the data in the encoded stream
        Object o;
        try {
            PGPObjectFactory factory = new PGPObjectFactory(org.bouncycastle.openpgp.PGPUtil.getDecoderStream(new java.io.ByteArrayInputStream(pgpPrivateAsc.getBytes("UTF-8"))), new BcKeyFingerprintCalculator());
            // these files should really just have one object in them,
            // and that object should be a PGPPublicKeyRing.
            o = factory.nextObject();
        } catch (IOException e) {
            throw PGPKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
        if (o instanceof PGPSecretKeyRing) {
            PGPSecretKeyRing secretRing = (PGPSecretKeyRing) o;
            keys.setPGPSecretKeyRing(secretRing);

            String newpass = CipherUtil.getSHA(guardPassword, keys.getSalt());

            PGPSecretKeyRing newkeyring = null;
            try {
                newkeyring = PGPUtil.duplicateSecretKeyRing(keys.getPGPSecretKeyRing(), pgpPassword, newpass, getPGPEncryptedData());
            } catch (Exception ex) {
                if (ex.getMessage().contains("checksum")) {
                    throw PGPKeyExceptionCodes.PGP_ERROR.create(ex, "Bad password");
                } else {
                    LOG.error("Error changing password of imported key ", ex);
                    return null;
                }
            }
            if (newkeyring == null) {
                LOG.error("Error importing PGP Private key");
                return null;
            } else {
                keys.setPGPSecretKeyRing(newkeyring);
            }

            try {
                PGPPrivateKey pkey = PGPUtil.decodePrivate(keys.getPGPSecretKey(), guardPassword, keys.getSalt());
                if (pkey == null) {
                    LOG.error("Error decoding PGP Private key for inport");
                    return null;
                }
            } catch (PGPException e) {
                LOG.error("Problem importing PGP Private ", e);
                return null;
            }

        }
        return keys;
    }
}
