/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.internal.key;

import java.io.IOException;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Date;
import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.generators.RSAKeyPairGenerator;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.RSAKeyGenerationParameters;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.crypto.util.PublicKeyFactory;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.openpgp.operator.PGPDigestCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.bouncycastle.openpgp.operator.bc.BcPGPKeyPair;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPKeyConverter;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.GuardMasterKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.exceptions.GuardCryptoExceptionCodes;
import com.openexchange.guard.keys.exceptions.GuardKeyTableExceptionCodes;
import com.openexchange.guard.keys.exceptions.KeysExceptionCodes;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;
import com.openexchange.guard.keys.storage.KeyCacheStorage;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.keys.util.PGPUtil;

/**
 * {@link GuardKeyServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardKeyServiceImpl implements GuardKeyService {

    private static final Logger LOG = LoggerFactory.getLogger(GuardKeyServiceImpl.class);
    private static final int[] PREFERRED_SYMETRIC_ALGORITHMS = new int[] { SymmetricKeyAlgorithmTags.AES_256, SymmetricKeyAlgorithmTags.AES_192, SymmetricKeyAlgorithmTags.AES_128 };
    private static int[] PREFERRED_HASH_ALGORITHMS = new int[] { HashAlgorithmTags.SHA512, HashAlgorithmTags.SHA384, HashAlgorithmTags.SHA256, HashAlgorithmTags.SHA224, HashAlgorithmTags.SHA1 };

    /**
     * Initialises a new {@link GuardKeyServiceImpl}.
     */
    public GuardKeyServiceImpl() {
        super();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#getKeys(int, int)
     */
    @Override
    public GuardKeys getKeys(int id, int cid) throws OXException {
        int shard = 0;
        if (cid < 0) {
            GuardShardingService sharding = Services.getService(GuardShardingService.class);
            shard = sharding.getShard(id, cid);
        }
        return getKeys(id, cid, shard, null, true);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#getKeys(int, int, int)
     */
    @Override
    public GuardKeys getKeys(int id, int cid, int shard) throws OXException {
        return getKeys(id, cid, shard, null, true);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#getKeys(int, int, long, boolean)
     */
    @Override
    public GuardKeys getKeys(int id, int cid, long keyId, boolean current) throws OXException {
        int shard = 0;
        if (cid < 0) {
            GuardShardingService sharding = Services.getService(GuardShardingService.class);
            shard = sharding.getShard(id, cid);
        }
        return getKeys(id, cid, shard, Long.toString(keyId), current);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#getKeys(int, int, int, java.lang.String, boolean)
     */
    @Override
    public GuardKeys getKeys(int id, int cid, int shard, String keyId, boolean current) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        if (current) {
            return ogKeyTableStorage.getCurrentKeyForUser(id, cid, shard);
        }
        return ogKeyTableStorage.getKeyForUserById(id, cid, shard, keyId);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#storeKeys(com.openexchange.guard.keys.dao.GuardKeys, boolean)
     */
    @Override
    public GuardKeys storeKeys(GuardKeys key, boolean userCreated) throws OXException {
        GuardKeys ret = null;

        //Getting the shard to use
        int shard = 0;
        GuardShardingService shardingService = Services.getService(GuardShardingService.class);
        if (key.getContextid() < 0) {// If guest, get shard number, then check tables exist
            shard = shardingService.getNextShard();
            if (shard == 0) {
                LOG.error("Unable to get non-ox shard id");
                return null;
            }
        }

        //Adding the key to the storage
        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);
        keyTableStorage.insert(key, userCreated, shard);
        //retrieve the created key in order to get the userid which was created while inserting
        ret = keyTableStorage.getKeyForEmailAndContext(key.getEmail(), key.getUserid(), key.getContextid(), shard);

        //Store key mappings
        if (ret.getUserid() > 0) {
            //Adding the new key to the email mapping storage
            EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
            ogEmailStorage.insertOrUpdate(ret.getEmail(), ret.getContextid(), ret.getUserid(), shard);

            //Adding the new key to the PGP key mapping storage
            PGPKeysStorage keysStorage = Services.getService(PGPKeysStorage.class);
            keysStorage.addPublicKeyIndex(ret.getContextid(), 0L, ret.getEmail(), ret.getPublicKeyRing());
        }

        return ret;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#getKeysFromEmail(java.lang.String)
     */
    @Override
    public GuardKeys getKeysFromEmail(String email) throws OXException {
        return getKeysFromEmail(email, 0L);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#getKeysFromEmail(java.lang.String, long)
     */
    @Override
    public GuardKeys getKeysFromEmail(String email, long keyId) throws OXException {
        GuardKeys ret = null;

        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        Email ogEmail = ogEmailStorage.getByEmail(email);

        if (ogEmail != null) {
            if (keyId != 0L) {
                ret = getKeys(ogEmail.getUserId(), ogEmail.getContextId(), ogEmail.getShardingId(), Long.toString(keyId), false);
            } else {
                ret = getKeys(ogEmail.getUserId(), ogEmail.getContextId(), ogEmail.getShardingId());
            }
        }
        return ret;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#updateKeys(int, int, com.openexchange.guard.keys.dao.GuardKeys, java.lang.String, boolean)
     */
    @Override
    public void updateKeys(int id, int cid, GuardKeys key, String recovery, boolean reset) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.updatePassword(key, recovery, !reset /* setLastMod */);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#createRecovery(com.openexchange.guard.keys.dao.GuardKeys, java.lang.String)
     */
    @Override
    public String createRecovery(GuardKeys keys, String password) throws OXException {
        if (Strings.isEmpty(password)) {
            return "-1";
        }
        password = CipherUtil.getSHA(password, keys.getSalt());
        return encryptRecoveryHash(keys, password);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#generateKeyPair(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public GuardKeys generateKeyPair(String name, String email, String password) throws OXException {
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);

            // If we are not using the RSA_Cache, just return real-time created key
            if (!guardConfigService.getBooleanProperty(GuardProperty.rsaCache)) {
                return generateKeys(name, email, password);
            }
            // Load pre-generated key pairs
            AsymmetricCipherKeyPair mst = getKeyPair();
            if (mst == null) {
                LOG.info("No pre-generated key pair for master key available. Generate on-the-flow ...");
                return generateKeys(name, email, password);
            }
            // Create OGPGPKey Pairs
            PGPKeyPair rsa_master = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, mst, new Date());

            AsymmetricCipherKeyPair enc = getKeyPair();
            if (enc == null) {
                LOG.info("No pre-generated key pair for sub key available. Generate on-the-flow ...");
                return generateKeys(name, email, password);
            }
            // SubKey
            PGPKeyPair rsa_encrypt = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, enc, new Date());

            // return generated oxGuardKeys
            return generateKeys(name, email, password, rsa_master, rsa_encrypt);
        } catch (Exception e) {
            LOG.error("Problem retrieving pregenerated keys.  Change in password file???", e);
            return null;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#generateKeys(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public GuardKeys generateKeys(String name, String email, String password) throws OXException {
        // Generate key-pairs
        RSAKeyPairGenerator kpgen = new RSAKeyPairGenerator();
        // 2048 strength. Change for export?
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        int keyLength = guardConfigService.getIntProperty(GuardProperty.rsaKeyLength);
        int rsaCertainty = guardConfigService.getIntProperty(GuardProperty.rsaCertainty);
        kpgen.init(new RSAKeyGenerationParameters(BigInteger.valueOf(0x10001), new SecureRandom(), keyLength, rsaCertainty));

        try {
            // Create master (signing) key -- These had to be RSA_GENERAL FOR GNU TO work with the signatures
            PGPKeyPair rsa_master = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, kpgen.generateKeyPair(), new Date());
            // SubKey
            PGPKeyPair rsa_encrypt = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, kpgen.generateKeyPair(), new Date());
            return generateKeys(name, email, password, rsa_master, rsa_encrypt);
        } catch (PGPException e) {
            throw GuardCryptoExceptionCodes.PGP_ENCODING_DECODING_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#generateKeys(java.lang.String, java.lang.String, java.lang.String, org.bouncycastle.openpgp.PGPKeyPair, org.bouncycastle.openpgp.PGPKeyPair)
     */
    @Override
    public GuardKeys generateKeys(String name, String email, String password, PGPKeyPair rsaMaster, PGPKeyPair rsaEncrypt) throws OXException {
        long start = System.currentTimeMillis();
        String salt = CipherUtil.generateSalt();
        char[] pass = CipherUtil.getSHA(password, salt).toCharArray();
        if (password.equals("")) { // If password is blank (unknown), then just create with salt as password
            pass = CipherUtil.getSHA(salt, salt).toCharArray();
            password = salt;
        }

        // Add self-signature
        PGPSignatureSubpacketGenerator signhashgenGenerator = new PGPSignatureSubpacketGenerator();

        // This is for master, just signing and certifying
        signhashgenGenerator.setKeyFlags(false, KeyFlags.SIGN_DATA | KeyFlags.CERTIFY_OTHER);
        signhashgenGenerator.setPreferredSymmetricAlgorithms(false, PREFERRED_SYMETRIC_ALGORITHMS);
        signhashgenGenerator.setPreferredHashAlgorithms(false, PREFERRED_HASH_ALGORITHMS);

        // Now for subKey (encrypting)
        PGPSignatureSubpacketGenerator enchashgen = new PGPSignatureSubpacketGenerator();
        enchashgen.setKeyFlags(false, KeyFlags.ENCRYPT_COMMS | KeyFlags.ENCRYPT_STORAGE);
        enchashgen.setPreferredSymmetricAlgorithms(false, PREFERRED_SYMETRIC_ALGORITHMS);
        enchashgen.setPreferredHashAlgorithms(false, PREFERRED_HASH_ALGORITHMS);

        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        int keyValidDays = guardConfigService.getIntProperty(GuardProperty.keyValidDays);
        if (keyValidDays != 0) {  // If expiration date is chosen, set now
            int seconds = keyValidDays * 24 * 60 * 60;
            signhashgenGenerator.setKeyExpirationTime(false, seconds);
            enchashgen.setKeyExpirationTime(false, seconds);
        }

        try {
            PGPDigestCalculator sha1Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA1);

            PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());

            // Encrypting the private key
            PBESecretKeyEncryptor keyEncr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(pass);

            // Generate Keyring with name and email, signed
            PGPKeyRingGenerator keyRingGen = new PGPKeyRingGenerator(PGPSignature.POSITIVE_CERTIFICATION, rsaMaster, (name + " <" + email + ">"), sha1Calc, signhashgenGenerator.generate(), null, new BcPGPContentSignerBuilder(rsaMaster.getPublicKey().getAlgorithm(), HashAlgorithmTags.SHA1), keyEncr);

            keyRingGen.addSubKey(rsaEncrypt, enchashgen.generate(), null);

            JcaPGPKeyConverter conv = new JcaPGPKeyConverter();

            // Get the private and public encr RSA keys
            PrivateKey privkey = conv.getPrivateKey(rsaEncrypt.getPrivateKey());
            PublicKey pubkey = conv.getPublicKey(rsaEncrypt.getPublicKey());
            String encryptedPrivateKey = encryptPrivateKey(privkey, password, salt);
            // Save to keys format
            GuardKeys keys = new GuardKeys(keyRingGen, encryptedPrivateKey, pubkey, password, salt);
            LOG.debug("Created PGP/RSA Keys at " + start);
            keys.setEmail(email);
            return keys;
        } catch (PGPException e) {
            throw GuardCryptoExceptionCodes.PGP_ENCODING_DECODING_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#addNewKey(com.openexchange.guard.keys.dao.GuardKeys)
     */
    @Override
    public void addNewKey(GuardKeys key) throws OXException {
        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);

        //removing the "current" flag from existing keys,
        //because the new created key will be the "current" key by default
        keyTableStorage.unsetCurrentFlag(key.getUserid(), key.getContextid());
        //Marking the new key as the "current" key
        key.setCurrent(true);

        //Getting the key with the highest version
        GuardKeys highestKeys = keyTableStorage.getHighestVersionKeyForUser(key.getUserid(), key.getContextid());

        //Incrementing key version, if highest was found, and adding the new key to the storage
        key.setVersion(highestKeys != null ? highestKeys.getVersion() + 1 : 0);
        keyTableStorage.insert(key, true /* set lastMod to NOW() */);

        //Adding the new key to the public key index (lookup table)
        PGPKeysStorage keysStorage = Services.getService(PGPKeysStorage.class);
        keysStorage.addPublicKeyIndex(key.getContextid(), 0L, key.getEmail(), key.getPublicKeyRing());
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#updatePin(int, int, java.lang.String)
     */
    @Override
    public void updatePin(int id, int cid, String pin) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.updatePin(id, cid, pin);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#removePin(java.lang.String)
     */
    @Override
    public void removePin(String email) throws OXException {
        GuardKeys key = getKeysFromEmail(email);

        if (key == null) {
            throw PublicKeyExceptionCodes.KEY_NOT_FOUND.create();
        }
        if (key.getContextid() > 0) {
            throw PublicKeyExceptionCodes.NOT_A_GUEST_ACCOUNT.create();
        }
        storeQuestion(key.getUserid(), key.getContextid(), null, null);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#recoverPasswordHash(int, int)
     */
    @Override
    public String recoverPasswordHash(int id, int cid) throws OXException {
        String hash = "";
        try {
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            GuardKeys currentKey = ogKeyTableStorage.getCurrentKeyForUser(id, cid);
            if (currentKey != null && !Strings.isEmpty(currentKey.getRecovery())) {
                GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
                GuardCipherService rsaCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.RSA);
                GuardKeys masterKeys = ogKeyTableStorage.getGuardDaemonKey();
                PrivateKey privateKey = decryptPrivateKey(masterKeys.getEncodedPrivate(), masterKeyService.getMasterKey().getMC(), masterKeys.getSalt());
                String encrhash = rsaCipherService.decrypt(currentKey.getRecovery(), privateKey);
                if (Strings.isEmpty(encrhash)) {
                    LOG.error("Unable to decode master key.  Possible corrupted data");
                    throw new Exception("Bad master key");
                }
                GuardCipherService aesCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
                String rsapub = currentKey.getEncodedPublic();
                hash = aesCipherService.decrypt(encrhash, masterKeyService.getMasterKey().getRC() + (rsapub == null ? "" : Integer.toString(rsapub.hashCode())), currentKey.getSalt());
            }
        } catch (Exception ex) {
            LOG.error("Error recovering the password hash for " + cid + "/" + id, ex);
        }

        return hash;
    }

    @Override
    public boolean isRecoveryEnabled(int userId, int contextId) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        GuardKeys currentKey = ogKeyTableStorage.getCurrentKeyForUser(userId, contextId);

        if (currentKey != null && !Strings.isEmpty(currentKey.getRecovery())) {
            return true;
        }
        return false;
    }

    @Override
    public String generatePassword() throws OXException {
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        return cipherService.generateRandomPassword();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#resetPassword(java.lang.String, java.lang.String, boolean)
     */
    @Override
    public void resetPassword(String email, String newpass) throws OXException {
        GuardKeys oldkeys = getKeysFromEmail(email);
        if (oldkeys != null) {
            String hash = recoverPasswordHash(oldkeys.getUserid(), oldkeys.getContextid());
            if (hash.equals("")) {
                throw KeysExceptionCodes.NO_RECOVERY_ERROR.create();
            }

            GuardKeys newkeys = changePasswordWithRecovery(hash, newpass, oldkeys);
            String newrecovery = createRecovery(newkeys, newpass);
            updateKeys(oldkeys.getUserid(), oldkeys.getContextid(), newkeys, newrecovery, true);
        } else {
            //No key found for this email so we return null
            LOG.error("Error resetting password for " + email + ". No keys found for email");
            throw KeysExceptionCodes.NO_KEY_ERROR.create(email);
        }
    }

    @Override
    public GuardKeys changeTemporaryPassword (int userid, int cid, String newpass) throws OXException {
    	GuardKeys oldkeys = getKeys(userid, cid);
    	if (oldkeys == null) {
    		LOG.error("Unable to retrieve keys for " + userid + " cid " + cid);
    		throw KeysExceptionCodes.NO_KEY_ERROR.create();
    	}
    	if (oldkeys.getLastup() != null || oldkeys.getQuestion() != null) {  // Only allowed for first use
    		LOG.error("Unable to change temporary password for %d cid %d due to already updated", userid, cid);
    		throw KeysExceptionCodes.PREVIOUSLY_CHANGED.create();
    	}
    	String hash = recoverPasswordHash(oldkeys.getUserid(), oldkeys.getContextid());
    	if (hash.equals("")) {
            throw KeysExceptionCodes.NO_RECOVERY_ERROR.create();
        }
    	GuardKeys newkeys = changePasswordWithRecovery(hash, newpass, oldkeys);
        return (newkeys);

    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#changePassword(java.lang.String, java.lang.String, com.openexchange.guard.keys.dao.GuardKeys)
     */
    @Override
    public GuardKeys changePassword(String oldpass, String newpass, GuardKeys oldKey) throws OXException {
        if (!oldKey.privKeyNull()) {
            PrivateKey priv = decryptPrivateKey(oldKey.getEncodedPrivate(), oldpass, oldKey.getSalt());
            if (priv == null) {
                return null;
            }
            String encodedPrivateKey = encryptPrivateKey(priv, newpass, oldKey.getSalt());
            oldKey.setEncodedPrivate(encodedPrivateKey);
        }
        oldpass = CipherUtil.getSHA(oldpass, oldKey.getSalt());
        newpass = CipherUtil.getSHA(newpass, oldKey.getSalt());
        PGPSecretKeyRing keyring = oldKey.getPGPSecretKeyRing();
        try {
            PGPSecretKeyRing newkeyring = PGPUtil.duplicateSecretKeyRing(keyring, oldpass, newpass, getPGPEncryptedData());
            if (newkeyring == null) {
                return null;
            }
            oldKey.setPGPSecretKeyRing(newkeyring);
            return oldKey;
        } catch (PGPException e) {
            LOG.error("Error changing password: {}", e.getMessage(), e);
            return null;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#changePasswordWithRecovery(java.lang.String, java.lang.String, com.openexchange.guard.keys.dao.GuardKeys)
     */
    @Override
    //FIXME: double code with changePassword, remove that
    public GuardKeys changePasswordWithRecovery(String oldPassHash, String newPass, GuardKeys oldKey) throws OXException {
        if (!oldKey.privKeyNull()) {
            PrivateKey priv = decryptPrivateKeyWithHash(oldKey.getEncodedPrivate(), oldPassHash, oldKey.getSalt());
            if (priv == null) {
                LOG.error("Failed to change password");
                throw KeysExceptionCodes.NO_KEY_ERROR.create(oldKey.getEmail());
            }
            String encodedPrivateKey = encryptPrivateKey(priv, newPass, oldKey.getSalt());
            oldKey.setEncodedPrivate(encodedPrivateKey);
        }
        newPass = CipherUtil.getSHA(newPass, oldKey.getSalt());
        PGPSecretKeyRing keyring = oldKey.getPGPSecretKeyRing();
        try {
            PGPSecretKeyRing newkeyring = PGPUtil.duplicateSecretKeyRing(keyring, oldPassHash, newPass, getPGPEncryptedData());
            if (newkeyring == null) {
                LOG.error("Failed to change password");
                throw KeysExceptionCodes.NO_KEY_ERROR.create(oldKey.getEmail());
            }
            oldKey.setPGPSecretKeyRing(newkeyring);
            return oldKey;
        } catch (PGPException e) {
            LOG.error("Error changing password: {}", e.getMessage(), e);
            throw GuardCryptoExceptionCodes.PGP_ENCODING_DECODING_ERROR.create(e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#storeQuestion(int, int, java.lang.String, java.lang.String)
     */
    @Override
    public boolean storeQuestion(int id, int cid, String question, String answer) throws OXException {
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        String encranswer = null;
        if (answer != null && !question.equals("")) {
            GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
            encranswer = cipherService.encrypt(answer, masterKeyService.getMasterKey().getRC(), question);
        }
        try {
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            ogKeyTableStorage.updateQuestionForUser(id, cid, question, encranswer);
        } catch (Exception e) {
            LOG.error("Error while storing reminder question", e);
            return false;
        }
        return true;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#encryptPrivateKey(java.security.PrivateKey, java.lang.String, java.lang.String)
     */
    @Override
    public String encryptPrivateKey(PrivateKey privateKey, String password, String salt) throws OXException {
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        return cipherService.encrypt(Base64.encodeBase64String(privateKey.getEncoded()), CipherUtil.getSHA(password, salt), salt);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keys.GuardKeyService#decryptPrivateKey(java.lang.String, java.lang.String, java.lang.String, boolean)
     */
    @Override
    public PrivateKey decryptPrivateKey(String privateKey, String password, String salt) throws OXException {
        return decryptPrivateKeyWithHash(privateKey, CipherUtil.getSHA(password, salt), salt);
    }

    @Override
    public PrivateKey decryptPrivateKeyWithHash(String privateKey, String password, String salt) throws OXException {
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        byte[] decoded;
        try {
            decoded = cipherService.decrypt(privateKey, password, salt).getBytes("UTF-8");
            if (decoded.length == 0) {
                throw GuardAuthExceptionCodes.BAD_PASSWORD_DECRYPTION.create();
            }
        } catch (UnsupportedEncodingException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
        // for private keys use PKCS8EncodedKeySpec; for public keys use X509EncodedKeySpec
        PKCS8EncodedKeySpec ks = new PKCS8EncodedKeySpec(Base64.decodeBase64(decoded));

        try {
            KeyFactory kf = KeyFactory.getInstance("RSA");
            PrivateKey priv = kf.generatePrivate(ks);
            return priv;
        } catch (InvalidKeySpecException | NoSuchAlgorithmException e) {
            LOG.error("Unable decode private", e);
            return null;
        }
    }

    /////////////////////////////////////// HELPERS/////////////////////////////////////

    /**
     * Encrypt the recovery hashed password
     *
     * @param keys The guard key to use
     * @param password The hashed password
     * @return The encrypted password
     * @throws OXException
     */
    private String encryptRecoveryHash(GuardKeys keys, String password) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        GuardKeys guardDaemonKey = ogKeyTableStorage.getGuardDaemonKey();

        String recovery = "";
        if (guardDaemonKey != null) {
            GuardCipherService aesCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
            GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
            String encrPass = aesCipherService.encrypt(password, masterKeyService.getMasterKey().getRC() + (keys.getPublicKey() == null ? "" : Integer.toString(keys.getEncodedPublic().hashCode())), keys.getSalt());
            GuardCipherService rsaCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.RSA);
            recovery = rsaCipherService.encrypt(encrPass, guardDaemonKey.getPublicKey());
        }

        if (Strings.isEmpty(recovery) && keys.getUserid() > 0) {  // make sure not for system email
            throw GuardKeyTableExceptionCodes.UNABLE_TO_CREATE_RECOVERY.create(keys.getUserid(), keys.getContextid(), keys.getEmail());
        }
        return recovery;
    }

    /**
     * Get a pre-generated key pair
     *
     * @return
     * @throws IOException
     * @throws OXException
     */
    private synchronized AsymmetricCipherKeyPair getKeyPair() throws IOException, OXException {
        String data = "";
        try {
            KeyCacheStorage keyCacheStorage = Services.getService(KeyCacheStorage.class);
            data = keyCacheStorage.popNext();
        } catch (Exception ex) {
            LOG.error("Error getting prepared key pair", ex);
        }

        if (Strings.isEmpty(data)) {
            return null;
        }

        GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        data = cipherService.decrypt(data, masterKeyService.getMasterKey().getRC(), "key");
        if (Strings.isEmpty(data)) {
            LOG.error("Unable to decode pre-compiled keys.  Password file change??");
            return null;
        }
        StringReader reader = new StringReader(data);
        PEMParser parse = new PEMParser(reader);
        PEMKeyPair kp = (PEMKeyPair) parse.readObject();
        reader.close();

        AsymmetricKeyParameter privKey = PrivateKeyFactory.createKey(kp.getPrivateKeyInfo());
        AsymmetricKeyParameter pubKey = PublicKeyFactory.createKey(kp.getPublicKeyInfo());
        AsymmetricCipherKeyPair kp2 = new AsymmetricCipherKeyPair(pubKey, privKey);
        parse.close();
        reader.close();

        return kp2;
    }

    /**
     * Get the key length
     *
     * @return The key length
     * @throws OXException
     */
    private int getPGPEncryptedData() throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        int keyLength = guardConfigService.getIntProperty(GuardProperty.rsaKeyLength);
        if (keyLength == 256) {
            return PGPEncryptedData.AES_256;
        }
        return PGPEncryptedData.AES_128;
    }

    /**
     * Get the hash algorithm tag
     *
     * @return The hash algorithm tag
     */
    private int getHashAlgorithmTags() {
        return HashAlgorithmTags.SHA256;
    }
}
