
package com.openexchange.guard.keys.internal.hkpclient;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.SRVRecord;
import com.openexchange.exception.OXException;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;

/**
 *
 * {@link SRVHKPClientService} Service for querying DNS SRV configured HKP Servers
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class SRVHKPClientService implements HKPClientService {

    private static final Logger logger = LoggerFactory.getLogger(RemoteHKPClientService.class);
    private final HKPClientService delegate;

    /**
     *
     * Initializes a new {@link SRVHKPClientService}.
     */
    public SRVHKPClientService() {
        this(null);
    }

    /**
     *
     * Initializes a new {@link SRVHKPClientService}.
     * 
     * @param delegate The service to delegate search queries to in a public key ring was not found
     */
    public SRVHKPClientService(HKPClientService delegate) {
        this.delegate = delegate;
    }

    /**
     * Internal method to query HKP servers which were configured in the email-domain's DNS SRV record
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email the email to query
     * @return a list of found PGP key rings for the given email, an empty collection if no keys were found or the email's domain has no HKP SRV set
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findForSRVRecords(String clientToken, String email) throws Exception {
        DNSService dnsService = Services.getService(DNSService.class);
        List<SRVRecord> records = dnsService.getSrvs(email);

        Collection<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing>();
        int current = 0;
        SRVRecord nextcheck;
        while ((nextcheck = dnsService.findNext(current, records)) != null) {
            String target = nextcheck.getTarget().toString();
            if (target.endsWith(".")) {
                target = target.substring(0, target.length() - 1);
            }
            int port = nextcheck.getPort();
            rings = new HKPClient(clientToken).findKeys(target, port, email);
            if (rings != null) {
                return rings;
            }
            current = nextcheck.getPriority();
        }
        return rings;
    }

    /**
     * Internal method to query SRV records
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email The email
     * @return A list of public keys for the email
     * @throws Exception
     */
    private Collection<RemoteKeyResult> findAllInternal(String clientToken, String email) throws Exception {

        //Searching DNS SVR records
        try {
            Collection<PGPPublicKeyRing> results = findForSRVRecords(clientToken, email);
            if (results != null && results.size() > 0) {
                logger.debug("Remote keys found through SRV");
                return RemoteKeyResult.createCollectionFrom(results, RemoteKeyResult.Source.SRV);
            }
        } catch (Exception e) {
            logger.error("Error querying remote HKP server form SRV record", e);
        }
        if (delegate != null) {
            return delegate.find(clientToken, email);
        }
        return new ArrayList<RemoteKeyResult>();
    }

    /**
     * Internal method to find key by id
     * Searches based on the from email domain and queries for the keyid
     * 
     * @param clientToken
     * @param id
     * @param fromEmail
     * @return
     * @throws Exception
     */
    private RemoteKeyResult findInternal(String clientToken, Long id, String fromEmail) throws Exception {
        // To do SRV lookup, we have to search by email address to get domain
        if (fromEmail != null) {
            Collection<RemoteKeyResult> res = findAllInternal(clientToken, fromEmail);
            if (res != null && !res.isEmpty()) {  //If results, then search all to see if has the key by id
                for (PGPPublicKeyRing ring : RemoteKeyResult.getRingsFrom(res)) {
                    PGPPublicKey found = ring.getPublicKey(id);
                    if (found != null) {
                        return new RemoteKeyResult(ring, RemoteKeyResult.Source.SRV);
                    }
                }
            }
        }
        RemoteKeyResult result = new RemoteKeyResult();
        if (delegate != null) {
            result = delegate.find(clientToken, id, fromEmail);
        }
        return result;

    }

    @Override
    public Collection<RemoteKeyResult> find(String clientToken, String email) throws OXException {
        try {
            return findAllInternal(clientToken, email);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public RemoteKeyResult find(String clientToken, Long id, String fromEmail) throws OXException {
        try {
            return findInternal(clientToken, id, fromEmail);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }
}
