
package com.openexchange.guard.keys.internal.hkpclient;

import java.util.ArrayList;
import java.util.Collection;
import org.bouncycastle.openpgp.PGPPublicKeyRing;

/**
 * Handle the result of a remote PGP lookup
 * Assign trust level depending on the type of lookup
 * 
 * @author Greg Hill
 *
 */
public class RemoteKeyResult {

    // Define the source of the keys
    // Different trust levels depending on the source
    // Score above 3 trusted
    public enum Source {
        Trusted(5), DNSSEC(5), SRV(4), Cache(4), Public(1);

        private int trust;

        Source(int trust) {
            this.trust = trust;
        }

        public int getTrust() {
            return trust;
        }

        public String getName() {
            return this.toString();
        }

        public boolean isTrusted() {
            return trust > 3;
        }
    }

    private PGPPublicKeyRing ring = null;
    private Source source;

    // Constructors
    public RemoteKeyResult() {
        this.ring = null;
    }

    public RemoteKeyResult(PGPPublicKeyRing ring, Source source) {
        this.ring = ring;
        this.source = source;
    }

    public PGPPublicKeyRing getRing() {
        return ring;
    }

    public void setRing(PGPPublicKeyRing ring) {
        this.ring = ring;
    }

    public Source getSource() {
        return source;
    }

    public void setSource(Source source) {
        this.source = source;
    }

    public boolean hasResult() {
        return ring != null;
    }

    /**
     * Creates a result collection from the given PGPPublicKeyRing collection
     * @param keyRings The PGPPublicKeyRing collection to create the results from
     * @param source The source of the keyRings
     * @return
     */
    public static Collection<RemoteKeyResult> createCollectionFrom(Collection<PGPPublicKeyRing> keyRings, Source source) {
        ArrayList<RemoteKeyResult> ret = new ArrayList<RemoteKeyResult>();
        for (PGPPublicKeyRing keyRing : keyRings) {
            ret.add(new RemoteKeyResult(keyRing, source));
        }
        return ret;
    }

    /**
     * Extracts a collection of PGPPublicKeyRing from a collection of RemoteKeyResult
     * 
     * @param keyResults The collection to extract the key rings from
     * @return A collection of extracted key rings
     */
    public static Collection<PGPPublicKeyRing> getRingsFrom(Collection<RemoteKeyResult> keyResults) {
        Collection<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();
        if (keyResults != null) {
            for (RemoteKeyResult remoteKeyResult : keyResults) {
                ret.add(remoteKeyResult.getRing());
            }
        }
        return ret;
    }

    /**
     * Checks if all results in a given collection are considered to come from a trusted source
     * 
     * @param keyResults The collection
     * @return True, if all results in the given collection come from a trusted source, false if a least one result does not come from a trusted source
     */
    public static boolean areTrusted(Collection<RemoteKeyResult> keyResults) {
        return getTrusted(keyResults).size() == keyResults.size();
    }

    /**
     * Gets the sub set of results which comes from a trusted source
     * 
     * @param keyResults The results
     * @return A sub set of results which comes from a trusted source
     */
    public static Collection<RemoteKeyResult> getTrusted(Collection<RemoteKeyResult> keyResults) {
        Collection<RemoteKeyResult> ret = new ArrayList<RemoteKeyResult>();
        for (RemoteKeyResult remoteKeyResult : keyResults) {
            if (remoteKeyResult.getSource().isTrusted()) {
                ret.add(remoteKeyResult);
            }
        }
        return ret;
    }
}
