
package com.openexchange.guard.keys.internal.hkpclient;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;
import com.openexchange.guard.keys.storage.RemoteKeyCacheStorage;

/**
 *
 * {@link CachingHKPClientService} Service for querying guard's remote key cache
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class CachingHKPClientService implements HKPClientService {

    private static final Logger logger = LoggerFactory.getLogger(CachingHKPClientService.class);
    private final HKPClientService delegate;

    /**
     * Initializes a new {@link CachingHKPClientService}.
     */
    public CachingHKPClientService() {
        this(null);
    }

    /**
     * Initializes a new {@link CachingHKPClientService}.
     * 
     * @param delegate The service to delegate search queries to in a public key ring was not found
     */
    public CachingHKPClientService(HKPClientService delegate) {
        this.delegate = delegate;
    }

    /**
     * Internal method to query the OX Guard remote key cache for the given email
     * 
     * @param email The email to query
     * @return The key for the given email from the cache, or null if no such key was found in the cache
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findInCache(String email) throws Exception {
        RemoteKeyCacheStorage remoteKeyCacheStorage = Services.getService(RemoteKeyCacheStorage.class);
        List<PGPPublicKeyRing> rings = remoteKeyCacheStorage.getByEmail(email);
        return rings;
    }

    private PGPPublicKeyRing findInCache(Long id) throws Exception {
        RemoteKeyCacheStorage remoteKeyCacheStorage = Services.getService(RemoteKeyCacheStorage.class);
        PGPPublicKeyRing ring = remoteKeyCacheStorage.getByID(id);
        return ring;
    }

    /**
     * Internal method to store one or more rings in the cache for later re-use
     * 
     * @param rings the key rings to save to the cache
     * @throws OXException
     */
    private void writeToCache(PGPPublicKeyRing... rings) throws OXException {
        RemoteKeyCacheStorage remoteKeyCacheStorage = Services.getService(RemoteKeyCacheStorage.class);
        for (PGPPublicKeyRing ring : rings) {
            remoteKeyCacheStorage.insert(ring);
        }
    }

    /**
     * Internal method for searching for public keys within the cache or delegate if not found
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email The email to get the public key rings for
     * @return A list for key rings found for the given email
     * @throws Exception
     */
    private Collection<RemoteKeyResult> findAllInternal(String clientToken, String email) throws Exception {

        //Searching keys in the cache
        Collection<PGPPublicKeyRing> ret = null;
        try {
            ret = findInCache(email);
            if (ret != null && ret.size() > 0) {
                logger.debug("Remote keys found in cache");
                //TODO
                // Change the source to cached trust result
                return RemoteKeyResult.createCollectionFrom(ret, RemoteKeyResult.Source.Cache);
            }
        } catch (Exception e) {
            logger.error("Error querying HKP key cache", e);
        }

        Collection<RemoteKeyResult> result = new ArrayList<RemoteKeyResult>();
        //Search delegate
        if (delegate != null) {
            result = delegate.find(clientToken, email);
        }

        if (result != null && !result.isEmpty()) {
            //Only save from trusted sources
            logger.debug("Writing found keys from HKP server into cache");
            Collection<PGPPublicKeyRing> trustedResults = RemoteKeyResult.getRingsFrom(RemoteKeyResult.getTrusted(result));
            writeToCache(trustedResults.toArray(new PGPPublicKeyRing[trustedResults.size()]));
        }
        return result;
    }

    /**
     * Search for a key by long ID in the remote key cache
     * 
     * @param clientToken
     * @param id
     * @param fromEmail
     * @return
     * @throws Exception
     */
    private RemoteKeyResult findInternal(String clientToken, Long id, String fromEmail) throws Exception {

        PGPPublicKeyRing ring = findInCache(id);
        if (ring != null) {
            return new RemoteKeyResult(ring, RemoteKeyResult.Source.Cache);
        }

        RemoteKeyResult result = new RemoteKeyResult();
        //Search delegate
        if (delegate != null) {
            result = delegate.find(clientToken, id, fromEmail);
        }

        if (result.hasResult()) {
            if (result.getSource().isTrusted()) { // Only save from trusted sources
                logger.debug("Writing found keys from HKP server into cache");
                writeToCache(result.getRing());
            }
        }
        return result;
    }

    @Override
    public Collection<RemoteKeyResult> find(String clientToken, String email) throws OXException {
        try {
            return findAllInternal(clientToken, email);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public RemoteKeyResult find(String clientToken, Long id, String fromEmail) throws OXException {
        try {
            return findInternal(clientToken, id, fromEmail);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }
}
