/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.email.storage.ogEmail.internal;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.guard.database.utils.GuardConnectionWrapper;
import com.openexchange.guard.email.storage.exceptions.GuardEmailStorageExceptionCodes;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.email.storage.osgi.Services;

public class EmailStorageImpl implements EmailStorage {

    @Override
    public void insertOrUpdate(String email, int contextId, int userId, int dbShard) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();
        try {
            insertOrUpdate(connection, email, contextId, userId, dbShard);
        } finally {
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    @Override
    public void insertOrUpdate(Connection connection, String email, int contextId, int userId, int dbShard) throws OXException {

        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(EmailSql.INSERT_OR_UPDATE_STMT);
            stmt.setString(1, email);
            stmt.setInt(2, userId);
            stmt.setInt(3, contextId);
            stmt.setInt(4, dbShard);
            stmt.setInt(5, userId);
            stmt.setInt(6, contextId);
            stmt.setInt(7, dbShard);
            stmt.executeUpdate();
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
        }
    }

    @Override
    public void deleteAllForUser(int contextId, int userId) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();

        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(EmailSql.DELETE_ALL_FOR_USER_STMT);
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            stmt.executeUpdate();
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    @Override
    public void deleteAllForContext(int contextId) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();

        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(EmailSql.DELETE_ALL_FOR_CONTEXT_STMT);
            stmt.setInt(1, contextId);
            stmt.executeUpdate();
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    @Override
    public Email getByEmail(String email) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(EmailSql.SELECT_BY_EMAIL_STMT);
            stmt.setString(1, email);
            resultSet = stmt.executeQuery();

            if (resultSet.next()) {
                return new Email(resultSet.getString("email"), resultSet.getInt("id"), resultSet.getInt("cid"), resultSet.getInt("db"));
            }
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
        return null;
    }

    @Override
    public List<Email> getById(int contextId, int userId) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        List<Email> ret = new ArrayList<Email>();
        try {
            stmt = connection.prepareStatement(EmailSql.SELECT_BY_USERID_STMT);
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            resultSet = stmt.executeQuery();

            while (resultSet.next()) {
                ret.add(new Email(resultSet.getString(3), resultSet.getInt(2), resultSet.getInt(1), resultSet.getInt(4)));
            }
            return ret;
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
    }

    @Override
    public int getHighestGuestId() throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(EmailSql.SELECT_HIGHEST_GUEST_USERID_STMT);
            resultSet = stmt.executeQuery();

            int max = 1;
            if (resultSet.next()) {
                max = resultSet.getInt(1);
            }
            return max;
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.email.storage.ogEmail.EmailStorage#getEmail(int, int)
     */
    @Override
    public String getEmail(int userId, int cid) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getReadOnly(userId, cid, 0);
        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        String email = "";
        try {
            stmt = connectionWrapper.getConnection().prepareStatement(EmailSql.SELECT_EMAIL);
            stmt.setInt(1, cid);
            stmt.setInt(2, userId);

            resultSet = stmt.executeQuery();
            if (resultSet.next()) {
                email = resultSet.getString("mail");
            }
        } catch (SQLException e) {
            throw GuardEmailStorageExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnly(connectionWrapper);
        }
        return email;
    }
}
