/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.common.servlets.utils;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonElement;
import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * Utility class for servlet handling
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 */
public class ServletUtils {

    private static final Logger LOG = LoggerFactory.getLogger(ServletUtils.class);

    /**
     * Gets the remote address of the requesting client
     *
     * @param request the request to get the client address for
     * @return the requesting client's remote address
     */
    public static String getClientIP(HttpServletRequest request) {

        if (request.getHeader("X-Forwarded-For") != null) {
            return (request.getHeader("X-Forwarded-For"));
        }

        if (request.getRemoteAddr().equals("127.0.0.1")) {
            return null;
        }

        return (request.getRemoteAddr());
    }

    /**
     * Sends a response
     *
     * @param response the response object
     * @param returnCode the return code to send
     * @param answer the answer to send
     * @throws IOException
     */
    public static void sendResponse(HttpServletResponse response, int returnCode, String answer) throws IOException {
        response.setContentType("text/html; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(returnCode);
        response.getWriter().print(answer);
    }

    /**
     * Sends 200 - OK
     *
     * @param response the response to send the 200 code to
     * @param type The content type to use
     * @param answer an additional answer to put into the body
     */
    public static void sendOK(HttpServletResponse response, String type, String answer) {
        sendOK(response, type, answer.getBytes(Charset.forName("UTF-8")));
    }

    /**
     * Sends 200 - OK
     *
     * @param response the response to send the 200 code to
     * @param type The content type to use
     * @param answer an additional answer to put into the body
     */
    public static void sendOK(HttpServletResponse response, String type, byte[] answer) {
        response.setContentType(type);
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.flush();
            out.close();
        } catch (IOException e) {
            LOG.error("Error sending OK message", e);
        }
    }

    /**
     * Sends 200 - OK
     *
     * @param response the response to send the 200 code to
     * @param answer an additional answer to put into the body with content type text/html
     */
    public static void sendHtmlOK(HttpServletResponse response, String answer) {
        response.setContentType("text/html; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            response.getWriter().print(answer);
        } catch (IOException e) {
            LOG.error("Error sending OK message", e);
        }
    }

    /**
     * Send JSON response
     *
     * @param response the response to send the 200 code to
     * @param json JSON object to put into the body with content type application/json
     */
    public static void sendJsonOK(HttpServletResponse response, JsonElement json) {
        response.setStatus(HttpServletResponse.SC_OK);
        sendJson(response,json);
    }

    /**
     * Send JSON response
     *
     * @param response the response to send the 200 code to
     * @param json JSON object to put into the body with content type application/json
     */
    public static void sendJson(HttpServletResponse response, JsonElement json) {
        response.setContentType("application/json; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        try {
            response.getWriter().write(json.toString());
        } catch (IOException e) {
            LOG.error("Error sending OK message", e);
        }
    }

    public static void sendAnswer(HttpServletResponse response, int returnCode, String answer) throws IOException {
        response.setContentType("text/html; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(returnCode);
        response.getWriter().print(answer);
    }

    /**
     * Sends 406 - Not Acceptable
     *
     * @param response the response to send the 406 code to
     * @param answer body text
     */
    public static void sendNotAcceptable(HttpServletResponse response, String answer) {
        response.setContentType("text/html; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
        try {
            response.getWriter().print(answer);
        } catch (IOException e) {
            LOG.error("Error sending NOT ACCEPTABLE message", e);
        }
    }

    /**
     * Sends 405 - Internal server error
     *
     * @param response the response to send the 405 code to
     * @param answer body text
     */
    public static void sendError(HttpServletResponse response, String answer) {
        response.setContentType("text/html; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        try {
            response.getWriter().print(answer);
        } catch (IOException e) {
            LOG.error("Error sending Error message", e);
        }
    }

    /**
     * Get String from HttpServletRequest
     *
     * @param request the request to get the string from
     * @param name the name of the parameter to get as String
     * @param mandatory true to throw a GuardMissingParameter if the parameter was not found, false to return null
     * @return The value of the requested parameter as string, or null if the parameter was not found and mandatory was set to false
     * @throws OXException If the requested parameter was not found and mandatory was set to true
     */
    public static String getStringParameter(HttpServletRequest request, String name, boolean mandatory) throws OXException {
        String item = request.getParameter(name);
        try {
            if (item != null) {
                if (item.equals("null")) {
                    item = null;
                }
            }
        } catch (Exception e) {
            LOG.error("Problem getting parameter " + name, e);
        }

        if (mandatory && (item == null)) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(name);
        }
        return item;
    }

    /**
     * Get String from HttpServletRequest
     *
     * @param request the request to get the string from
     * @param name the name of the parameter to get as String
     * @return The value of the requested parameter as string, or null if the parameter was not found
     */
    public static String getStringParameter(HttpServletRequest request, String name) {
        try {
            return (ServletUtils.getStringParameter(request, name, false));
        } catch (OXException e) {
            return null;
        }
    }

    /**
     * Get integer from HttpServletRequest
     *
     * @param request the request to get the parameter from
     * @param name the name of the parameter to get
     * @param mandatory true to throw a GuardMissingParameter if the parameter was not found, false to return null
     * @return The value of the requested parameter as integer, or 0 if the parameter was not found and mandatory was set to false
     * @throws OXException If the requested parameter was not found and mandatory was set to true
     */
    public static Integer getIntParameter(HttpServletRequest request, String name, boolean mandatory) throws OXException {
        String item = request.getParameter(name);
        if (item == null || item.equals("null")) {
            if (mandatory) {
                throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(name);
            } else {
                //TODO/FIXME:  Should return NULL with return-value type Integer
                return 0;
            }
        }
        try {
            int val = Integer.parseInt(item);
            return (val);
        } catch (Exception e) {
            LOG.error("Problem getting integer from parameter", e);
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Get integer from HttpServletRequest
     *
     * @param request the request to get the integer from
     * @param name the name of the parameter to get as integer
     * @return The value of the requested parameter as integer, or null if the parameter was not found
     */
    public static Integer getIntParameter(HttpServletRequest request, String name) {
        try {
            return (getIntParameter(request, name, false));
        } catch (OXException e) {
            //TODO/FIXME:  Should return NULL with return-value type Integer
            return 0;
        }
    }

    /**
     * Get long value from HttpServletRequest
     *
     * @param request the request to get the integer from
     * @param name the name of the parameter to get as long
     * @param mandatory true to throw a GuardMissingParameter if the parameter was not found, false to return null
     * @return The value of the requested parameter as long, or NULL if the parameter was not found and mandatory was set to false
     * @throws OXException If the parameter was not found and mandatory was set to true
     */
    public static Long getLongParameter(HttpServletRequest request, String name, boolean mandatory) throws OXException {
        String item = request.getParameter(name);
        if (item == null || item.equals("null")) {
            if (mandatory) {
                throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(name);
            } else {
                return null;
            }
        }
        try {
            return Long.parseLong(item.trim());
        } catch (Exception e) {
            LOG.error("Problem getting long from parameter", e);
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Get long from HttpServletRequest
     *
     * @param request the request to get the long from
     * @param name the name of the parameter to get as long
     * @return The value of the requested parameter as long, or NULL if the parameter was not found
     */
    public static Long getLongParameter(HttpServletRequest request, String name) {
        try {
            return getLongParameter(request, name, false);
        } catch (OXException e) {
            return null;
        }
    }

    /**
     * Get boolean from HttpServletRequest
     *
     * @param request the request to get the boolean value from
     * @param name the name of the parameter to get as boolean
     * @return The value of the requested parameter as boolean
     * @throws OXException If the requested parameter was not found
     */
    public static boolean getBooleanParameter(HttpServletRequest request, String name) {
        try {
            return getBooleanParameter(request, name, false);
        } catch (OXException e) {
            return false;
        }
    }

    /**
     * Get boolean from HttpServletRequest
     *
     * @param request the request to get the boolean value from
     * @param name the name of the parameter to get as boolean
     * @param mandatory true, if the parameter is mandatory, false if not
     * @return The value of the requested parameter as boolean
     * @throws OXException If the requested parameter was not found and mandatory is set to true
     */
    public static boolean getBooleanParameter(HttpServletRequest request, String name, boolean mandatory) throws OXException {
        String var = getStringParameter(request, name, mandatory);
        if (var != null) {
            if (var.toLowerCase().contains("true")) {
                return true;
            }
        }
        return false;
    }

    public static boolean hasParameter(HttpServletRequest request, String name) {
        return request.getParameterMap().containsKey(name);
    }
}
