/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.util;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.ratifier.GuardRatifierService;

/**
 * {@link RatifierUtil}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class RatifierUtil {

    /**
     * Normalises the specified e-mail (in JSON format) with the specified GuardRatifierService
     *
     * @param email The e-mail in JSON format
     * @param ratifierService The GuardRatifierService service to use
     * @throws OXException if the normalisation fails
     */
    public static void normaliseEmail(JsonObject email, GuardRatifierService ratifierService) throws OXException {
        if (email == null) {
            throw new IllegalArgumentException("The provided payload is not a JsonObject");
        }

        if (!email.has("attachments")) {
            return;
        }

        final JsonArray attachments = email.get("attachments").getAsJsonArray();
        for (int i = 0; i < attachments.size(); i++) {
            final JsonObject attach = attachments.get(i).getAsJsonObject();
            if (attach.has("content")) {
                if (!attach.get("content").isJsonNull()) {
                	if (attach.has("content_type")) {
                		if (attach.get("content_type").getAsString().contains("/plain")) {  // If plaintext, don't do full normalizing
                			attach.addProperty("content", safePlainText(attach.get("content").getAsString()));
                			continue;
                		}
                	}
                    String normalisedData = ratifierService.normalise(attach.get("content").getAsString());
                    attach.addProperty("content", normalisedData);
                }
            }
        }
    }

    /**
     * Since plaintext could be displayed on html browser, wipe script tags to be safe
     *
     * @param data
     * @return
     */
    private static String safePlainText (String data) {
    	if (data == null) return (null);
    	data = data.replaceAll("(?i)<script", "<sc**pt");
    	return (data);
    }

    /**
     * Normalises the specified e-mail (in String format) with the specified GuardRatifierService
     *
     * @param email The e-mail in String format
     * @param ratifierService The GuardRatifierService service to use
     * @throws OXException if the normalisation fails
     */
    public static void normaliseEmail(String email, GuardRatifierService ratifierService) throws OXException {
        if (Strings.isEmpty(email)) {
            return;
        }
        JsonParser parser = new JsonParser();
        try {
            JsonElement element = parser.parse(email);
            if (element.isJsonObject()) {
                normaliseEmail(element.getAsJsonObject(), ratifierService);
            }
        } catch (JsonSyntaxException e) {
            // Ignore the exception
        }
    }
}
