/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.util.Collection;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.PublicKeyService;
import com.openexchange.guard.keys.dao.RecipKey;
import com.openexchange.guard.keys.internal.hkpclient.RemoteKeyResult;
import com.openexchange.guard.keys.util.PGPUtil;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PgpHandler;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;

/**
 * Retrieves the public key of a mail recipient
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since 2.4.0
 */
public class GetPublicKeyByMailAction extends GuardServletAction {

    private static Logger logger = LoggerFactory.getLogger(GetPublicKeyByMailAction.class);

    private static final String PARAMETER_CREATE_IF_MISSING = "createIfMissing";
    private static final String PARAMETER_DOWNLOAD = "download";
    private static final String PARAMETER_EMAIL = "email";

    /**
     * Internal method to create a JSON response object which will be send to the client
     *
     * @param requestedEMail the email of the key requested by the client
     * @param download true, if the plain key data should be server, false to server key information
     * @param recipient The key data to return to the client
     * @return A JSON object ready to be returned to the client
     * @throws SignatureException
     * @throws PGPException
     * @throws IOException
     */
    private JsonObject createReponseObject(String requestedEMail, boolean download, RecipKey recipient) throws SignatureException, PGPException, IOException {
        JsonObject ret = new JsonObject();
        ret.addProperty("email", requestedEMail);
        if (recipient != null) {
            String keyreply = "unable";
            ret.addProperty("guest", recipient.isGuest());
            //ret.addProperty("inline", recipient.isInline() || recipient.reqInline());
            // For now, only return inline for keys set for inline, not sending preference
            ret.addProperty("inline", recipient.isInline());

            if (recipient.getPubkey() != null) {
                keyreply = normalize(recipient.getPubkey().toString());
                ret.addProperty("publicKey", keyreply);
            }
            else {
                if (!recipient.isGuest()) {
                    if (recipient.getCid() > 0) {
                        keyreply = "Will be created on first send";
                    }
                }
            }
            ret.add("pgp", new JsonPrimitive(recipient.isPgp()));
            ret.add("expired", new JsonPrimitive(recipient.isExpired()));
            ret.addProperty("publicKey", keyreply);

            if (recipient.isGuest()) {
                ret.addProperty("data", "guest");
            }
            else {
                if (download) {
                    ByteArrayOutputStream out = exportKey(recipient);
                    ret.addProperty("data", new String(out.toByteArray()));
                }
                else {
                    if (recipient.getPgpring() != null) {
                        ret.add("data", PgpHandler.getKeyInfo(recipient.getPgpring()));
                    }
                    else {
                        ret.add("data", new JsonArray());
                    }
                }
            }
        }
        return ret;
    }

    /**
     * Internal method to remove control characters from key data
     *
     * @param keyData the key data to normalize
     * @return the normalized key data
     */
    private String normalize(String keyData) {
        return keyData.replace("\r", "").replace("\n", "");
    }

    /**
     * Internal method to export key data as ASCII-Armored byte stream
     *
     * @param recipientKey The key to export as ASCII-Armored byte stream
     * @return The stream containing the ASCII-Armored key data
     * @throws IOException
     */
    private ByteArrayOutputStream exportKey(RecipKey recipientKey) throws IOException {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream();
            ArmoredOutputStream arm = new ArmoredOutputStream(out)) {
            if (recipientKey.getPgpring() != null) {
                arm.write(recipientKey.getPgpring().getEncoded());
            }
            return out;
        }
    }

    /**
     * Internal method to perform a HKP remote search
     *
     * @param hkpClientToken a token sent to the HKP server
     * @param email the email to search a key for
     * @return The key found, or null if no such key was found
     * @throws NoSuchAlgorithmException
     * @throws UnsupportedEncodingException
     * @throws OXException
     */
    private RecipKey remoteHKPSearch(String hkpClientToken, String email) throws OXException {
        RecipKey ret = null;
        HKPClientService hkpClient = Services.getService(HKPClientService.class);
        Collection<RemoteKeyResult> remoteKeys = hkpClient.find(hkpClientToken, email);
        if (remoteKeys != null && !remoteKeys.isEmpty()) {
            PGPPublicKeyRing remoteKey = remoteKeys.iterator().next().getRing();
            if (remoteKey != null) {
                ret = new RecipKey();
                ret.setPgpring(remoteKey);
                ret.setEmail(email);
                ret.setPgp(true);
                ret.setExpired(PGPUtil.checkAllExpired(remoteKey));
            }
        }
        return ret;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        int userId = userSession.getUserId();
        int cid = userSession.getContextId();
        OxCookie cookie = new OxCookie(request.getCookies());
        String email = ServletUtils.getStringParameter(request, PARAMETER_EMAIL, true);
        GuardRatifierService ratifierService = Services.getService(GuardRatifierService.class);
        ratifierService.validate(email); // Will throw error if invalid
        Api api = new Api(cookie, request);
        String senderEmail = api.getPrimary();
        boolean createIfMissing = ServletUtils.getBooleanParameter(request, PARAMETER_CREATE_IF_MISSING);
        boolean serveAsDownload = ServletUtils.getBooleanParameter(request, PARAMETER_DOWNLOAD);

        PublicKeyService publicKeyService = Services.getService(PublicKeyService.class);
        RecipKey recip = null;
        if (createIfMissing) {
            //Creating keys for GUARD user but not for guests, therefore passing remote=true and send=true, because then no guest account will be created;
            //In addition a remote lookup is missing if remote=true, therefore we need to do a remote HKP lookup here
            //TODO: refactor this behavior in order to be more intuitive
            recip = publicKeyService.getPubKey(email, userId, cid, senderEmail, true, true, request.getServerName());
            if (recip == null) {
                //Doing a remote lookup because it was missing while searching for keys, because we had to pass (true,true) in order to
                //not create a guest account, but the downside of the code is that the remote lookup is not executed is this case.
                //TODO: refactor
                recip = remoteHKPSearch(CipherUtil.getMD5(senderEmail), email);
            }
        }
        else {
            //Looking for the requested key; we do not create keys if missing
            recip = publicKeyService.getPubKey(email, userId, cid, senderEmail, false, false, request.getServerName());
        }

        JsonObject jsonResponse = createReponseObject(email, serveAsDownload, recip);
        response.setContentType("application/json; charset=utf-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.getWriter().write(jsonResponse.toString());
    }
}
