/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.IOException;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import org.apache.commons.io.IOUtils;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.IDNUtil;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.util.RatifierUtil;

/**
 * {@link DecryptedMailResponseFactory}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class DecryptedMailResponseFactory {

	// Send HTML/Text body through normalizer
	private static String normalizePart (Part p) throws OXException, IOException, MessagingException {
		String data = (String) p.getContent();
		GuardRatifierService ratifier = Services.getService(GuardRatifierService.class);
		return (ratifier.normalise(data));
	}

    public static void ParseMultipart(Multipart mp, JsonArray attachments) throws MessagingException, IOException, OXException {
        JsonObject plain = null;
        boolean html = false;
        for (int i = 0; i < mp.getCount(); i++) {
            JsonObject attachment = new JsonObject();
            Part p = mp.getBodyPart(i);
            if (p.getContentType() != null) {
                String contentType = p.getContentType();
                boolean isAttachment = false;
                if (contentType.startsWith("multip")) {
                    ParseMultipart((Multipart) p.getContent(), attachments);
                } else {
                    if (contentType.contains(";")) {
                        contentType = contentType.substring(0, contentType.indexOf(";"));
                    }
                    attachment.add("content_type", new JsonPrimitive(contentType));
                    if (contentType.contains("text/html") || contentType.contains("text/plain")) {
                        attachment.add("content", new JsonPrimitive(normalizePart(p)));
                        if (p.getFileName() == null) {
                            attachment.add("disp", new JsonPrimitive("inline"));
                        } else {
                            isAttachment = true;
                        }
                    } else {
                        attachment.addProperty("content", "");
                    }

                }
                if (p.getFileName() != null) {
                    attachment.add("filename", new JsonPrimitive(p.getFileName()));
                }
                if (p.getDisposition() != null) {
                    attachment.add("disp", new JsonPrimitive(p.getDisposition()));
                }
                attachment.add("size", new JsonPrimitive(p.getSize()));

                attachment.add("id", new JsonPrimitive(i));
                if (contentType.contains("text/html")) {
                    html = true;
                }
                if (contentType.contains("text/plain")) {
                    if (isAttachment) {  // If attachment, then just append
                        attachments.add(attachment);
                    } else {
                        plain = attachment;
                    }
                } else if (!contentType.startsWith("multip")) {
                    attachments.add(attachment);
                }
            }
        }
        if (!html) {  // If no HTML, then add the plain
            if (plain != null) {
                attachments.add(plain);
            }
        }
    }

    public static JsonObject getInlineResponse(PGPDecryptionResult decryptedInlineEmail, String epassword) throws IOException, OXException {
        byte[] decryptedContent = IOUtils.toByteArray(decryptedInlineEmail.getInputStream());
        JsonObject ret = new JsonObject();
        GuardRatifierService ratifierService = Services.getService(GuardRatifierService.class);
        ret.addProperty("newcontent", ratifierService.normalise(new String(decryptedContent, "UTF-8")));
        JsonObject pgpresults = new JsonObject();

        pgpresults.add("signature", new JsonPrimitive(decryptedInlineEmail.hasSignature()));
        pgpresults.add("verified", new JsonPrimitive(decryptedInlineEmail.isSignatureVerified()));
        ret.add("results", pgpresults);
        if (decryptedInlineEmail.getError() != null) {
            ret.add("error", new JsonPrimitive(decryptedInlineEmail.getError()));
        }
        ret.addProperty("epassword", epassword);

        return ret;
    }

    public static JsonObject getMimeResponse(PGPDecryptionResult decryptMimeEmail, String epassword) throws MessagingException, IOException, OXException {
        Session s = Session.getDefaultInstance(new Properties());
        MimeMessage message = new MimeMessage(s,decryptMimeEmail.getInputStream());

        JsonObject data = new JsonObject();

        //Decryption results
        JsonObject Results = new JsonObject();
        Results.add("signature", new JsonPrimitive(decryptMimeEmail.hasSignature()));
        Results.add("verified", new JsonPrimitive(decryptMimeEmail.isSignatureVerified()));
        Results.add("missingPublicKey", new JsonPrimitive(decryptMimeEmail.isMissingPublic()));
        Results.addProperty("epassword", epassword);

        //Mail attachments
        JsonArray attachments = new JsonArray();
        if (message.getContent().getClass() == String.class) {
            //Plain text message
            String content = (String) message.getContent();
            JsonObject attachment = new JsonObject();
            attachment.addProperty("content_type", "text/plain");
            attachment.addProperty("content", content);
            attachment.addProperty("size", content.length());
            attachment.add("disp", new JsonPrimitive("inline"));
            attachment.add("id", new JsonPrimitive(1));
            attachments.add(attachment);
        } else {
            //Multipart message
            Multipart mp = (Multipart) message.getContent();
            ParseMultipart(mp, attachments);
        }
        data.add("attachments", attachments);

        Address[] toaddrs = message.getRecipients(RecipientType.TO);
        if (toaddrs != null) {
            JsonArray to = new JsonArray();
            for (Address add : toaddrs) {
                JsonArray recip = new JsonArray();

                InternetAddress inetAdd = (InternetAddress) add;
                if (inetAdd.getPersonal() != null) {
                    recip.add(new JsonPrimitive(MimeUtility.decodeText(inetAdd.getPersonal())));
                }
                recip.add(new JsonPrimitive(IDNUtil.decodeEmail(inetAdd.getAddress())));
                to.add(recip);
            }
            data.add("to", to);

        }
        Address[] ccaddrs = message.getRecipients(RecipientType.CC);
        if (ccaddrs != null) {
            JsonArray cc = new JsonArray();
            for (Address add : ccaddrs) {
                JsonArray recip = new JsonArray();
                InternetAddress inetAdd = (InternetAddress) add;
                if (inetAdd.getPersonal() != null) {
                    recip.add(new JsonPrimitive(MimeUtility.decodeText(inetAdd.getPersonal())));
                }
                recip.add(new JsonPrimitive(IDNUtil.decodeEmail(((InternetAddress) add).getAddress())));
                cc.add(recip);
            }
            data.add("cc", cc);

        }
        data.add("results", Results);

        // Return normalised result
        RatifierUtil.normaliseEmail(data, Services.getService(GuardRatifierService.class));
        return data;
    }


}
