/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.servlets.GuardServletAction;

/**
 * BASIC-AUTH authentication for OX Guard requests
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class BasicAuthServletAuthenticationHandler implements GuardAuthenticationHandler {

    private static final String WWW_AUTHENTICATE_BASIC_AUTH = "BASIC-AUTH";

    private static final Logger LOG = LoggerFactory.getLogger(BasicAuthServletAuthenticationHandler.class);

    private static final String AUTHORISATION_HEADER = "Authorization";
    private static final String BASIC_AUTH_PREFIX = "Basic";

    private final String userName;
    private final String password;

    /**
     * Initializes a new {@link BasicAuthServletAuthenticationHandler}.
     *
     * @param userName The user name to check
     * @param password The password to check
     */
    public BasicAuthServletAuthenticationHandler(String userName, String password) {
        this.userName = userName;
        this.password = password;
    }

    /**
     * Extracts the credentials from the incoming requests
     *
     * @param request the request
     * @return Credential-Array consisting of user name and password, or null if no credentials were found in the given request
     */
    private String[] getCredentialsFromRequest(HttpServletRequest request) {
        String authorisationHeader = request.getHeader(AUTHORISATION_HEADER);
        if (authorisationHeader != null && authorisationHeader.startsWith(BASIC_AUTH_PREFIX)) {
            String base64Credentials = authorisationHeader.substring(BASIC_AUTH_PREFIX.length());
            byte[] decodedCredentials = org.apache.commons.codec.binary.Base64.decodeBase64(toAsciiBytes(base64Credentials));
            String credentialString = new String(decodedCredentials);
            final String[] credentials = credentialString.split(":");
            return credentials;
        }
        return null;
    }

    /**
     * Gets specified string's ASCII bytes
     *
     * @param str The string
     * @return The ASCII bytes
     */
    private static byte[] toAsciiBytes(final String str) {
        if (null == str) {
            return null;
        }
        final int length = str.length();
        if (0 == length) {
            return new byte[0];
        }
        final byte[] ret = new byte[length];
        str.getBytes(0, length, ret, 0);
        return ret;
    }

    /**
     * Checks the credentials
     *
     * @param credentials The credentials to check
     * @return true, if user name and password are correct, false otherwise
     */
    private boolean checkCredentials(String[] credentials) {
        return credentials[0].equals(userName) && credentials[1].equals(password);
    }

    @Override
    public boolean authenticate(GuardServletAction action, GuardUserSession userSession, HttpServletRequest request) throws Exception {
        String[] credentials = getCredentialsFromRequest(request);
        if (credentials != null) {
            boolean ret = checkCredentials(credentials);
            if (!ret) {
                LOG.error(String.format("BASIC_AUTH authentication failed from IP %s for request %s", ServletUtils.getClientIP(request), request.toString()));
            }
            return ret;
        }
        return false;
    }

    @Override
    public String getWWWAuthenticateHeader() {
        return WWW_AUTHENTICATE_BASIC_AUTH;
    }
}
