
package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.entity.mime.content.StringBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.StringEscapeUtils;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.IDNUtil;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.encryption.EncryptedItemsStorage;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.RecipKey;
import com.openexchange.guard.mail.Attachment;
import com.openexchange.guard.mail.GuardMimeMessage;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.Utilities;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.translation.GuardTranslationService;
import net.htmlparser.jericho.Source;

public class PGPSender {

    private static Logger LOG = LoggerFactory.getLogger(PGPSender.class);

    /**
     * Sends provided email
     *
     * @param mail
     * @throws OXException
     * @throws Exception
     */
    public String sendPGPMail(PGPMail mail, String host) throws OXException {
        boolean lInline = mail.isInline();

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys senderkey = keyService.getKeys(mail.getUserid(), mail.getCid());
        if (senderkey == null) {  // Not a good userid and cid
            throw GuardPGPExceptionCodes.SENDER_KEYS_MISSING.create();
        }

        // Check if inline requested by a recipient
        List<RecipKey> recipients = mail.getRecipients();
        for (int i = 0; i < recipients.size(); i++) {
        	if (recipients.get(i).isInline()) {
                lInline = true;
            }
        }

        // we need to add the sender to the list of recipients so can be read in sent folder
        RecipKey sender = new RecipKey();
        sender.setPgpring(senderkey.getPublicKeyRing());
        mail.getRecipients().add(sender);

        if (lInline && !mail.isDraft()) {  // Drafts will be saved as PGP Mime
            sendPGPInline(mail, senderkey, host);
        } else {
            return sendPGPMime(mail, mail.getRecipients(), senderkey, host);
        }
        return null;
    }

    private void sendPGPInline(PGPMail mail, GuardKeys senderkey, String host) throws OXException {
        List<RecipKey> guestRecip = new ArrayList<RecipKey>();
        List<RecipKey> inlineRecip = new ArrayList<RecipKey>();
        RecipKey sender = new RecipKey();
        List<RecipKey> recipients = mail.getRecipients();

        for (int i = 0; i < recipients.size(); i++) {
            RecipKey recip = recipients.get(i);
            if (recip.getEmail() == null) {
                sender = recip;
            } else {
                if ((recip.getCid() < 0) && !isdirect(recip)) {
                    guestRecip.add(recip);
                } else {
                    inlineRecip.add(recip);
                }
            }
        }
        JsonObject content = mail.getContent();

        // Send guest emails (these are PGP Mime)
        if (guestRecip.size() > 0) {
            guestRecip.add(sender);
            sendPGPMime(mail, guestRecip, senderkey, host);
            // If we have sent some of the emails to a guest, then we need to remove them for the inline emails and mark the headers
            updateJsonRecip(mail, inlineRecip);
        }

        // Modify the json for the inline message and encrypt attachments
        if (inlineRecip.size() > 0) {
            inlineRecip.add(sender);
            sendInline(mail, inlineRecip, senderkey);

            List<Attachment> attachments = mail.getAttachments();
            if (mail.getCid() > 0) {
                MultipartEntity entity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, null, Charset.forName("UTF-8"));

                // Add the email json
                if (content != null) {
                    String data = new GsonBuilder().disableHtmlEscaping().create().toJson(content.getAsJsonObject("data"));
                    try {
                        entity.addPart("json_0", new StringBody(data, ContentType.APPLICATION_JSON.getMimeType(), ContentType.APPLICATION_JSON.getCharset()));
                    } catch (UnsupportedEncodingException e) {
                        throw GuardPGPExceptionCodes.ENCODING_NOT_SUPPORTED.create(e);
                    }
                }
                int attachcount = 0;
                if (attachments != null) {  // Add all the attachments to the body
                    for (Attachment attach : attachments) {
                        ContentType type = ContentType.create(attach.getType() + "\r\nX-Forced-MIME-Type:true");
                        ContentBody file = new ByteArrayBody(attach.getEncrContent(), type.getMimeType(), attach.getFilename());
                        entity.addPart("file_" + attachcount++, file);
                    }
                }

                Api ap = new Api(mail.getOXCookie(), mail.getSession(), mail.getUseragent());
                JsonObject result = ap.sendMail(entity);
                if (Utilities.checkFail(result)) {
                    throw GuardPGPExceptionCodes.SEND_FAILED.create(result.get("error_params").toString());
                } else {
                    // If success, set replied flags
                    handleFlags(mail);
                }
            } else {
                JsonObject data = content.get("data").getAsJsonObject();
                JsonArray fromarray = data.get("from").getAsJsonArray().get(0).getAsJsonArray();
                GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
                List<BodyPart> bpAttachments = getBodyParts(attachments);
                guardNotificationService.send(content, bpAttachments, getFromEmailAddress(recipients, fromarray.get(1).getAsString()), sender.getLang(), getMessageID());
            }
        }
    }

    protected void updateJsonRecip(PGPMail mail, List<RecipKey> inlineRecip) throws OXException {
        JsonObject data = mail.getContent().get("data").getAsJsonObject();

        JsonObject headers = data.has("headers") ? data.get("headers").getAsJsonObject() : new JsonObject();
        JsonArray toRecips = data.has("to") ? data.get("to").getAsJsonArray() : new JsonArray();

        JsonArray ccRecips = data.has("cc") ? data.get("cc").getAsJsonArray() : new JsonArray();
        try {
            if (toRecips.size() > 0) {
                headers.addProperty("X-To-Recips", getRecipsString(toRecips));
            }
            if (ccRecips.size() > 0) {
                headers.addProperty("X-CC-Recips", getRecipsString(ccRecips));
            }
        } catch (UnsupportedEncodingException e) {
            throw GuardPGPExceptionCodes.ENCODING_NOT_SUPPORTED.create(e);

        }
        JsonArray newToList = new JsonArray();
        JsonArray newCCList = new JsonArray();
        JsonArray newBCCList = new JsonArray();
        for (int i = 0; i < inlineRecip.size(); i++) {
            RecipKey recip = inlineRecip.get(i);
            if (recip.getEmail() != null) {
                JsonArray addr = new JsonArray();
                addr.add(new JsonPrimitive(recip.getName()));
                addr.add(new JsonPrimitive(recip.getEmail()));
                if (recip.getType() == RecipientType.TO) {
                    newToList.add(addr);
                }
                if (recip.getType() == RecipientType.CC) {
                    newCCList.add(addr);
                }
                if (recip.getType() == RecipientType.BCC) {
                    newBCCList.add(addr);
                }
            }
        }
        data.remove("to");
        data.remove("cc");
        data.remove("bcc");
        data.add("to", newToList);
        data.add("cc", newCCList);
        data.add("bcc", newBCCList);
    }

    /**
     * Generate comma seperated list of recipients from JsonArray
     *
     * @param recips
     * @return
     * @throws UnsupportedEncodingException
     */
    private String getRecipsString(JsonArray recips) throws UnsupportedEncodingException {
        StringBuilder toString = new StringBuilder();
        for (int i = 0; i < recips.size(); i++) {
            JsonArray recip = recips.get(i).getAsJsonArray();
            if (i > 0) {
                toString.append(", ");
            }
            InternetAddress addr = new InternetAddress(recip.get(1).getAsString(), recip.get(0).getAsString(), "UTF-8");
            toString.append(addr.toString());
        }
        return toString.toString();
    }

    private String sendPGPMime(PGPMail mail, List<RecipKey> guestRecip, GuardKeys senderkey, String host) throws OXException {
        byte[] mime = null;
        String messageId = getMessageID();
        try {
            mime = createMime(mail, messageId);
        } catch (IOException | MessagingException e) {
            LOG.error("", e);
            throw GuardPGPExceptionCodes.MAIL_CONSTRUCTION_ERROR.create(e);
        }
        if (mime == null) {
            LOG.error("Unable to encrypt PGP message");
            return null;
        }
        ByteArrayOutputStream encrypted = new ByteArrayOutputStream();

        try {
            if (mail.isPgpsign()) {  // If PGP Signing
                PGPUtils.signEncryptFile(encrypted, mime, "encrypted.asc", guestRecip, senderkey, mail.getPassword(), true, true);
            } else {
                PGPUtils.encrypt(encrypted, mime, "encrypted.asc", guestRecip, true, true);
            }
            String resp = sendMimeEmails(mail, guestRecip, encrypted, messageId, host, senderkey.getUserid(), senderkey.getContextid());
            if (Utilities.checkFail(resp) || !resp.contains("fail")) {
                handleFlags(mail);
            } else {
                throw GuardPGPExceptionCodes.SEND_FAILED.create(resp);
            }
            return (resp);
        } catch (OXException ex) {
            throw ex;
        } catch (Exception e) {
            throw GuardPGPExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Set original emails as replied to, or delete draft emails that were sent
     *
     * @param mail
     */
    private void handleFlags(PGPMail mail) {
        JsonObject data = mail.getContent().get("data").getAsJsonObject();
        if (data.has("msgref")) {
            JsonElement msg = data.get("msgref");
            boolean reply = false;
            try {
                if (data.has("headers")) {
                    JsonObject headers = data.get("headers").getAsJsonObject();
                    if (headers.has("In-Reply-To")) {
                        reply = true;
                    }
                }
            } catch (Exception e) {
                LOG.error("Error checking in-reply-to headers in email ", e);
            }
            Api ap = new Api(mail.getOXCookie(), mail.getSession(), mail.getUseragent());
            if (msg.isJsonPrimitive()) {
                ap.handleRef(data.get("msgref").getAsString(), reply, mail.getUserid(), mail.getCid());
            } else {
                if (msg.isJsonObject()) {
                    JsonObject msgref = msg.getAsJsonObject();
                    ap.handleRef(msgref.get("folder_id").getAsString() + "/" + msgref.get("id").getAsString(), reply, mail.getUserid(), mail.getCid());
                }
            }
        }
    }



    /**
     * Send the MIME email to OX users or Guest
     *
     * @param mail
     * @param guestRecip
     *
     * @param encrypted
     * @param mesageId
     * @return
     * @throws MessagingException
     */
    private String sendMimeEmails(PGPMail mail, List<RecipKey> guestRecip, ByteArrayOutputStream encrypted, String messageId, String host, int senderId, int senderCid) throws Exception {
        JsonObject data = mail.getContent().getAsJsonObject("data");
        List<String> failures = new ArrayList<String>();
        Api ap = new Api(mail.getOXCookie(), mail.getSession(), mail.getUseragent());
        JsonArray fromarray = data.get("from").getAsJsonArray().get(0).getAsJsonArray();

        String fromName = fromarray.get(0) != JsonNull.INSTANCE ? fromarray.get(0).getAsString().trim() : null;
        String fromEmail = fromarray.get(1).getAsString().trim();
        fromEmail = IDNUtil.aceEmail(fromEmail);
        InternetAddress fromaddr = fromName == null ? new InternetAddress(fromEmail) : new InternetAddress(fromEmail, fromName, "UTF-8");

        String subject = data.has("subject") ? data.get("subject").getAsString() : "";
        JsonObject headers = new JsonObject();
        try {
            if (data.has("headers")) {
                headers = data.get("headers").getAsJsonObject();
            }
        } catch (Exception ex) {
            LOG.error("Problem extracting headers");
        }

        GuardMimeMessage newMessage = new GuardMimeMessage(Session.getInstance(new Properties()), messageId);
        newMessage.setContent(new PGPMailCreator().getPGPMime(encrypted.toByteArray(), getLanguages(guestRecip)));

        String uid = CipherUtil.getUUID();
        String id = "pgp-" + uid;
        for (int i = 0; i < guestRecip.size(); i++) {
            if (guestRecip.get(i).isGuest() == true && !mail.isDraft()) {
                saveEmailData(mail, uid, mail.getRecipients(), guestRecip.get(i).getUserid(), guestRecip.get(i).getCid(), fromaddr);
            }
        }

        addRecips(newMessage, guestRecip, true, mail.isDraft());

        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        if (!mail.isDraft()) {  // If draft, we don't actually send
            for (int i = 0; i < guestRecip.size(); i++) {
                if (guestRecip.get(i).getEmail() != null) {  // Make sure legit user to send to
                    RecipKey recip = guestRecip.get(i);
                    if (!isdirect(recip)) {
                        // Guest account
                        Storage storage = Services.getService(Storage.class);
                        storage.saveEncrObj(recip.getUserid(), recip.getCid(), id, encrypted.toByteArray());
                        JsonObject email = mailCreatorService.createBlankGuestMail(recip.getLang(), mail.getTemplateId(), messageId, host, senderId, senderCid);
                        if (email == null) {
                            return ("Unable to create template");
                        }
                        email = mailCreatorService.addTo(email, recip.getName(), recip.getEmail(), recip.getType());
                        email = mailCreatorService.addFrom(email, fromName, fromEmail);
                        email = mailCreatorService.addSubject(email, subject);
                        email = mailCreatorService.addPlainText(email, mail.getGuestmessage());
                        email = mailCreatorService.noSave(email);
                        try {
                            if (!data.has("headers")) {
                                data.add("headers", new JsonObject());
                            }
                            data.get("headers").getAsJsonObject().addProperty("X-OXGUARD-GUEST", "true");
                            if (data.has("disp_notification_to")) {  // Add return receipt request if needed
                                if (data.get("disp_notification_to").getAsBoolean()) {
                                	email.get("data").getAsJsonObject().addProperty("disp_notification_to", true);
                                }
                            }
                        } catch (Exception ex) {
                            LOG.error("Problem adding guest header", ex);
                        }
                        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
                        try {
                            String url = guardConfigService.getProperty(GuardProperty.externalReaderURL, mail.getUserid(), mail.getCid());
                            if (url == null || url.isEmpty()) {
                                url = "https://" + guardConfigService.getProperty(GuardProperty.externalReaderPath);
                            }
                            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                            email = mailCreatorService.addURL(email, url + "?email=" + id + "&user=" + URLEncoder.encode(recip.getEmail(), "UTF-8") + (translationService.customTemplateExists(mail.getTemplateId(), "guesttempl.html") ? ("&templid=" + mail.getTemplateId()) : "") + "&lang=" + recip.getLang());
                        } catch (UnsupportedEncodingException e) {
                            email = mailCreatorService.addURL(email, "https://" + guardConfigService.getProperty(GuardProperty.externalReaderPath) + "?email=" + id);
                        }
                        List<Attachment> attachments = new ArrayList<Attachment>();
                        Attachment guestattach = new Attachment();
                        guestattach.setEncrContent(encrypted.toByteArray());
                        guestattach.setFilename("encrypted.asc");
                        guestattach.setType("application/octet-stream");
                        attachments.add(guestattach);
                        if (mail.getCid() > 0) {
                            MultipartEntity entity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, null, Charset.forName("UTF-8"));
                            // Add the email json
                            if (data != null) {
                                entity.addPart("json_0", new StringBody(new Gson().toJson(email.getAsJsonObject("data")), ContentType.APPLICATION_JSON.getMimeType(), ContentType.APPLICATION_JSON.getCharset()));
                            }
                            int attachcount = 0;
                            if (attachments != null) {  // Add all the attachments to the body
                                for (Attachment attach : attachments) {
                                    ContentType type = ContentType.create(attach.getType() + "\r\nX-Forced-MIME-Type:true");
                                    ContentBody file = new ByteArrayBody(attach.getEncrContent(), type.getMimeType(), attach.getFilename());
                                    entity.addPart("file_" + attachcount++, file);
                                }
                            }
                            String result = ap.sendMail(entity).toString();
                            if (Utilities.checkFail(result)) {
                                return (result);
                                //   failures.add(recip.email);
                            }
                        } else {
                            GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
                            List<BodyPart> bpAttachments = getBodyParts(attachments);
                            try {
                                guardNotificationService.send(email, bpAttachments, getFromEmailAddress(guestRecip, fromEmail), recip.getLang(), messageId);
                            } catch (OXException e) {
                                return "fail";
                            }
                        }
                    }
                }
            }
        }
        if (data.has("priority")) {
            try {
                int priority = data.get("priority").getAsInt();
                newMessage.addHeader("X-Priority", Integer.toString(priority));
                if (priority < 3) {
                    newMessage.addHeader("Importance", "High");
                }
                if (priority > 3) {
                    newMessage.addHeader("Importance", "Low");
                }
            } catch (Exception e) {
                LOG.error("problem adding priority to email", e);
            }
        }

        if (data.has("disp_notification_to")) {
            if (data.get("disp_notification_to").getAsBoolean()) {
                newMessage.addHeader("Disposition-Notification-To", fromaddr.toString());
            }
        }

        newMessage.setFrom(fromaddr);
        newMessage.setSentDate(new Date());
        newMessage.setSubject(subject, "UTF-8");
        newMessage.saveChanges();
        addReferences(newMessage, headers);
        boolean draft = mail.isDraft();
        int cid = mail.getCid();
        String resp = null;
        if (newMessage.getAllRecipients() != null || draft) {  // Standard delivery.  Check we have recipients.

            if (draft) {
                addDraftHeaders(data, newMessage);
            }
            if (cid > 0) { // If not guest sender, we use the OX backend
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                newMessage.writeTo(out);
                if (draft) {
                    LOG.debug("Saving draft to " + mail.getDraftFolder());
                }
                resp = ap.sendMimeMail(out.toString(), draft ? mail.getDraftFolder() : null);
                if (Utilities.checkFail(resp)) {
                    return ("fail " + JsonUtil.getError(resp));
                }
                out.close();
                if (draft) {
                    try {
                        if (data.has("headers")) {
                            if (data.get("headers").getAsJsonObject().has("X-OxGuard-Draft")) {
                                String fullid = data.get("headers").getAsJsonObject().get("X-OxGuard-Draft").getAsString();
                                int i = fullid.lastIndexOf("/");
                                String draftid = fullid.substring(i + 1);
                                String folder = fullid.substring(0, i);
                                ap.deleteDraft(draftid, folder);
                            }
                        }
                    } catch (Exception ex) {
                        LOG.error("Problem deleting old draft email", ex);
                    }
                    return resp;
                }
            } else { // Otherwise, use the SMTP service
                GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
                try {
                    guardNotificationService.sendMessage(newMessage, getFromEmailAddress(guestRecip, fromaddr.getAddress()), guestRecip.get(0).getLang());
                } catch (OXException e) {
                    return "fail";
                }
                return "OK";
            }
        }
        if (cid > 0) {
            createSavedItem(newMessage, guestRecip, mail.getPin(), resp, ap);
        }
        if (failures.size() > 0) {
            return "fail some failed to send";
        }
        return "OK";
    }

    private List<BodyPart> getBodyParts(List<Attachment> attachments) {
        List<BodyPart> bpAttachments = new ArrayList<>();
        for (Attachment attachment : attachments) {
            try {
                final BodyPart att = new MimeBodyPart();
                att.setText(new String(attachment.getEncrContent()));
                att.setHeader("Content-Type", Strings.removeCarriageReturn(attachment.getType()));
                att.setFileName(Strings.removeCarriageReturn(attachment.getFilename()));

                //FIXME in previous versions the header wasn't added to attachment. BUG?! what should we do with the InternetHeaders instance?
                final InternetHeaders header = new InternetHeaders();
                header.addHeader("Content-Type", Strings.removeCarriageReturn(attachment.getType()));
                header.addHeader("Content-Transfer-Encoding", "base64");
                bpAttachments.add(att);
            } catch (MessagingException e) {
                LOG.warn("Unable to add attachment information. Will skip the attachment with content id {}, filename {} and type {}.", attachment.getContentId(), attachment.getFilename(), attachment.getType(), e);
            }
        }
        return bpAttachments;
    }

    private String getFromEmailAddress(List<RecipKey> recipients, String fromaddres) throws OXException {
        int cid = 0;
        int id = 0;
        for (int i = 0; i < recipients.size(); i++) {// Find at least one member with positive cid, and use that configured address
            if (recipients.get(i).getCid() > 0) {
                cid = recipients.get(i).getCid();
                id = recipients.get(i).getUserid();
                break;
            }
        }
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        return mailCreatorService.getFromAddress(fromaddres, fromaddres, id, cid).get(1);
    }

    /**
     * Need to create combined sent item with all of the recipients, guests, etc
     *
     * @param newmessage
     * @param recipients
     * @param resp
     * @param ap
     * @throws MessagingException
     * @throws IOException
     */
    private void createSavedItem(Message newMessage, List<RecipKey> recipients, String pin, String resp, Api ap) throws MessagingException, IOException {
        if (resp != null) {  // If we have resp from previously saved email, use to delete
            try {
                JsonObject saved = JsonUtil.parseAsJsonObject(resp);
                if (saved.has("data")) {
                    String folder = saved.get("data").getAsJsonObject().get("folder_id").getAsString();
                    String id = saved.get("data").getAsJsonObject().get("id").getAsString();
                    ap.deleteMail(id, folder);  // First, remove previously saved
                }
            } catch (Exception e) {
                LOG.error("Problem getting info to delete saved email", e);
            }
        }
        MimeMessage msg = (MimeMessage) newMessage;
        Address[] x = null;
        msg.setRecipients(Message.RecipientType.TO, x);
        msg.setRecipients(Message.RecipientType.CC, x);
        msg.setRecipients(Message.RecipientType.BCC, x);
        msg.addHeader("X-OxGuard-PIN", pin); // Save pin to sent folder email for retrieval
        addRecips(msg, recipients, true, true);
        msg.saveChanges();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        msg.writeTo(out);
        ap.sendMimeMail(out.toString(), ap.getSentFolder());
    }

    private void addDraftHeaders(JsonObject data, Message newMessage) throws MessagingException {
        if (!data.has("headers")) {
            return;
        }
        JsonObject headers = data.get("headers").getAsJsonObject();
        for (Map.Entry<String, JsonElement> entry : headers.entrySet()) {
            try {
                if (entry.getValue() != null) {
                    if (entry.getValue().isJsonPrimitive()) {
                        if (addHeader(entry.getKey())) {
                            newMessage.addHeader(entry.getKey(), entry.getValue().getAsString());
                        }
                    }
                }
            } catch (Exception e) {
                LOG.error("Non fatal Problem saving draft header: ", e);
            }
        }
    }

    // When saving draft headers, we don't want to save the original DKIM, etc
    private boolean addHeader(String header) {
        try {
            switch (header) {
                case "Message-ID":
                    return (false);
                case "DKIM-Signature":
                    return (false);
                case "Received":
                    return (false);
                case "X-FromIP":
                    return (false);
            }
            return true;
        } catch (Exception e) {
            return true;
        }
    }

    /**
     * Add reply-to and refernces headers to outgoing email if present
     *
     * @param newmessage
     * @param headers
     * @throws MessagingException
     */
    private void addReferences(Message newMessage, JsonObject headers) throws MessagingException {
        if (headers == null) {
            return;
        }
        try {
            if (headers.has("In-Reply-To")) {
                String replyto = "";
                if (headers.get("In-Reply-To").isJsonArray()) { // Occasionally, this is sent as an array
                    replyto = headers.get("In-Reply-To").getAsJsonArray().get(0).getAsString();
                } else {
                    replyto = headers.get("In-Reply-To").getAsString();
                }
                newMessage.addHeader("In-Reply-To", replyto);
            }
            if (headers.has("References")) {
                String ref = "";
                if (headers.get("References").isJsonArray()) {
                    ref = headers.get("References").getAsJsonArray().get(0).getAsString();
                } else {
                    ref = headers.get("References").getAsString();
                }
                newMessage.addHeader("References", ref);
            }
        } catch (Exception e) {
            LOG.error("Error adding headers to email, ", e);
        }
    }

    /**
     * Get an {@link List} of languages from all recipients
     *
     * @param recipients
     * @return
     */
    private List<String> getLanguages(List<RecipKey> recipients) {
        List<String> languages = new ArrayList<String>();
        for (RecipKey recip : recipients) {
            if (!languages.contains(recip.getLang())) {
                languages.add(recip.getLang());
            }
        }
        return languages;
    }

    /**
     * Evaluate if Guest recipient wants direct email.
     *
     * @param recip
     * @return
     */
    private static boolean isdirect(RecipKey recip) {
        try {
            if (recip.getCid() > 0) {
                return (true);  // Members get direct
            }
            if (recip.isGuest() == false) {
                return (true);  // Have uploaded key, not guest
            }
            if (recip.getSettings() == null) {
                return (false);
            }
            if (!recip.getSettings().has("direct")) {
                return (false);
            }
            return recip.getSettings().get("direct").getAsBoolean();
        } catch (Exception e) {
            LOG.error("Trouble checking is direct", e);
            return false;
        }
    }

    /**
     * Add all of the recipients to the Mail Message
     *
     * @param newMessage
     * @param recipients
     * @param addBcc Don't add BCC if this is for display later
     * @param all Don't check if guest or not -- if just draft, mime header
     * @throws UnsupportedEncodingException
     * @throws MessagingException
     */
    private void addRecips(Message newMessage, List<RecipKey> recipients, boolean addBcc, boolean all) throws UnsupportedEncodingException, MessagingException {
        List<Address> to = new ArrayList<Address>();
        List<Address> cc = new ArrayList<Address>();
        List<Address> bcc = new ArrayList<Address>();
        for (int i = 0; i < recipients.size(); i++) {
            RecipKey recip = recipients.get(i);
            if (recip.getEmail() != null) {
                InternetAddress addr = new InternetAddress(IDNUtil.aceEmail(recip.getEmail()), recip.getName().replace("\"", ""), "UTF-8");
                // If not draft email, only saving the addresses that are directly mailed (IE not guest emails)
                if (all || isdirect(recip)) {
                    if (recip.getType() == RecipientType.TO) {
                        to.add(addr);
                    }
                    if (recip.getType() == RecipientType.CC) {
                        cc.add(addr);
                    }
                    if (recip.getType() == RecipientType.BCC) {
                        bcc.add(addr);
                    }
                }
            }
        }
        if (to.size() > 0) {
            newMessage.addRecipients(RecipientType.TO, to.toArray(new Address[to.size()]));
        }
        if (cc.size() > 0) {
            newMessage.addRecipients(RecipientType.CC, cc.toArray(new Address[cc.size()]));
        }
        if (addBcc && bcc.size() > 0) {
            newMessage.addRecipients(RecipientType.BCC, bcc.toArray(new Address[bcc.size()]));
        }
    }

    private void sendInline(PGPMail mail, List<RecipKey> inlineRecip, GuardKeys keys) throws OXException {
        JsonArray mailattach = mail.getContent().getAsJsonObject("data").get("attachments").getAsJsonArray();
        for (int i = 0; i < mailattach.size(); i++) {
            JsonObject attach = mailattach.get(i).getAsJsonObject();
            if (attach.has("content")) {
                String content = attach.get("content").isJsonNull() ? "" : attach.get("content").getAsString();
                if (content.equals("")) {
                    content = "  ";  // if blank content, just add some white spaces to establish PGP Email
                }
                if (content.length() > 0) {
                    try {
                        ByteArrayOutputStream encrypted = new ByteArrayOutputStream();
                        if (mail.isPgpsign()) {
                            PGPUtils.signEncryptFile(encrypted, content.getBytes("UTF-8"), "encrypted.asc", inlineRecip, keys, mail.getPassword(), true, true);
                        } else {
                            PGPUtils.encrypt(encrypted, content.getBytes("UTF-8"), "encrypted.asc", inlineRecip, true, true);
                        }
                        attach.remove("content");
                        String encr = encrypted.toString();
                        encrypted.close();
                        String type = attach.get("content_type").getAsString();
                        if (type.contains("altern") || type.contains("html")) {
                            encr = encr.replace("\n", "<br>");
                        }
                        attach.addProperty("content", encr);
                    } catch (Exception e) {
                        throw GuardPGPExceptionCodes.IO_ERROR.create(e, e.getMessage());
                    }
                }
            }
        }
        encryptAttachments(mail, keys);
    }

    private void encryptAttachments(PGPMail mail, GuardKeys keys) throws OXException {
        List<Attachment> attachments = mail.getAttachments();
        for (int j = 0; j < attachments.size(); j++) {
            Attachment attach = attachments.get(j);
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            try {
                List<RecipKey> recipients = mail.getRecipients();
                if (mail.isPgpsign()) {
                    PGPUtils.signEncryptFile(out, attach.getContent(), attach.getFilename(), recipients, keys, mail.getPassword(), true, true);
                } else {
                    PGPUtils.encrypt(out, attach.getContent(), attach.getFilename(), recipients, true, true);
                }
                out.close();
            } catch (Exception e) {
                throw GuardPGPExceptionCodes.IO_ERROR.create(e, e.getMessage());
            }
            attach.setEncrContent(out.toByteArray());
            attach.setType("application/octet-stream");
            String filename = attach.getFilename();
            attach.setFilename(filename + ".pgp");
            attach.setPgp(true);
        }
    }

    private void saveEmailData(PGPMail mail, String emailID, List<RecipKey> recipients, int recipientUserid, int recipientCid, InternetAddress from) throws OXException {
        StringBuilder addxml = new StringBuilder();
        addxml.append("<share>");
        for (RecipKey user : recipients) {
            if (user.getEmail() != null) {
                addxml.append("<u><e>" + StringEscapeUtils.escapeXml11(user.getEmail()) + "</e><i>" + user.getUserid() + "</i><c>" + user.getCid() + "</c></u>");
            }
        }
        // add sender
        addxml.append("<u><e>" + StringEscapeUtils.escapeXml11(from.getAddress()) + "</e><r>N</r><i>" + mail.getUserid() + "</i><c>" + mail.getCid() + "</c></u>");
        addxml.append("</share>");

        EncryptedItemsStorage itemsStorage = Services.getService(EncryptedItemsStorage.class);
        itemsStorage.insertForRecipientTruncated(emailID, recipientUserid, recipientCid, mail.getUserid(), mail.getCid(), 0, 0, addxml.toString(), "");
    }

    /**
     * Add the from, subject, sent time to PGP Mime body
     *
     * @param data
     * @param msg
     */
    private void addHeaders(JsonObject data, MimeMessage msg) {
        try {
            JsonArray fromarray = data.get("from").getAsJsonArray().get(0).getAsJsonArray();
            String senderName = fromarray.get(0) != JsonNull.INSTANCE ? fromarray.get(0).getAsString() : null;
            String senderEmail = IDNUtil.aceEmail(fromarray.get(1).getAsString());
            InternetAddress from = senderName == null ? new InternetAddress(senderEmail) : new InternetAddress(senderEmail, senderName, "UTF-8");
            String subject = data.has("subject") ? data.get("subject").getAsString() : "";
            msg.setFrom(from);
            msg.setSubject(subject, "UTF-8");
            msg.setSentDate(new Date());
        } catch (Exception ex) {
            LOG.error("Error adding headers to PGP Mime encrypted content ", ex);
        }

    }

    private byte[] createMime(PGPMail mail, String messageId) throws IOException, MessagingException {
        JsonObject data = mail.getContent().getAsJsonObject("data");

        MimeMultipart mp = new MimeMultipart("mixed");
        JsonArray mailattach = data.get("attachments").getAsJsonArray();
        for (int i = 0; i < mailattach.size(); i++) {
            JsonObject attach = mailattach.get(i).getAsJsonObject();
            if (attach.has("content_type") || attach.has("type")) {
                String type = attach.has("content_type") ? attach.get("content_type").getAsString() : attach.get("type").getAsString();
                switch (type.toLowerCase()) {
                    case "alternative":
                        MimeBodyPart multipart = new MimeBodyPart();
                        String htmlcontent = attach.get("content").getAsString();
                        Multipart alternate = new MimeMultipart("alternative");
                        BodyPart pt = new MimeBodyPart();
                        StringReader sr = new StringReader(htmlcontent);
                        Source src = new Source(sr);
                        pt.setContent(src.getRenderer().toString(), "text/plain; charset=utf-8");
                        alternate.addBodyPart(pt);
                        sr.close();
                        BodyPart htm = new MimeBodyPart();
                        htm.setContent(htmlcontent, "text/html; charset=utf-8");
                        alternate.addBodyPart(htm);
                        multipart.setContent(alternate);
                        mp.addBodyPart(multipart);
                        break;

                    case "text/plain":
                        if (attach.has("content")) {
                            BodyPart ptext = new MimeBodyPart();
                            ptext.setContent(attach.get("content").getAsString(), "text/plain; charset=utf-8");
                            mp.addBodyPart(ptext);
                        }
                        break;
                    case "text/html":
                        if (attach.has("content")) {
                            BodyPart html = new MimeBodyPart();
                            html.setContent(attach.get("content").getAsString(), "text/html; charset=utf-8");
                            mp.addBodyPart(html);
                        }
                        break;
                }
            }
        }
        Properties prop = new Properties();
        List<Attachment> attachments = mail.getAttachments();
        for (int i = 0; i < attachments.size(); i++) {
            Attachment attach = attachments.get(i);
            BodyPart pt = new MimeBodyPart();
            if (attach.getFilename() != null) {
                pt.setFileName(attach.getFilename());
            }
            if (attach.getType().contains("text")) {
                pt.setContent(new String(attach.getContent(), "UTF-8"), attach.getType());
            }
            else if (attach.getType().equals("message/rfc822")) {
                InputStream content = new ByteArrayInputStream(attach.getContent());
                MimeMessage m = new MimeMessage(Session.getInstance(prop), content);
                pt.setContent(m, attach.getType());
            }
            else {
                pt.setContent(attach.getContent(), attach.getType());
                if (attach.getContentId() != null) {
                    pt.setHeader("Content-ID", "<" + attach.getContentId() + ">");
                    pt.setDisposition("inline");
                }
            }
            mp.addBodyPart(pt);
        }

        GuardMimeMessage message = new GuardMimeMessage(Session.getInstance(prop), messageId);
        addRecips(message, mail.getRecipients(), false, true);  // add the recipients, but not bcc
        addHeaders(data, message);
        message.setContent(mp);
        message.saveChanges();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        message.writeTo(out);
        byte[] mime = out.toByteArray();
        out.close();
        return mime;
    }

    /**
     * Get the message identifier
     *
     * @return the message identifier
     * @throws OXException if the GuardConfigurationService is absent
     */
    protected String getMessageID() throws OXException {
        final GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
        String mailIdDomain = guardConfigurationService.getProperty(GuardProperty.mailIdDomain);
        final String oxURL = guardConfigurationService.getProperty(GuardProperty.externalEmailURL);
        if (mailIdDomain.equals("")) {  // If maildomain not defined, try to extract domain from URL
            if (oxURL.contains("/")) {
                mailIdDomain = oxURL.substring(0, oxURL.indexOf("/"));
            }
            if (mailIdDomain.contains(":")) {
                mailIdDomain = mailIdDomain.substring(0, mailIdDomain.indexOf(":"));
            }
        }
        if (mailIdDomain.equals("")) {
            mailIdDomain = "GUARD";
        }  // Otherwise, just define it as Guard
        return ("<" + CipherUtil.getUUID() + "@" + mailIdDomain + ">");
    }
}
