/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.util.Iterator;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyDecryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.pgpcore.services.TokenAuthenticationService;
import com.openexchange.guard.pgpcore.services.UserIdentity;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;
import com.openexchange.guard.session.GuardSessionService;

/**
 * {@link TokenAuthenticationServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class TokenAuthenticationServiceImpl implements TokenAuthenticationService {

    private final GuardKeyService guardKeyService;
    private final GuardSessionService guardSessionService;
    private final GuardCipherService guardCipherService;
    private static final Logger logger = LoggerFactory.getLogger(TokenAuthenticationServiceImpl.class);

    /**
     * Initializes a new {@link TokenAuthenticationServiceImpl}.
     * 
     * @param guardKeyService
     */
    public TokenAuthenticationServiceImpl(GuardKeyService guardKeyService, GuardSessionService guardSessionService, GuardCipherService guardCipherService) {
        this.guardKeyService = guardKeyService;
        this.guardSessionService = guardSessionService;
        this.guardCipherService = guardCipherService;
    }

    /**
     * Decodes a private key from a PGPSecretKey package
     * 
     * @param secretKey The package to decode the private key from
     * @param password The password required for extracting the key
     * @return The extracted private key
     * @throws Exception
     */
    private PGPPrivateKey decodePrivateKey(PGPSecretKey secretKey, char[] password) throws Exception {
        PBESecretKeyDecryptor extractor = new BcPBESecretKeyDecryptorBuilder(new BcPGPDigestCalculatorProvider()).build(password);
        return secretKey.extractPrivateKey(extractor);
    }

    /**
     * Verifies the password for a given GuardKeys object
     * 
     * @param userKeys The key object
     * @param password The password to verify
     * @return True if the password is correct for the given key object, false otherwise
     */
    private boolean verifyPassword(GuardKeys userKeys, char[] password) {
        //This will verify the password against the "current" key
        //TODO: verify against other keys as well
        Iterator<PGPSecretKey> secretKeyIterator = userKeys.getPGPSecretKeyRing().getSecretKeys();
        while (secretKeyIterator.hasNext()) {
            PGPSecretKey secretKey = secretKeyIterator.next();
            try {
                char[] hashedPassword = CipherUtil.getSHA(new String(password), userKeys.getSalt()).toCharArray();
                PGPPrivateKey decodePrivateKey = decodePrivateKey(secretKey, hashedPassword);
                if (decodePrivateKey != null) {
                    return true;
                }
            } catch (Exception e) {
                logger.error("Error while decoding private key for user {}: {}", userKeys.getUserid(), e.getMessage());
            }
        }
        return false;
    }

    /**
     * Creates a new JSONObject which can be used to create the auth-token
     * 
     * @param userIdentity The user's identity
     * @param password The user's password
     * @return A JSON object containing userIdentity and password
     * @throws JSONException
     */
    private JSONObject createPlainToken(String userIdentity, char[] password) throws JSONException {
        JSONObject plainToken = new JSONObject();
        plainToken.put("UserIdentity", userIdentity);
        plainToken.put("Password", new String(password));
        return plainToken;
    }

    /**
     * Creates a new UserIdenenty from the given JSON
     * 
     * @param json The json to create the UserIdentity from
     * @return The UserIdentity created from the given json, or null if the json does not provide all required information
     * @throws JSONException
     */
    private UserIdentity createUserIdentity(JSONObject json) throws JSONException {
        if (json.has("UserIdentity") && json.has("Password")) {
            return new UserIdentity(json.getString("UserIdentity"), json.getString("Password"));
        }
        else if(json.has("email") && json.has("ox_password")) {
            return new UserIdentity(json.getString("email"), json.getString("ox_password"));
        }
        return null;
    }

    /**
     * Encrypts the JSON object with the given key
     * 
     * @param tokenKey The key to use for encryption
     * @param plainAuthenticationToken The JSON object to encrypt
     * @return The encrypted data
     */
    private String encryptAuthenticationToken(String tokenKey, JSONObject plainAuthenticationToken) {
        return guardCipherService.encrypt(plainAuthenticationToken.toString(), tokenKey);
    }

    /**
     * Decrypts the given data into a JSON object
     * 
     * @param tokenKey The key to use for decryption
     * @param authToken The decrypted data
     * @return The decrypted JSON object, or null if decryption failed
     */
    private JSONObject decryptAuthenticationToken(String tokenKey, String authToken) {
        String jsonData = guardCipherService.decrypt(authToken, tokenKey);
        if (jsonData != null) {
            try {
                return new JSONObject(jsonData);
            } catch (JSONException e) {
                logger.error("Error while parsing JSON object: " + e.getMessage());
            }
        }
        logger.error("Unable to decrypt authentication token");
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgpcore.services.TokenAuthenticationService#authenticate(java.lang.String, java.lang.String)
     */
    @Override
    public String createAuthenticationToken(String sessionIdentifier, String userIdentity, char[] password) throws OXException {

        //Get user's keyring
        GuardKeys userKeys = guardKeyService.getKeysFromEmail(userIdentity);
        if (userKeys != null) {
            //Verify password 
            boolean accessGranted = verifyPassword(userKeys, password);
            if (accessGranted) {

                //Check for existing token keys for the given identifier
                String tokenKey = guardSessionService.getToken(sessionIdentifier);
                if (tokenKey == null) {
                    //Create a new tokenKey, if we do not already have one
                    tokenKey = guardSessionService.newToken(sessionIdentifier, userKeys.getUserid(), userKeys.getContextid());
                }

                try {
                    //Create and encrypt the authentication token
                    return encryptAuthenticationToken(tokenKey, createPlainToken(userIdentity, password));
                } catch (JSONException e) {
                    throw PGPCoreServicesExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
                }
            }
            else {
                logger.error("Unable to verify password for user {}", userIdentity);
            }
        }
        else {
            logger.error("Unable to create authentication token: No PGP key found for user {}", userIdentity);
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgpcore.services.TokenAuthenticationService#deleteAuthenticationToken(java.lang.String)
     */
    @Override
    public void deleteAuthenticationToken(String sessionIdentifier) throws OXException {
        guardSessionService.deleteToken(sessionIdentifier);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgpcore.services.TokenAuthenticationService#authenticate(java.lang.String, java.lang.String)
     */
    @Override
    public UserIdentity decryptUserIdentity(String sessionIdentifier, String authToken) throws OXException {
        String tokenKey = guardSessionService.getToken(sessionIdentifier);
        if (tokenKey != null) {
            try {
                JSONObject jsonUserIdentity = decryptAuthenticationToken(tokenKey, authToken);
                if (jsonUserIdentity != null) {
                    return createUserIdentity(jsonUserIdentity);
                }
            } catch (JSONException e) {
                logger.error("Error while parsing user identity JSON object: " + e.getMessage());
            }
        }
        else {
            logger.error("Unknown session identifier while decrypting user identity");
        }
        return null;
    }
}
