/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.osgi;

import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.GuardMasterKeyService;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.PublicKeyService;
import com.openexchange.guard.keys.internal.PublicKeyServiceImpl;
import com.openexchange.guard.keys.internal.hkpclient.CachingHKPClientService;
import com.openexchange.guard.keys.internal.hkpclient.RemoteHKPClientService;
import com.openexchange.guard.keys.internal.hkpclient.SRVHKPClientService;
import com.openexchange.guard.keys.internal.key.GuardKeyServiceImpl;
import com.openexchange.guard.keys.internal.master.internal.GuardMasterKeyServiceImpl;
import com.openexchange.guard.keys.internal.pgp.PGPKeyServiceImpl;
import com.openexchange.guard.keys.storage.DeletedKeysStorage;
import com.openexchange.guard.keys.storage.KeyCacheStorage;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.OGPGPKeysStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.keys.storage.RemoteKeyCacheStorage;
import com.openexchange.guard.keys.storage.deletedKey.DeletedKeysStorageImpl;
import com.openexchange.guard.keys.storage.keyCache.KeyCacheStorageImpl;
import com.openexchange.guard.keys.storage.keyCache.PopulateRSA;
import com.openexchange.guard.keys.storage.ogKeyTable.KeyTableStorageImpl;
import com.openexchange.guard.keys.storage.ogPGPKeys.OGPGPKeysStorageImpl;
import com.openexchange.guard.keys.storage.pgpKeys.PGPKeysStorageImpl;
import com.openexchange.guard.keys.storage.remoteKeyCache.RemoteKeyCacheStorageImpl;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.oxapi.pooling.HttpConnectionPoolService;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.user.GuardUserService;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.timer.TimerService;

/**
 * {@link GuardKeyServicesActivator}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardKeyServicesActivator extends HousekeepingActivator {

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { GuardConfigurationService.class, GuardDatabaseService.class, GuardShardingService.class, GuardCipherFactoryService.class,
            GuardUserService.class, MailCreatorService.class, TimerService.class, GuardNotificationService.class, EmailStorage.class, GuardRatifierService.class,
            DNSService.class, GuardTranslationService.class, HttpConnectionPoolService.class };
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        Logger logger = LoggerFactory.getLogger(GuardKeyServicesActivator.class);
        logger.info("Starting bundle {}", context.getBundle().getSymbolicName());

        Services.setServiceLookup(this);

        GuardConfigurationService guardConfigService = getService(GuardConfigurationService.class);

        // Register storages
        registerService(KeyCacheStorage.class, new KeyCacheStorageImpl());
        registerService(KeyTableStorage.class, new KeyTableStorageImpl());
        registerService(OGPGPKeysStorage.class, new OGPGPKeysStorageImpl());
        registerService(PGPKeysStorage.class, new PGPKeysStorageImpl());
        registerService(DeletedKeysStorage.class, new DeletedKeysStorageImpl());
        registerService(RemoteKeyCacheStorage.class, new RemoteKeyCacheStorageImpl());

        // Track storages
        trackService(KeyCacheStorage.class);
        trackService(KeyTableStorage.class);
        trackService(OGPGPKeysStorage.class);
        trackService(PGPKeysStorage.class);
        trackService(DeletedKeysStorage.class);
        trackService(RemoteKeyCacheStorage.class);

        // Register key services
        registerService(GuardKeyService.class, new GuardKeyServiceImpl());
        trackService(GuardKeyService.class);

        registerService(PGPKeyService.class, new PGPKeyServiceImpl());
        trackService(PGPKeyService.class);

        //Register HKP client service
        //Setting up HKP client query logic in a decorator style
        registerService(HKPClientService.class, new CachingHKPClientService(new SRVHKPClientService(new RemoteHKPClientService())));
        trackService(HKPClientService.class);

        openTrackers();

        registerService(GuardMasterKeyService.class, new GuardMasterKeyServiceImpl());
        trackService(GuardMasterKeyService.class);
        // Open trackers to track the GuardMasterKeyService
        openTrackers();

        registerService(PublicKeyService.class, new PublicKeyServiceImpl());

        //RSA caching
        if (guardConfigService.getBooleanProperty(GuardProperty.rsaCache)) {
            TimerService timerService = Services.getService(TimerService.class);
            timerService.scheduleAtFixedRate(new PopulateRSA(), 0, guardConfigService.getIntProperty(GuardProperty.keyCacheCheckInterval), TimeUnit.SECONDS);
        }

        logger.info("Key services and storages registered.");
    }

    @Override
    protected void stopBundle() throws Exception {
        final Logger logger = LoggerFactory.getLogger(GuardKeyServicesActivator.class);
        logger.info("Stopping bundle {}", context.getBundle().getSymbolicName());

        unregisterService(KeyCacheStorage.class);
        unregisterService(KeyTableStorage.class);
        unregisterService(OGPGPKeysStorage.class);
        unregisterService(PGPKeysStorage.class);
        unregisterService(DeletedKeysStorage.class);
        unregisterService(RemoteKeyCacheStorage.class);
        unregisterService(GuardKeyService.class);
        unregisterService(PGPKeyService.class);
        unregisterService(GuardMasterKeyService.class);
        unregisterService(PublicKeyService.class);

        logger.info("Key services and storages unregistered.");
        Services.setServiceLookup(null);
        super.stopBundle();
    }
}
