define([
    'mailvelope/main',
    'spec/shared/capabilities',
    'io.ox/core/manifests',
    'io.ox/core/capabilities',
    'io.ox/core/extensions',
    'settings!oxguard',
    'mailvelope/editor/main',
    'oxguard/mail/register_compose'
], function (api, caputil, manifests, cap, ext, settings) {

    'use strict';

    var capabilities = caputil.preset('common').init(['io.ox/core/manifests'], [manifests]);

    describe('Mailvelope integration', function () {

        beforeEach(function () {
            settings.set('cryptoProvider', 'mailvelope');
            return capabilities.enable(['mailvelope']);
        });

        //FIXME: test environment does not inject manifests
        //it works, if ox.serverConfig.manifests is set accordingly, but that's not what we want to test, here
        it.skip('should register a new editor', function () {
            manifests.reset();
            var plugins = manifests.manager.pluginsFor('io.ox/mail/compose/editor/mailvelope');

            expect(plugins).not.to.be.empty;
        });

        //TODO: this feature has been moved into oxguard/mail/compose
        it('should add the lock item to mail compose fields', function (done) {
            var pointDefined = ext.point('io.ox/mail/compose/fields').has('toggle-encryption');
            expect(pointDefined).to.be.true;
            ext.point('io.ox/mail/compose/fields').get('toggle-encryption', function (p) {
                expect(p.draw, 'draw method').to.be.a('function');
                done();
            });
        });

        describe('editorMode/encrypt event handlers', function () {
            var baton, point;

            beforeEach(function (done) {
                var FakeComposeView = Backbone.View.extend({
                    stopAutoSave: function () {
                        this.autoSave = false;
                    },
                    initAutoSaveAsDraft: function () {
                        this.autoSave = true;
                    }
                });
                baton = new ext.Baton({
                    view: new FakeComposeView({}),
                    model: new Backbone.Model({
                        editorMode: 'text',
                        attachments: new Backbone.Collection([])
                    })
                });
                baton.view.model = baton.model;
                ext.point('io.ox/mail/compose/fields').get('monitor-mailvelope', function (p) {
                    point = p;
                    done();
                });
            });
            it('should register  for mail compose view/model', function () {
                expect(point.draw, 'draw method').to.be.a('function');
                //need to bind to a jQuery element, so do not call directly
                point.draw.apply($('<div>'), [baton]);
                baton.model.set('encrypt', true);
                expect(baton.model.get('editorMode')).to.equal('mailvelope');
            });

            describe('mailvelope editor activated', function () {
                it('should hide the save button', function () {
                    var saveButton = $('<button data-action="save">');
                    var node = $('<div>').addClass('io-ox-mail-compose-window').append(
                        $('<div data-extension-id="header">').append(saveButton),
                        $('<div class="mail-compose">')
                    );
                    $('body').append(node);
                    expect(saveButton.is(':visible')).to.be.true;
                    point.draw.apply(node.find('.mail-compose'), [baton]);

                    baton.model.set('encrypt', true);
                    expect(saveButton.is(':visible')).to.be.false;
                    node.remove();
                });

                it('should hide the toolbar', function () {
                    var toolbar = $('<div data-extension-id="composetoolbar">');
                    var node = $('<div>').append(
                        toolbar
                    );

                    $('body').append(node);
                    point.draw.apply(node, [baton]);

                    expect(toolbar.is(':visible')).to.be.true;
                    baton.model.set('encrypt', true);
                    expect(toolbar.is(':visible')).to.be.false;

                    node.remove();
                });

                it('should disable autosave', function () {
                    expect(baton.view.autoSave).to.be.undefined;
                    point.draw.apply($('<div>'), [baton]);
                    baton.model.set('encrypt', true);
                    expect(baton.view.autoSave).to.be.false;
                });
            });

            describe('mailvelope editor deactivated', function () {
                it('should show the save button', function () {
                    var saveButton = $('<button data-action="save">');
                    var node = $('<div>').addClass('io-ox-mail-compose-window').append(
                        $('<div data-extension-id="header">').append(saveButton),
                                                                                       $('<div class="mail-compose">')
                    );
                    $('body').append(node);
                    baton.model.set('encrypt', true);
                    point.draw.apply(node.find('.mail-compose'), [baton]);

                    baton.model.set('encrypt', false);
                    expect(saveButton.is(':visible')).to.be.true;
                    node.remove();
                });

                it('should show the toolbar', function () {
                    var toolbar = $('<div data-extension-id="composetoolbar">');
                    var node = $('<div>').append(
                        toolbar
                    );

                    $('body').append(node);
                    baton.model.set('encrypt', true);
                    point.draw.apply(node, [baton]);

                    baton.model.set('encrypt', false);
                    expect(toolbar.is(':visible')).to.be.true;

                    node.remove();
                });

                it('should activate autosave', function () {
                    expect(baton.view.autoSave).to.be.undefined;
                    baton.model.set('encrypt', true);
                    point.draw.apply($('<div>'), [baton]);
                    baton.model.set('encrypt', false);
                    expect(baton.view.autoSave).to.be.true;
                });
            });
        });

        describe('implement the Editor interface for new mail compose', function () {
            var editor,
                editorNode = $('<div class="editable" data-editor-id="editor-1337">');

            beforeEach(function () {
                window.mailvelope = {};
                window.mailvelope.getKeyring = function (id) {
                    return $.when({
                        id: id,
                        exportOwnPublicKey: function (email) {
                            if (email === 'jan.doe@example.com') return $.when('the public key of jan.doe');
                            return $.Deferred().reject({
                                code: 'NO_KEY_FOR_ADDRESS'
                            });
                        }
                    });
                };
                window.mailvelope.createEditorContainer = function (selector) {
                    var node = $(selector);
                    node.append($('<iframe>'));
                    return $.when(window.mailvelope);
                };

                require.undef('mailvelope/main');
                require.undef('mailvelope/editor/main');
                var def = require(['mailvelope/editor/main']).then(function (Editor) {
                    $('body').append(editorNode);
                    editor = new Editor(editorNode, {
                        model: new Backbone.Model({
                            from: [['Jan Doe', 'jan.doe@example.com']]
                        })
                    });
                });
                $(window).trigger('mailvelope');
                return def;
            });
            afterEach(function () {
                editorNode.empty().remove();
                require.undef('mailvelope/main');
                require.undef('mailvelope/editor/main');
            });

            it('should provide a done method which returns a deferred object', function () {
                expect(editor.done).to.be.a('function');
            });

            it('should call createEditorContainer API method during initialization', function (done) {
                editor.done(function () {
                    //fake implementation will append an iframe
                    expect(editorNode.find('iframe').length, 'fake element added to editorNode during initialization').to.equal(1);
                    done();
                });
            });

            it('should provide a setContent method', function () {
                expect(editor.setContent).to.be.a('function');
            });

            it('should provide a setPlainText method', function () {
                expect(editor.setPlainText).to.be.a('function');
            });

            it('should provide show and hide methods', function () {
                expect(editor.show).to.be.a('function');
                editor.show();
                expect(editorNode.find('#editor-1337').is(':visible'), 'editor node is visible').to.be.true;
                editor.hide();
                expect(editorNode.find('#editor-1337').is(':visible'), 'editor node is visible').to.be.false;
            });

            it('should provide a destroy method', function () {
                expect(editor.destroy).to.be.a('function');
            });

            describe('extends io.ox/mail/compose/actions/send', function () {
                var point = ext.point('io.ox/mail/compose/actions/send');
                var extension, baton;

                beforeEach(function (done) {
                    baton = new ext.Baton({
                        model: new Backbone.Model({
                            editorMode: 'mailvelope',
                            from: [['Jan Doe', 'jan.doe@example.com']],
                            to: [['James Kirk', 'captain@enterprise']],
                            cc: [['Leonard McCoy', 'bones@enterprise']],
                            bcc: [['NSA', 'spy@starfleet']]
                        }),
                        view: {
                            getEditor: function () {
                                return $.when(editor);
                            }
                        }
                    });

                    baton.model.setContent = function (text) {
                        expect(text).to.equal('Some example text from mailvelope');
                    };
                    baton.model.getContent = function (text) {
                        return 'Some example text from mailvelope';
                    };

                    point.get('send-mailvelope', function (e) {
                        extension = e;
                        done();
                    });
                });
                afterEach(function () {
                    extension = undefined;
                    baton = undefined;
                });

                it('should add "send-mailvelope" action', function () {
                    expect(extension).to.be.defined;
                    expect(extension.perform).to.be.a('function');
                });

                //do not run in PhantomJS, since it does not provide TextEncode/TextDecode API
                if (_.device('PhantomJS')) return;
                it('should encrypt for all recipients', function () {
                    window.mailvelope.encrypt = function (recipients) {
                        expect(recipients).to.contain('captain@enterprise');
                        expect(recipients).to.contain('bones@enterprise');
                        expect(recipients).not.to.contain('spy@starfleet');
                        return $.when('Some example text from mailvelope');
                    };

                    return extension.perform(baton);
                });

                it('should fail when encryption is not possible', function () {
                    window.mailvelope.encrypt = function (recipients) {
                        expect(recipients).to.contain('captain@enterprise');
                        expect(recipients).to.contain('bones@enterprise');
                        expect(recipients).not.to.contain('spy@starfleet');
                        return $.Deferred().reject({ message: 'undefined example error' });
                    };

                    ext.point('io.ox/mail/compose/actions/send').extend({
                        id: 'errors',
                        perform: function (baton) {
                            expect(baton.error).to.equal('undefined example error');
                            return $.when();
                        }
                    });

                    return extension.perform(baton).then(_.identity, function () {
                        expect(baton.isPropagationStopped(), 'point propagation is stopped').to.be.true;
                        //yes, we are good!
                        return $.when();
                    });
                });
            });

            describe('key status extension for tokenfield', function () {
                //TODO: implement me, once this feature gets ironed out
                it('should indicate, that a key is known');

                it('should indicate, that a key is not known');
            });
        });
    });
});
