/* All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/pgp/checker', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'oxguard/core/og_http',
    'settings!oxguard',
    'gettext!oxguard'
], function (ext, links, http, settings, gt) {
    'use strict';

    var firstprompt = false;

    // Do the check if the email is PGP email.  Check for inline, MIME, or signature
    function checkPGP (baton, location) {
        if (settings.get('cryptoProvider') === 'mailvelope') return;

        if (baton.data.PGPDecoded === true) {
            require (['oxguard/pgp/reader'], function (reader) {
                reader.addIcons(baton);
            });
            if (window.oxguarddata.settings.oxguard === false) { // If not guard user, display upsell if any
                var loc = baton.view.$el;
                if (loc.find('.guardUpsell').length < 1) { // Make sure not already drawn
                    require (['oxguard/mail/upsell'], function (upseller) {
                        upseller.upsell(loc);
                    });
                }
            }
            return; // If already decoded
        }
        if (baton.view.model === null) {
            return;
        }
        var headers = baton.view.model.get('headers');
        if (headers === undefined) {// not yet loaded
            return;
        }
        // Init
        if (baton.data.headers === undefined) { // If email wasn't fully loaded into baton, do so now.
            baton.data = baton.view.model.toJSON();
        }
        var mail = baton.data;
        if (mail.results !== undefined) return;
        var attachments = baton.data.attachments;
        if (attachments === undefined) return;
        if (headers['X-OXGUARD-GUEST'] !== undefined) {
            baton.view.model.set('pgp_found', true);
            pgpAttach(baton);
            pgpFound.call(this, baton);
        }
        for (var i = 0; i < attachments.length; i++) {
            //var top = (attachments[i].size < 30) ? attachments[i].size : 30;
            if (attachments[i].content_type.toLowerCase().indexOf('pgp-encrypted') > -1) {
                baton.view.model.set('pgp_found', true);
                pgpAttach(baton);
                pgpFound.call(this, baton, location);
            }
            if (attachments[i].display = 'inline') {
                if (attachments[i].content !== null) {
                    if (attachments[i].content.indexOf('-----BEGIN PGP MESSAGE-----') > -1) {
                        baton.view.model.set('pgp_found', true);
                        inlineFound.call(this, baton, i, location);
                    }
                    if (attachments[i].content.indexOf('-----BEGIN PGP SIGNED') > -1) {
                        verify(i, baton, true, location);
                    }
                }

            }
            if (attachments[i].content_type.toLowerCase().indexOf('pgp-signature') > -1) {
                verify(i, baton, false, location);
                attachments[i].disp = 'none';  // Don't display the signature attachment
            }
        }
    }

    // Verify signature
    function verify (index, baton, isinline, location) {
        if (baton.signatureDone) return;
        var params = '&folder=' + baton.data.folder_id +
        '&userid=' + ox.user_id + '&cid=' + ox.context_id +
        '&emailid=' + baton.data.id + '&attach=' + index;
        if (isinline === true) params += '&inline=true';
        http.get(ox.apiRoot + '/oxguard/pgpmail?action=verify', params)
        .done(function (data) {
            baton.signatureDone = true;
            var results = JSON.parse(data);
            var result = results.result ? JSON.parse(results.result) : results;
            var signedicon = '';
            if (result.verified === true) {
                signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" aria-hidden="true" title="' + gt('This Email was signed and verified.') + '"/>');
                if (isinline) {
                    var newbody = $('<section class="body user-select-text">').append(result.content);
                    $(location).replaceWith(newbody);
                }
            } else if (result.error !== null && result.missingPublicKey) {
                signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" style="color:lightgrey" aria-hidden="true" title="' + gt('This Email was signed but unable to verify.') + '"/>');
            } else if (!result.verified && result.signature) {
                signedicon = $('<i class="oxguard_icon_fa_alert fa-pencil-square fa guard_signed" aria-hidden="true" title="' + gt('This Email was signed and failed verification.') + '"/>');
            } else {
                return;
            }
            if (_.device('small')) {
                signedicon.css('margin-right', '35px');
            } else {
                signedicon.css('margin-right', '-9px');
            }
            //FIXME: when switching mail-detail very fast, the view might already be disposed
            if (baton.view.$el === null) return;
            var detail = baton.view.$el.find('.detail-view-header');
            if (detail.find('.guard_signed').length < 1) {  // add icon if not already added
                var flagpicker = detail.find('.flag-picker');
                flagpicker.after(signedicon);
            }
            if (baton.data.attachments.length === 2) {
                detail.find('.fa-paperclip').hide();
                baton.data.attachment = false;  // Hide paperclip if just signature
            }
        });
    }

    function busy (location) {
        var working = $('<i class="fa fa-spinner fa-spin" id="busygif" style="padding:20px;"></i>');
        $(location).replaceWith(working);
    }

    // Inline is found, send for decode
    function inlineFound (baton, index, location) {

        var goFunction = function (id) {
            $('#error' + id).show();
            var password = ($('#oxgrpass' + id).val() === undefined) ? '' : $('#oxgrpass' + id).val();
            require (['oxguard/pgp/reader'], function (reader) {
                reader.getInline(baton, password, index);
            });
        };
        baton.data.PGP = true;
        if (window.oxguarddata.passcode === null) {
            require (['oxguard/mail/oxguard_mail'], function (oxmail) {

                $(location).replaceWith(oxmail.passwordPrompt(baton, false, this, goFunction));
                baton.view.$el.find('.attachments').hide();
                if (!_.device('ios')) {
                    baton.view.$el.find('#oxgrpass').focus();
                }
                baton.view.$el.show();

            });
        } else {
            busy (location);
            var go = function (baton, password, index, auth) {
                require (['oxguard/pgp/reader'], function (reader) {
                    reader.getInline(baton, password, index, auth);
                });
            };
            checkauth (go, baton, index);
        }
    }

    // Mime found, do send for decode
    function pgpFound (baton, location) {
        var goFunction = function (id) {
            $('#error' + id).show();
            var password = ($('#oxgrpass' + id).val() === undefined) ? '' : $('#oxgrpass' + id).val();
            require (['oxguard/pgp/reader'], function (reader) {
                reader.getPGP(baton, password);
            });
        };

        if (window.oxguarddata.passcode === null) {
            require (['oxguard/mail/oxguard_mail'], function (oxmail) {
                $(location).replaceWith(oxmail.passwordPrompt(baton, false, this, goFunction));
                baton.view.$el.find('.attachments').remove();
                baton.view.attachmentView = undefined;
                if (!_.device('ios')) {
                    baton.view.$el.find('#oxgrpass').focus();
                }
                baton.view.$el.show();
            });
        } else {
            busy (location);
            var go = function (baton, password, index, auth) {
                require (['oxguard/pgp/reader'], function (reader) {
                    reader.getPGP(baton, password, index, auth);
                });
            };
            checkauth (go, baton, undefined);

        }
    }

    function checkauth (go, baton, index) {
        if (window.oxguarddata.passcode.indexOf('Password Needed') > -1) {// If no password for user has been defined, open dialog
            createOxGuardPasswordPrompt(baton, go, '', undefined, index, location);
            return;
        }
        //FIXME: use util.isGuardConfigured, instead? not sure in this case
        if (window.oxguarddata.passcode.indexOf('No Key') > -1) {// If received a first encrypted email while already logged in, will still have 'no key' in passcode
            createOxGuardPasswordPrompt(baton, go, '', undefined, index, location);
            return;
        }
        go (baton, '', index, window.oxguarddata.passcode);
    }
    // Valide legit email address
    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    function pgpAttach(baton) {
        baton.data.PGP = true;
    }

    // Prompt for initial password change
    function createOxGuardPasswordPrompt(baton, go, errors, oldpass, index) {
        if (firstprompt) return;
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            firstprompt = true;
            var dialog = new dialogs.ModalDialog({ width: 450, center: true, enter: 'okpass' });
            var isnew = window.oxguarddata['new'];
            var explain = '';
            if (isnew) {
                //#. %s product name
                dialog.header($('<h4>').text(gt('First %s Security Use', window.oxguarddata.productName)));
                explain  = $('<div>');
                explain.append($('<p>').append(gt('You have been sent an email using %s to secure the contents', window.oxguarddata.productName)));
                explain.append($('<p>').append(gt('Please select a new password for use with %s.  This password will be used to open encrypted emails and files that may be sent to you.', window.oxguarddata.productName)));
            } else {
                //#. %s product name
                dialog.header($('<h4>').text(gt('First %s Security Password Use', window.oxguarddata.productName)));
                explain = $('<div><p>' + gt('Please change the initial temporary %s security password that was sent to you in a separate email.', window.oxguarddata.productName) + '</p></div>');
            }

            var passdiv = $('<div>').addClass('row-fluid');
            var newogpassword = $('<input name="newogpassword" type="password" autocomplete="off"/>');
            var td1 = $('<td>').append(newogpassword);
            var newogpassword2 = $('<input name="newogpassword2" type="password" autcomplete="off"/>');
            var td2 = $('<td>').append(newogpassword2);
            var hint = $('<td>');
            var table = $('<table class="og_password_prompt"/>');
            if (!isnew) {
                table
                .append('<tr><td class="pw"><em>' + gt('Initial Password') + ':</em></td><td><input name="oldogpassword" type="password" autocomplete="off"/></td></tr>');
            }
            table
            .append($('<tr><td class="pw"><em>' + gt('New Password') + ':</em></td></tr>').append(td1).append(hint))
            //#. Confirm new password
            .append($('<tr><td class="pw"><em>' + gt('Confirm New') + ':</em></td></tr>').append(td2).append('<td></td>'));
            passdiv.append(table);
            dialog.getBody().append(explain).append(passdiv);
            require (['oxguard/core/passwords'], function (pass) {
                pass.passwordCheck(newogpassword, hint);
                pass.passwordCompare(newogpassword2, newogpassword, hint);
            });
            if (window.oxguarddata.settings.noRecovery !== true) {
                var rule = $('<hr/>');
                var recovery = $('<p>' + gt('Please enter a secondary email address in case you need to reset your password.' + '</p>'));

                var table2 = $('<table class="og_password_prompt"/>');
                var email1 = $('<input name="recoveryemail"/>');
                var email2 = $('<input name="verifyemail"/>');
                var hint2 = $('<td>');
                var row = $('<tr>').append($('<td class="pw"><em>' + gt ('Email:') + '</em></td>')).append($('<td>').append(email1)).append(hint2);
                var row2 = $('<tr>').append($('<td class="pw"><em>' + gt ('Verify:') + '</em></td>')).append($('<td>').append(email2));
                table2.append(row).append(row2);
                dialog.getBody().append(rule).append(recovery).append(table2);
                require (['oxguard/core/emailverify'], function (verify) {
                    verify.setValidate(email1, hint2);
                    verify.autoCompare(email1, email2, hint2);
                });
            } else {
                var warning = $('<p>').append($('<b>').append(gt('Warning: This password for encryption cannot be restored or recovered in any way.  If forgotten, all encrypted data will be lost.')));
                dialog.getBody().append($('<hr/>')).append(warning);
            }
            if (errors !== undefined) {
                dialog.getBody().append('<div><span style="color: red;">' + errors + '</span></div>');
            }
            dialog
            .addPrimaryButton('okpass', gt('OK'), 'okpass')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('cancel', function () {
                firstprompt = false;
            })
            .on('okpass', function () {
                firstprompt = false;
                var oldpass = $('input[name="oldogpassword"]').val() === undefined ? '' : $('input[name="oldogpassword"]').val();
                var pass1 = $('input[name="newogpassword"]').val();
                var pass2 = $('input[name="newogpassword2"]').val();
                var emailaddr = $('input[name="recoveryemail"]').val() === undefined ? '' : $('input[name="recoveryemail"]').val();
                var verify = $('input[name="verifyemail"]').val() === undefined ? '' : $('input[name="verifyemail"]').val();
                if (emailaddr !== verify) {
                    $('input[name="verifyemail"]').css('background-color', 'salmon');
                    createOxGuardPasswordPrompt(baton, go, gt('Emails not equal'), oldpass);
                    return;
                }
                if (pass1 !== pass2) {
                    $('input[name="newogpassword2"]').css('background-color', 'salmon');
                    createOxGuardPasswordPrompt(baton, go, gt('Passwords not equal'), oldpass);
                    return;
                }
                var userdata = {
                    newpass: pass1,
                    oldpass: oldpass,
                    email: emailaddr,
                    user_id: ox.user_id,
                    sessionID: ox.session,
                    cid: ox.context_id
                };
                if ((emailaddr.length > 1) && (!validateEmail(emailaddr))) {
                    createOxGuardPasswordPrompt(baton, go, gt('Enter new secondary Email address'), oldpass);
                    return;
                }
                if (validateEmail(emailaddr)) userdata.email = emailaddr;
                require (['io.ox/core/notifications'], function (notifications) {
                    http.post(ox.apiRoot + '/oxguard/login?action=changepass', '', userdata)
                    .done(function (data) {
                        if (typeof data === 'string') data = $.parseJSON(data);
                        if (data.auth === undefined) {
                            require(['oxguard/core/errorHandler'], function (err) {
                                err.showError(data);
                            });
                            return;
                        }
                        if (data.auth.length > 20) {
                            $('#grdsettingerror').text(gt('Success'));
                            $('input[name="newogpassword"]').val('');
                            $('input[name="newogpassword2"]').val('');
                            window.oxguarddata.passcode = null;
                            window.oxguarddata.recoveryAvail = window.oxguarddata.recovery = !window.oxguarddata.settings.noRecovery;
                            go (baton, pass1, index);
                            notifications.yell('success', gt('Password changed successfully'));
                            // Here we want to set the cryptoProvider to Guard to avoid confusion with setup later
                            settings.set('cryptoProvider', 'guard').save();
                            window.oxguarddata['new'] = false;
                            // Complete, now offer tour
                            require(['io.ox/core/capabilities', 'io.ox/core/tk/wizard', 'oxguard/tour/main'], function (capabilities, Tour) {
                                if (capabilities.has('guard-mail')) { // only run tour if has guard-mail
                                    Tour.registry.get('default/oxguard/createKeys').get('run')();
                                }
                            });

                        } else {
                            if (data.auth === 'Bad new password') {
                                notifications.yell('error', gt('New password must be at least %s characters long', data.minlength));
                                return;
                            }
                            if (data.auth === 'Bad password') {
                                //#. Bad, incorrect password
                                notifications.yell('error', gt('Bad password'));
                                return;
                            }
                            if (data.auth === 'Key not found') {
                                notifications.yell('error', gt('Encryption key not found'));
                                return;
                            }
                            notifications.yell('error', gt('Failed to change password'));
                        }
                    })
                    .fail(function () {
                        notifications.yell('error', gt('Failed to change password'));
                    });
                });
            })
            .show(function () {
                if (oldpass !== undefined) {
                    $('input[name="oldogpassword"]').val(oldpass);
                    if (!_.device('ios')) {
                        $('input[name="newogpassword"]').focus();
                    }
                } else
                    if (!_.device('ios')) {
                        $('input[name="oldogpassword"]').focus();
                    }
                window.setTimeout(function () {
                    var j = 0;
                    $('.pw').each(function (i,v) {
                        var w = $(v).width();
                        if (w > j) j = w;
                    });
                    $('.pw').each(function (i,v) {
                        $(v).width(j);
                    });
                    var email = $('input[name="recoveryemail"]'); // Watch email entry for valid
                    email.on('keyup', function () {
                        if (validateEmail(email.val())) {
                            email.css('background-color', 'white');
                        } else {
                            email.css('background-color', 'rgb(253, 187, 179)');
                        }
                    });
                },0);
            });
        });
    }

    return {
        checkPGP: checkPGP,
        createOxGuardPasswordPrompt: createOxGuardPasswordPrompt
    };
});
