/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/oxguard_core', [
    'settings!io.ox/mail',
    'oxguard/core/og_http',
    'gettext!oxguard'
], function (settings, og_http, gt) {
    'use strict';

    var CreateKeysViewModel = Backbone.Model.extend({
        initialize: function () {
            //FIXME: reduce callback hell a little
            if (window.oxguarddata.username === undefined) {
                require(['io.ox/core/api/user']).done(function (user) {
                    window.oxguarddata.username = user.getName().done(function (data) {
                        window.oxguarddata.username = data;
                    });
                });
            }
        },
        defaults: {
            email: '',
            initialSetup: false,
            sent: false
        },
        toJSON: function () {
            var userdata = {
                user_id: ox.user_id,
                sessionID: ox.session,
                language: ox.language,
                cid: ox.context_id
            };

            if (window.oxguarddata.username !== undefined) userdata.name = window.oxguarddata.username;

            return _.extend(userdata, this.pick('email', 'password'));
        },
        send: function () {
            var self = this;
            this.trigger('before:send');
            if (!this.isValid()) {
                this.trigger('send send:error', this.validationError);
                return;
            }
            og_http.post(ox.apiRoot + '/oxguard/login?action=create', (this.get('initialSetup') ? '' : '&add=true'), this.toJSON())
            .done(function (data) {
                if (data.indexOf('ok') > 0) {
                    window.oxguarddata.passcode = null;
                    window.oxguarddata.recoveryAvail = window.oxguarddata.recovery = !window.oxguarddata.settings.noRecovery;
                    self.trigger('send send:ok');
                    self.set('sent', true);
                    if (window.oxguarddata.recovery) window.oxguarddata.recoveryAvail = true;
                } else {
                    self.trigger('send send:error', gt('Problems creating keys, please try again later.'));
                }
            })
            .fail(function (data) {
                var error;
                if (data.responseText.trim() === 'Existing') {
                    error = gt('Problems creating keys.  Keys already exist for this email address in another account.');
                } else {
                    error = gt('Problems creating keys, please try again later.');
                }
                self.trigger('send send:error', error);
            });
        },
        validate: function (attrs) {
            var errors = [];
            var minlen = window.oxguarddata.settings.min_password_length;
            if (minlen !== undefined && attrs.password !== undefined && attrs.password.length < minlen) {
                errors.push({
                    field: '.password',
                    message: gt('Passwords must be at least %s characters in length', minlen)
                });
            }
            if (attrs.password !== attrs.passwordValidation) {
                errors.push({
                    field: '.password-validation',
                    message: gt('Passwords not equal')
                });
            }
            if ((attrs.email.length > 1) && (!validateEmail(attrs.email))) {
                errors.push({
                    field: '.email',
                    message: gt('Enter new secondary Email address')
                });
            }
            return errors.length === 0 ? undefined : errors;
        },
        //handle password strength warnings (those not catched by validate) - adopted from oxquard/core/passwords.passwordCheck
        checkStrength: function () {
            var min = window.oxguarddata.settings.min_password_length;
            var len = window.oxguarddata.settings.password_length;
            if (min === undefined) min = 6;
            if (len === undefined) len = 10;

            if (len <= min) len = min + 1;
            if (!this.get('password')) return;

            if (this.get('password').length < min) {
                this.set('passwordStrength', 'bad');
                return;
            }

            var regex = /(?=^.{8,}$)(?=.*\d)(?=.*[!@#$%^&*]+)(?![.\n])(?=.*[A-Z])(?=.*[a-z]).*$/;
            if (this.get('password').match(regex)) {
                this.set('passwordStrength', 'good');
                return;
            }

            this.set('passwordStrength', 'weak');
        }
    });

    var CreateKeysView = Backbone.View.extend({
        className: 'guard-create-key form-horizontal',
        initialize: function () {
            if (!this.model) this.model = new CreateKeysViewModel();
            this.listenTo(this.model, 'change', this.handleChange);
            this.listenTo(this.model, 'before:send', this.busy);
            this.listenTo(this.model, 'send', this.idle);

            this.$wait = $('<div class="og_wait" id="keygen">').append(
                $('<i class="fa-key fa icon-key fa-spin icon-spin"/>'),
                $('<span>').text(gt('Generating key, Please wait'))
            ).hide();
        },

        events: {
            'keyup input[name=newogpassword]': 'onPasswordChange',
            'keyup input[name=newogpassword2]': 'onPasswordValidationChange',
            'change input[name=recoverymail]': 'onRecoverymailChange'
        },
        onPasswordChange: function (ev) {
            this.model.set('password', ev.target.value);
        },
        onPasswordValidationChange: function (ev) {
            this.model.set('passwordValidation', ev.target.value);
        },
        onRecoverymailChange: function (ev) {
            this.model.set('email', ev.target.value);
        },

        handleChange: function (model) {
            var $el = this.$el;
            //reset error fields
            $el.find('.has-error')
                .removeClass('has-error')
                .find('.error-msg').empty();

            if (!model.isValid()) {
                model.validationError.forEach(function (error) {
                    $el.find(error.field)
                        .addClass('has-error')
                        .find('.error-msg').text(error.message);
                });
            }
            model.checkStrength();
            var strength = model.get('passwordStrength');
            $el.find('.password')
                .toggleClass('has-success', strength === 'good')
                .toggleClass('has-warning', strength === 'weak');
        },

        busy: function () {
            this.$wait.show();
        },
        idle: function () {
            this.$wait.hide();
        },

        renderDescription: function (el) {
            el.append($('<div>').append(
                this.model.get('initialSetup') ? $('<p>').text(gt('Please choose the password you will use for %s. You will need to type this password whenever you want to encrypt or decrypt items. Remember it should be different from your login password, and will not change if you change your login password.', window.oxguarddata.productName)) : '',
                $('<p>').text(gt('Please enter a password to protect your new encrypted items.'))
            ));
            return this;
        },
        renderPasswordPrompt: function (el) {
            el.append(
                $('<div>').addClass('form-group password').append(
                    $('<label>')
                        .addClass('col-sm-12 col-md-4 control-label')
                        .text(gt('Password')),
                    $('<div>').addClass('col-sm-12 col-md-8').append(
                        $('<input name="newogpassword" type="password" autocomplete="off"/>').addClass('form-control')
                    ),
                    $('<div>').addClass('col-sm-12 error-msg')
                ),
                $('<div>').addClass('form-group password-validation').append(
                    $('<label>')
                        .addClass('col-sm-12 col-md-4 control-label')
                        .text(gt('Confirm')),
                    $('<div>').addClass('col-sm-12 col-md-8').append(
                        $('<input name="newogpassword2" type="password" autcomplete="off"/>').addClass('form-control')
                    ),
                    $('<div>').addClass('col-sm-12 error-msg')
                )
            );
            return this;
        },
        renderRecoverMailPrompt: function (el) {
            if (window.oxguarddata.settings.noRecovery === true) {
                el.append($('<hr style="padding:10px;"/>'),
                        $('<p class="oxguard_warning">').text(gt('Warning: This password for encryption cannot be restored or recovered in any way.  If forgotten, all encrypted data will be lost')));
                return this;  // If no recovery set, don't add prompt for second password
            }
            if (this.model.get('initialSetup')) {
                el.append(
                    $('<hr style="padding:10px;"/>'),
                    $('<p>').text(gt('Please enter a secondary email address in case you need to reset your password.')),
                        $('<div>').addClass('form-group email').append(
                        $('<label>').addClass('col-sm-12 col-md-4 control-label').text(gt('Email:')),
                        $('<div>').addClass('col-sm-12 col-md-8').append(
                            $('<input name="recoverymail">').addClass('form-control')
                        ),
                        $('<div>').addClass('col-sm-12 error-msg')
                    )
                );
            }
            return this;
        },
        render: function () {
            this.$el.empty().append(this.$wait);
            return this
                .renderDescription(this.$el)
                .renderPasswordPrompt(this.$el)
                .renderRecoverMailPrompt(this.$el);
        }
    });

    // Add lock to the message detail view and menu options
    var badCount = 0;

    function createKeysWizard(guardOnly) {
        var def = $.Deferred();
        require(['io.ox/core/tk/wizard', 'oxguard/tour/main'], function (Tour) {
            if (guardOnly) {
                Tour.registry.get('default/oxguard/createKeys').get('run')().then(def.resolve, def.reject);
            } else {
                Tour.registry.get('default/oxguard/choice').get('run')().then(def.resolve, def.reject);
            }
        });
        return def;
    }

    /**
     * @returns deferred object resolves to one of:
     *  * "OK" - everything is fine
     *  * "cancel" - user cancelled the dialog
     *  * object - failed with error message stored in responseText attribute
     */
    // Prompt for creating master keys for this user if not yet created.
    function createKeys() {
        var def = $.Deferred(),
            view = new CreateKeysView();

        require (['io.ox/core/notifications', 'io.ox/core/tk/dialogs'], function (notify, dialogs) {

            var dialog = new dialogs.CreateDialog({ width: 450, center: true, enter: 'ok', async: true });
            //#. %s product name
            dialog.header($('<h4>').text(gt('Create %s Security Keys', window.oxguarddata.productName)));
            dialog.getBody().append(view.render().$el);

            dialog
            .addPrimaryButton('ok', gt('OK'), 'ok')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('cancel', function () {
                def.reject('cancel');
            })
            .on('ok', function () {
                view.model.once('send', function (error) {
                    if (error) {
                        def.reject({
                            responseText: error
                        });
                        return;
                    }
                    def.resolve('OK');
                });
                view.model.send();
            })
            .show(function () {  // Normalize the two tables to have same column size
                if (!_.device('ios')) {
                    $('input[name="newogpassword"]').focus();
                }
            });
            view.model.on('change', function (model) {
                var el = dialog.getFooter().find('button[data-action=ok]');
                if (model.isValid()) {
                    el.removeAttr('disabled');
                } else {
                    el.attr('disabled', true);
                }
            });
            def.always(function () {
                view.remove();
                view.model.off();
                view.model = null;
                view = null;
                dialog.close();
            });
        });
        return def;
    }

    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    // Unified inbox combines folder and email ID, returns email id
    function getId(baton) {
        var id = decodeURIComponent(baton.data.id);
        try {
            id = id.substring(id.lastIndexOf('/') + 1);
        } catch (e) {}
        return (id);
    }

    function handleFailEmail (data, baton, deferred, goFunction) {
        baton.view.loaded = true;
        window.oxguarddata.priv = undefined;
        var eid = getId(baton);
        var errorResponse = {
            retry: false,
            errorMessage: 'Fail'
        };
        if (data.status == 503) {
            showError (baton, gt('Unable to connect to the encryption server.'), eid);
        } else {
            if (data.code !== undefined) {
                handleJsonError (data, errorResponse, baton);
            } else {
                var resp = data.responseText.split(':');
                switch (resp[0].trim()) {
                case 'noitem':
                    errorResponse.errorMessage = gt('Unable to find the item on the server');
                    break;
                case 'nocc':
                    errorResponse.errorMessage = gt('Unable to get the key for that item');
                    break;
                case 'nokey':
                    errorResponse.errorMessage = gt('Unable to retrieve your user keys to decode');
                    break;
                case 'Bad extra':
                    errorResponse.errorMessage = gt('Bad extra password');
                    errorResponse.retry = true;
                    break;
                case 'No Private Key associated with the Public Key':
                    errorResponse.errorMessage = gt('No private key to decode this message');
                    errorResponse.retry = false;
                    break;
                case 'Bad Password':
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require (['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && window.oxguarddata.recoveryAvail) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                case 'No secret key':
                    errorResponse.errorMessage = gt('No Private Key available for this encrypted item.') + (resp[1] !== undefined ? ('<br>' + gt('Key Ids:') + resp[1]) : '');
                    errorResponse.retry = false;
                    break;
                case 'Exceeded':
                    errorResponse.errorMessage = gt('Your account is locked out for %s minutes', resp[1]);
                    errorResponse.retry = false;
                    break;
                case 'Error':
                    errorResponse.errorMessage = gt('Error');
                    errorResponse.retry = false;
                    break;
                default:
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    errorResponse.retry = true;
                    break;
                }
            }

            if (data.status == 500) errorResponse.errorMessage = gt('Server Error: ') + data.responseText.trim();

        }
        if (errorResponse.retry) {
            require (['oxguard/mail/oxguard_mail'], function (og_mail) {
                baton.view.$el.find('#content' + eid).replaceWith(og_mail.passwordPrompt(baton, true, undefined, goFunction));
                showErrorOld (baton, errorResponse.errorMessage, eid);
                if (badCount > 2) {
                    baton.view.$el.find('#guardsettings').click(function (e) {
                        ox.launch('io.ox/settings/main', { id: 'oxguard' }).done(function () {
                            this.setSettingsPane({ id: 'oxguard' });
                        });
                        e.preventDefault();
                    });
                }
            });
            baton.view.$el.find('.body').show();
        } else {
            showErrorOld (baton, errorResponse.errorMessage, eid);
            $('.content').hide();
        }
        try {// blank out auth if present
            if (window.oxguarddata.passcode.length > 20) window.oxguarddata.passcode = null;
        } catch (ex) {}
        deferred.reject('error');
    }

    function showErrorOld (baton, error, eid) {
        baton.view.$el.find('#error' + eid).replaceWith('<div class="oxguard_error" id="error' + eid + '"><br>' + error + '</hr></div>');
    }

    function handleJsonError (error, errorResponse, baton) {
        errorResponse.errorMessage = gt('Error');
        errorResponse.retry = false;
        if (error.code !== undefined) {
            switch (error.code) {
                case 'GRD-PGP-0005':
                    if (window.oxguarddata.passcode && (window.oxguarddata.passcode.length > 20)) {  // If bad password based on auth, then clear auth and redo
                        window.oxguarddata.passcode = null;
                        if (baton) baton.view.redraw();
                        return;
                    }
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require (['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && window.oxguarddata.recoveryAvail) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                default:
                    if (error.error !== undefined) {
                        errorResponse.errorMessage = error.error;
                    }
            }

        }
    }

    function showError (errorJson) {
        var errorResp = {};
        try {
            handleJsonError (errorJson, errorResp);
        } catch (e) {
            errorResp.errorMessage = gt('Unspecified error');
        }
        require (['io.ox/core/notifications'], function (notifications) {
            notifications.yell('error', errorResp.errorMessage);
        });

    }

    // Parse the error JSON from an HTML returned error message
    function handleErrorHTML (error) {
        var i = error.indexOf('{');
        if (i >= 0) {
            var j = error.indexOf('}', i);
            if (j > 0) {
                var jsonString = error.substring(i, j + 1);
                var json = JSON.parse(jsonString);
                showError (json);
            }
        }
    }

    function getpassword (message, timeOption) {
        var def = $.Deferred();
        require (['io.ox/core/tk/dialogs', 'oxguard/passwords'], function (dialogs, passwords) {
            var dialog = new dialogs.CreateDialog({ width: 490, center: true,  enter: 'ok' });
            dialog.header($('<h4>').text(gt('Password Needed')));
            var noSaveWorkAround = $('<input style="display:none" type="text" name="dontremember"/><input style="display:none" type="password" name="dontrememberpass"/>');
            //#. %s product name
            var help =  $('<p>' + gt('Please enter your %s password', window.oxguarddata.productName) + '</p>');
            if (message !== undefined) {
                help = $('<p>' + message + '</p>');
            }
            var pass = passwords.createPasswordInput('password');
            dialog.getBody().append(help).append(noSaveWorkAround).append(pass);
            if (timeOption) {
                var remembertext = $('<span class="selectable">&nbsp ' + gt('Remember my password') + '</span>');
                var rememberpass = $('<div><input type="checkbox" name="rememberpass" value="true" style="margin-top:20px;vertical-align:text-bottom;margin-bottom:1px;"/></div>');
                var duration = $('<select id="duration" class="og_duration"><option value="10">' + gt('10 minutes') + '</option>' +
                        '<option value="20">' + gt('20 minutes') + '</option>' +
                        '<option value="30">' + gt('30 minutes') + '</option>' +
                        '<option value="60">' + gt('1 hour') + '</option>' +
                        '<option value="120">' + gt('2 hours') + '</option>' +
                        '<option value="99999">' + gt('Session') + '</option' +
                        '</select>')
                        .on('change', function () {
                            $('input[name="rememberpass"]').prop('checked', true);
                        });
                rememberpass.append(remembertext).append(duration);
                remembertext.click(function () {
                    $('[name="rememberpass"]').click();
                });
                dialog.getBody().append(rememberpass);
            }
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .on('ok', function () {
            })
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('cancel', function () {
                def.reject('cancel');
            })
            .on('ok', function () {
                var duration = -1;
                if ($('input[name="rememberpass"]').is(':checked')) {
                    duration = $('#duration').val();
                }
                var data = {
                    password: $('#password').val(),
                    duration: duration
                };
                def.resolve(data);
            })
            .show(function () {
                window.setTimeout(function () {
                    pass.focus();
                }, 500);
            });
        });
        return (def);
    }

    function sanitize (data) {
        var regex = new RegExp('<(?:!--(?:(?:-*[^->])*--+|-?)|script\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</script\\s*|style\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</style\\s*|/?[a-z](?:[^"\'>]|"[^"]*"|\'[^\']*\')*)>', 'gi');
        return (data.replace(regex, ''));
    }

    function saveAuthToSession (auth, time) {
        var def = $.Deferred();
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        var params = '';
        var authToken = {
            auth: auth,
            minutesValid: (time === undefined ? -1 : time)
        };
        if (auth === null) authToken = {};
        if (time !== undefined) {
            authToken.minutesValid = time;
        }
        og_http.put (url, params, authToken)
        .done (function (data) {
            def.resolve(data);
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    function removeAuth() {
        window.oxguarddata.passcode = null;
        return saveAuthToSession('');
    }

    // Save a password in both auth code and token
    function savePassword (password, duration) {
        var def = $.Deferred();
        authorize(ox.user_id, password)
        .done(function (data) {
            if (data.auth.length > 20) {
                if (duration > -1) {  // -1 is only this event, don't remember
                    window.oxguarddata.passcode = data.auth;
                    var time = duration * 1000 * 60;
                    window.setTimeout(function () {
                        window.oxguarddata.passcode = null;
                    }, time);
                }
                if (duration > 9999) duration = 0;  // 0 for session
                saveAuthToSession (data.auth, duration)
                .done (function () {
                    def.resolve(data);
                })
                .fail (function () {
                    def.reject(data);
                });
            } else {
                def.reject(data);
            }
        })
        .fail (function (e) {
            def.reject(e);
        });
        return def;
    }

    function checkAuth () {
        var def = $.Deferred();
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        og_http.get (url, '')
        .done (function (data) {
            if (data.data === undefined) {
                def.reject('none');
                return;
            }
            data = data.data;
            if (data.minutesValid !== undefined) {
                if (data.minutesValid >= 0) {
                    def.resolve('ok'); // OK only if exists and still time left.
                    return;
                }
            }
            def.reject('exp');
        })
        .fail (function (e) {
            console.log(e);
            def.reject('fail');
        });
        return def;
    }

    function authorize(userid, password, extra, keyid)
    {
        var def = $.Deferred();
        var defaultSendAddress = $.trim(settings.get('defaultSendAddress', ''));
        var userdata = {
            user_id: userid,
            ox_password: password,
            encr_password: '',
            sessionID: ox.session,
            lang: ox.language,
            cid: ox.context_id,
            email: defaultSendAddress,
            extrapass: extra ? extra : ''
        };
        var url = ox.apiRoot + '/oxguard/login?action=login';
        var params = '&time=' + new Date().getTime();
        if (keyid !== undefined) params += '&keyid=' + keyid;
        og_http.post (url, params, userdata)
        .done (function (data) {
            def.resolve(data);
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    function metrics (app, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: app,
                target: 'guard',
                type: 'click',
                action: action
            });
        });
    }

    return {
        // Send authorization to the oxguard server
        auth: authorize,
        createKeys: createKeys,
        createKeysWizard: createKeysWizard,
        CreateKeysViewModel: CreateKeysViewModel,
        CreateKeysView: CreateKeysView,
        validateEmail: validateEmail,
        handleFailEmail: handleFailEmail,
        getPassword: getpassword,
        showError: showError,
        sanitize: sanitize,
        handleErrorHTML: handleErrorHTML,
        saveAuthToSession: saveAuthToSession,
        checkAuth: checkAuth,
        savePassword: savePassword,
        removeAuth: removeAuth,
        metrics: metrics
    };
});

