/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/files/register', [
    'io.ox/core/extensions',
    'io.ox/core/capabilities',
    'oxguard/oxguard_core',
    'oxguard/files/api',
    'io.ox/core/folder/api',
    'oxguard/files/util',
    'io.ox/core/dropzone',
    'gettext!oxguard',
    'io.ox/core/extPatterns/links',
    'less!oxguard/style'
], function (ext, capabilities, oxguard, api, folderAPI, util, dropzone, gt, links) {

    'use strict';

    // For mobile, add a decrypt button
    ext.point('io.ox/files/details').extend({
        index: 500,
        id: 'downloadGuardsmall',
        draw: function (baton) {
            if (_.device('small')) {
                if (_.device('ios')) return;   // ios doesn't allow downloading.
                if (baton.data.ogfile === true) {
                    var download = $('<a href="#" class="btn btn-default">' + gt('Download Decrypted') + '</a>');
                    download.click(function (ev) {
                        ev.preventDefault();
                        var list = [baton.data];
                        authenticate().then(function (data) {
                            require (['oxguard/files/downloader'], function (downloader) {
                                metrics('guard', 'decrypt & download');
                                downloader.getFile(list, 'download', data.auth);
                            });
                        });
                    });
                    $(this).append(download);
                }
            }
        }
    });

    ///////////  Main toolbar in drive

    //////   Links
    // Add link for add and encrypt local file
    new links.ActionLink('io.ox/files/links/toolbar/default', {
        index: 110,
        id: 'uploadEncrypt',
        label: gt('Add and encrypt local file'),
        ref: 'io.ox/files/actions/uploadEncrypt'
    });

    ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link({
        id: 'remencrypt',
        index: 550,
        label: gt('Remove Encryption'),
        ref: 'oxguard/remencrypt',
        mobile: 'lo',
        section: 'guard'
    }));

    ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link({
        id: 'encrypt',
        index: 550,
        label: gt('Encrypt'),
        ref: 'oxguard/encrypt',
        mobile: 'lo',
        section: 'guard'
    }));

    // Change send copy to handle Guard emails
    ext.point('io.ox/files/actions/send').extend({
        id: 'og_stop_send',
        index: 1,
        requires: function (e) {
            if (e.collection.has('some', 'items') && !_.isEmpty(e.baton.data)) {
                var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
                if (util.hasEncrypted(toCheck)) {
                    e.stopPropagation();
                    return false;
                }
            }
        }
    });

    ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link({
        id: 'sendcopy',
        index: 1200,
        prio: 'lo',
        mobile: 'lo',
        label: gt('Send by mail'),
        ref: 'oxguard/sendcopy',
        section: 'share'
    }));

    //////////////  Viewer
    /// Links

    ext.point('io.ox/core/viewer/actions/toolbar/popoutstandalone').extend({
        id: 'popoutstandalone_guardcheck',
        index: 1,
        requires: function (e) {
            if (e.collection.has('one')) {
                if (e.baton.model.isEncrypted()) {
                    e.stopPropagation();
                    return false;
                }
            }
        }
    });

    ext.point('io.ox/core/viewer/toolbar/links/drive').extend(new links.Link({
        id: 'GuardViewerEncrypt',
        index: 200,
        label: gt('Encrypt'),
        ref: 'oxguard/encrypt',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guard').extend(new links.Link({
        id: 'GuardViewerRemEncrypt',
        index: 200,
        label: gt('Remove Encryption'),
        ref: 'oxguard/remencrypt',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guard').extend(new links.Link({
        id: 'GuardViewerDownload',
        index: 200,
        label: gt('Download Decrypted'),
        ref: 'oxguard/download',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guard').extend(new links.Link({
        id: 'GuardViewerSend',
        index: 201,
        label: gt('Send copy by mail'),
        ref: 'oxguard/sendcopy',
        mobile: 'lo'
    }));

    ext.point('io.ox/core/viewer/toolbar/links/guard').extend(new links.Link({
        prio: 'hi',
        mobile: 'lo',
        label: gt('Pop out'),
        icon: 'fa  fa-external-link-square',
        ref: 'oxguard/popoutstandalone',
        customize: function () {
            this.addClass('viewer-toolbar-popoutstandalone')
                .attr({
                    tabindex: '1',
                    title: gt('Pop out standalone viewer'),
                    'aria-label': gt('Pop out standalone viewer')
                });
        }
    }));

    // Versions
    ext.point('io.ox/files/versions/links/inline').extend(new links.Link({
        id: 'GuardView',
        index: 90,
        label: gt('View'),
        ref: 'oxguard/versionView'
    }));

    ext.point('io.ox/files/versions/links/inline').extend(new links.Link({
        id: 'GuardOpen',
        index: 100,
        label: gt('Decrypt and Open'),
        ref: 'oxguard/open'
    }));

    ext.point('io.ox/files/versions/links/inline').extend(new links.Link({
        id: 'GuardDownload',
        index: 200,
        label: gt('Download Decrypted'),
        ref: 'oxguard/download'
    }));

    // Remove the loc portion of the name in version list
    ext.point('io.ox/files/details/version').extend({
        index: 1,
        id: 'GuardVersion',
        draw: function () {}
    });

    //// Actions
    // Upload new encrypted file
    new links.Action('io.ox/files/actions/uploadEncrypt', {
        requires: function (e) {
            if (util.isMailvelope ()) return false;
            return e.baton.app.folder.getData().then(function (data) {
                //hide for virtual folders (other files root, public files root)
                var virtual = _.contains(['14', '15'], data.id);
                //no new files in trash folders
                return folderAPI.can('create', data) && !virtual && !folderAPI.is('trash', data);
            });
        },
        action: function (baton) {
            ensureSetup().then(function () {
                var elem = $(baton.e.target),
                    input;

                // remove old input-tags resulting from 'add local file' -> 'cancel'
                elem.siblings('input').remove();

                elem.after(
                    input = $('<input type="file" name="file" capture="camera" multiple>')
                    .css('display', 'none')
                    .on('change', function (e) {
                        var app = baton.app;
                        require(['io.ox/files/upload/main', 'io.ox/files/api'], function (fileUpload, origAPI) {
                            e.preventDefault();

                            var list = [];
                            _(e.target.files).each(function (file) {
                                list.push(_.extend(file, { group: 'file' }));
                            });
                            fileUpload.setWindowNode(app.getWindowNode());
                            fileUpload.setAPI(api);
                            metrics('guard', 'upload & encrypt');
                            var ul = fileUpload.create;
                            ul.on('stop', function () {
                                fileUpload.setAPI(origAPI);
                            });
                            ul.offer(list, { folder: app.folder.get() });
                        });
                        input.remove();
                    })
                );

                input.trigger('click');
            });
        }
    });

    ext.point('io.ox/files/actions/open').replace({
        id: 'default',
        requires: function (e) {
            if (e.collection.has('multiple')) return false;
            if (e.baton.data) {
                if (e.baton.data.meta) {
                    if (e.baton.data.meta.Encrypted === true || (isOGFile(e.baton.data))) {
                        return (false);
                    }
                }
            }
            // 'description only' items
            return !_.isEmpty(e.baton.data.filename) || e.baton.data.file_size > 0;
        }
    });

    // Replacement popout for Guard
    new links.Action('oxguard/popoutstandalone', {
        requires: function (e) {
            if (e.collection.has('one')) {
                if (e.baton.model.isEncrypted()) return true;
            }
            return false;
        },
        action: function (e) {
            ox.launch('io.ox/files/detail/main').done(function () {
                var node = this.getWindowNode();
                this.setTitle(e.data.filename);
                var app = this;
                ox.load(['io.ox/mail/actions/viewer']).done(function (action) {
                    action({ files: [e.data], app: app, container: node, standalone: true });
                });
            });
        }
    });

    // View link within versions
    new links.Action('oxguard/versionView', {
        requires: function (e) {
            return (isOGFile(e.baton.data));
        },
        multiple: function (list) {
            viewFile (list[0]);
        }
    });

    new links.Action('oxguard/sendcopy', {
        requires: function (e) {
            if (util.isMailvelope ()) return false;
            //all files must be encrypted
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            return e.collection.has('some') && util.hasEncrypted(toCheck) && util.isNotAttachment(toCheck) && capabilities.has('webmail');
        },
        multiple: function (list) {
            list.map(function (file) {
                if (file.source === 'guard') window.oxguarddata.action = 'fileAttach';
            });
            ext.point('io.ox/files/actions/send').invoke('multiple', this, list);
        }
    });

    new links.Action('oxguard/remencrypt', {
        requires: function (e) {
            //all files must be encrypted
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            return e.collection.has('some') && util.encryptedOnly(toCheck) && util.isNotAttachment(toCheck);
        },
        multiple: function (list, baton) {
            var opt = {
                extraPassword: extrapasslist (list)
            };
            var win = baton.app === undefined ? null : baton.app.getWindow();
            authenticate(opt).then(function (auth) {
                if (win !== null) win.busy();
                metrics('guard', 'remove-encryption');
                api.removeEncryption((baton.models === null ? baton.model : baton.models), auth)
                .then(function () {
                    if (baton.context !== undefined && baton.context.viewerEvents !== undefined) {
                        baton.context.viewerEvents.trigger('viewer:close');
                    }
                    reload();
                })
                .always(function () {
                    if (win !== null) win.idle();
                });
            });
            baton.openedBy = 'io.ox/files';
        }
    });

    new links.Action('oxguard/open', {
        requires: function (e) {
            if (e.baton.data) {
                if (e.baton.data.meta) {
                    if (e.baton.data.meta.Encrypted === true || (isOGFile(e.baton.data))) {
                        return (true);
                    }
                }
            }
            return (false);
        },
        multiple: function (list) {
            authenticate().then(function (data) {
                var popup = window.open('', '_blank');
                require (['oxguard/files/downloader'], function (downloader) {
                    metrics('guard', 'decrypt & open');
                    downloader.getFile(list, 'open', data.auth, null, popup);
                });
            });
        }
    });

    new links.Action('oxguard/download', {
        requires: function (e) {
            if (_.device('ios')) return false;
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            if (toCheck === undefined) return false;
            return e.collection.has('some') && util.encryptedOnly(toCheck) && util.isNotAttachment(toCheck);
        },
        multiple: function (list) {
            authenticate().then(function (data) {
                require (['oxguard/files/downloader'], function (downloader) {
                    metrics('guard', 'decrypt & download');
                    downloader.getFile(list, 'download', data.auth);
                });
            });
        }
    });

    ////// Main listview handling

    ext.point('io.ox/files/actions/default').extend({
        id: 'first',
        action: function (baton) {
            console.log(baton);
            if (baton.data.meta && baton.data.meta.Encrypted === true) {
                baton.stopPropagation();
                metrics('guard', 'view-encrypted');
                viewFile (baton.data);
            }
        }
    });

    // Call for viewer.  Check if Guard file
    ext.point('io.ox/files/actions/viewer').extend({
        id: 'guardview',
        index: 1,
        requires: function (e) {
            var toCheck = e.baton.models === null ? e.baton.model : e.baton.models;
            if (toCheck === undefined) return false;
            return e.collection.has('some') && util.encryptedOnly(toCheck);
        },
        action: function (baton) {
            if (baton.data.meta.Encrypted === true) {
                viewFile(baton.data);
                baton.preventDefault();
                //baton.disable('io.ox/files/actions/viewer', 'view');
            }
        }
    });

    // Handle viewing Guard file
    function viewFile (file) {
        if (isOGFile (file)) {
            var list = [file];
            authenticate().then(function (data) {
                require (['oxguard/files/downloader'], function (downloader) {
                    downloader.getFile(list, 'view', data.auth);
                });
            });
        }
    }

    // Handle the upload new version for files
    // Add drop zone for Guard.  For now, hide button
    ext.point('io.ox/core/viewer/views/sidebarview/detail').extend({
        id: 'guardDetailFile',
        index: 10000,
        draw: function (baton) {
            if (util.isMailvelope ()) return false;
            if (baton.model.isEncrypted()) {
                if (baton.context.zone !== null) {
                    baton.context.zone.unbind('drop');
                    baton.context.zone.bind('drop', uploadNewGuardVersion.bind(baton.context));
                }
            }
        }
    });

    ext.point('io.ox/core/viewer/views/sidebarview/uploadnewversion').extend({
        id: 'guardUploadNewVersion',
        index: 1,
        draw: function () {
            if (this.model !== null && this.model !== undefined) {
                if (this.model.isEncrypted()) {
                    var input = this.$el.find('input[type="file"]');
                    var file = this.model;
                    input.unbind().on('change', function (e) {
                        e.stopImmediatePropagation();
                        // do upload here
                        uploadNewVersion(this, file);
                    });
                }
            }
        }
    });

    // Do upload new Guard version from dropzone
    function uploadNewGuardVersion (files) {
        api.versions.upload({
            file: _.first(files),
            id: this.model.get('id'),
            folder: this.model.get('folder_id'),
            version_comment: ''
        });
    }

    // Add extra drop zone for upload new Guard file
    ext.point('io.ox/files/mediator').replace({
        id: 'files-dropzone',
        index: 1000000000000,
        setup: function (app) {
            // desktop only
            if (!_.device('desktop')) return;

            var zone = new dropzone.Inplace({
                caption: gt('Drop files here to upload')
            });

            var zone2 = new dropzone.Inplace({
                caption: gt('Drop files here to encrypt')
            });

            zone.on({
                'drop': function (files) {
                    require(['io.ox/files/upload/main'], function (fileUpload) {
                        fileUpload.setWindowNode(app.getWindowNode());
                        fileUpload.create.offer(files, { folder: app.folder.get() });
                    });
                }
            });

            zone2.on({
                'drop': function (files) {
                    ensureSetup().then(function () {
                        metrics('guard', 'dropfile upload & encrypt');
                        require(['io.ox/files/upload/main', 'io.ox/files/api'], function (fileUpload, origAPI) {
                            var list = [];
                            _(files).each(function (file) {
                                list.push(_.extend(file, { group: 'file' }));
                            });
                            fileUpload.setWindowNode(app.getWindowNode());
                            fileUpload.setAPI(api);
                            var ul = fileUpload.create;
                            ul.on('stop', function () {
                                fileUpload.setAPI(origAPI);
                            });
                            ul.offer(list, { folder: app.folder.get() });
                        });
                    });
                }
            });

            if (util.isMailvelope ()) {
                app.getWindowNode().find('.list-view-control').append(
                        (zone.render().$el).addClass('abs'));
            } else {
                app.getWindowNode().find('.list-view-control').append(
                        (zone.render().$el).addClass('abs').css('height', '50%')
                    ).append(
                        (zone2.render().$el).addClass('abs').css('top', '50%')
                    );
            }

        }
    });

    ////// Functions / util

    function authenticate (opt) {
        var def = $.Deferred();
        opt = opt || {};
        if (!opt.extraPassword && window.oxguarddata && window.oxguarddata.passcode && window.oxguarddata.passcode.length >= 20) {
            oxguard.checkAuth()  // make sure valid session before going on
            .done(function () {
                def.resolve({
                    auth: window.oxguarddata.passcode
                });
                return;
            })
            .fail(function () {
                window.oxguarddata.passcode = null;
                def.reject();
            });
            return def;
        }
        oxguard.getPassword(gt('Enter %s Security Password', window.oxguarddata.productName), true)  //Get password
        .done(function (passdata) {
            if (passdata.duration > -1) {
                oxguard.savePassword(passdata.password, passdata.duration)
                .done(function (data) {
                    def.resolve(data);
                })
                .fail(function (data) {
                    handleFail (data);
                    def.reject(data);
                });
            } else {
                oxguard.auth(ox.user_id, passdata.password)
                .done(function (data) {
                    if (data.auth.length > 20) {
                        def.resolve(data);
                    } else {
                        handleFail(data);
                        def.reject(data);
                    }
                });
            }
        });
        return def;
    }

    function ensureSetup() {

        if (!util.isGuardConfigured()) {
            oxguard.metrics('drive', 'create-keys');
            return oxguard.createKeysWizard(true);
        }

        return $.when();
    }

    ////////  Encrypting files

    new links.Action('oxguard/encrypt', {
        requires: function (e) {
            if (util.isMailvelope ()) return false;
            //all files must be unencrypted
            if (e.collection.has('folders')) return false;
            return e.collection.has('some') && !util.encryptedOnly(e.baton.models);
        },
        multiple: function (list, baton) {
            var win = baton.app === undefined ? null : baton.app.getWindow();
            ensureSetup().then(function () {
                // Check if there are multiple versions
                var multipleVersions = baton.models.reduce(function (acc, file) {
                    return acc || file.number_of_versions > 1;
                }, false);

                if (multipleVersions) {
                    var def = $.Deferred();
                    // If so, display warning that only latest version saved
                    require (['io.ox/core/tk/dialogs'], function (dialogs) {
                        var dialog = new dialogs.CreateDialog({ width: 450, center: true,  enter: 'ok' });
                        dialog.header($('<h4>').text(gt('Encrypt Files')));
                        dialog.getBody().append(
                            $('<p>').append(
                                gt('Only the latest version of the files will be encrypted.'),
                                $('<br/>'),
                                gt('All previous versions will be deleted.')
                            ),
                            $('<p style="margin-top:10px;">').append(
                                gt('Click OK to confirm')
                            )
                        );

                        dialog
                            .addPrimaryButton('ok', gt('OK'), 'ok')
                            .addButton('cancel', gt('Cancel'), 'cancel')
                            .on('ok', def.resolve)
                            .on('cancel', def.reject)
                            .show(function () {
                                if (!_.device('ios')) {
                                    $('input[name="ogpassword"]').focus();
                                }
                            });
                    });
                    return def;
                }
            }).then(function () {
                metrics('guard', 'encrypt-file');
                if (win !== null) win.busy();

                return api.encrypt(baton.models)
                .then(function () {
                    if (baton.context !== undefined && baton.context.viewerEvents !== undefined) {
                        baton.context.viewerEvents.trigger('viewer:close');
                    }
                    reload();
                })
                .fail(function (e) {
                    require (['io.ox/core/notifications'], function (notify) {
                        if (e.responseText.indexOf('FLS-0024') > -1) {
                            notify.yell('error', gt('The allowed quota is reached.'));
                        } else {
                            notify.yell('error', gt('This file could not be uploaded.') + e.responseText);
                        }

                    });
                })
                .always(function () {
                    if (win !== null) win.idle();
                });
            });
        }
    });

    /////////////////  Util, prompts, etc

    function metrics (target, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: 'drive',
                target: target,
                type: 'click',
                action: action
            });
        });
    }

    function uploadNewVersion (input, orig) {
        var files = input.files;
        require(['io.ox/core/tk/dialogs'], function (Dialogs) {
            new Dialogs.ModalDialog()
            .header(
                $('<h4>').text(gt('Version Comment'))
            )
            .append(
                $('<textarea rows="6" class="form-control" tabindex="1">')
            )
            .addPrimaryButton('upload', gt('Upload'), 'upload',  { 'tabIndex': '1' })
            .addButton('cancel', gt('Cancel'), 'cancel',  { 'tabIndex': '1' })
            .on('upload', function () {
                var comment = this.getContentNode().find('textarea').val() || '';

                api.versions.upload({
                    file: _.first(files),
                    id: orig.get('id'),
                    folder: orig.get('folder_id'),
                    version_comment: comment
                });
            })
            .show(function () {
                if (!_.device('ios')) {
                    this.find('.btn-primary').focus();
                }
            });
        });
    }

    function handleFail (data, popup) {
        require(['io.ox/core/notifications'], function (notifications) {
            if (data.auth === 'Bad RSA') {
                window.oxguarddata.pubkey = null;
                return;
            }
            if (data.auth === 'Bad Password') {
                if (popup) popup.close();
                //#. Bad, incorrect password
                notifications.yell('error', gt('Bad password'));
                return;
            }
            if (data.auth === 'Lockout') {
                if (popup) popup.close();
                notifications.yell('error', gt('Temporary Lockout'));
                return;
            }
        });

    }

    function reload () {
        require(['io.ox/files/main'], function (main) {
            main.getApp().listView.reload();
        });
    }

    function isOGFile(file) {
        try {
            if (file.filename === null) return (false);
            if (file.meta !== undefined) {
                if (file.meta.Encrypted) return (true);
            }
            if ((file.filename.indexOf('.grd') > 0) || (file.filename.indexOf('.grd2') > 0) || (file.filename.indexOf('.pgp') > 0)) return (true);
        } catch (e) {
            return (false);
        }
        return (false);
    }

    function extrapass(filename) {
        if (filename === null) return (false);
        if ((filename.indexOf('.grd2') > 0)) return (true);
        return (false);
    }

    function extrapasslist(list) {
        for (var i = 0; i < list.length; i++) {
            if (extrapass(list[i].filename) || extrapass(list[i].title)) return (true);
        }
        return (false);
    }

    return {
        viewFile: viewFile
    };
});
