/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.transport.listener;

import javax.mail.internet.MimeMessage;
import com.openexchange.exception.OXException;
import com.openexchange.guard.api.GuardApis;
import com.openexchange.guard.exceptions.GuardExceptionCodes;
import com.openexchange.guard.internal.AbstractGuardAccess;
import com.openexchange.guard.internal.GuardApiImpl;
import com.openexchange.guard.internal.authentication.AuthenticationToken;
import com.openexchange.guard.internal.authentication.AuthenticationTokenHandler;
import com.openexchange.mail.transport.listener.MailTransportListener;
import com.openexchange.mail.transport.listener.Reply;
import com.openexchange.mail.transport.listener.Result;
import com.openexchange.session.Session;

/**
 * {@link GuardTransportListener}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.8.0
 */
public class GuardTransportListener extends AbstractGuardAccess implements MailTransportListener {

    private static final String HEADER_GUARD = "X-OX-Guard-Marker";

    /**
     * Initializes a new {@link GuardTransportListener}.
     */
    public GuardTransportListener() {
        super();
    }

    @Override
    public Result onBeforeMessageTransport(MimeMessage message, Session session) throws OXException {
        if (!"true".equalsIgnoreCase(getHeaderSafe(message))) {
            return new GuardResult(message, Reply.NEUTRAL);
        }

        GuardApiImpl guardApi = getGuardApi(GUARDADMIN_ENDPOINT);
        if (null == guardApi) {
            // Guard end point not available
            return new GuardResult(message, Reply.NEUTRAL);
        }

        AuthenticationToken authToken = new AuthenticationTokenHandler().getForSession(session);
        if (authToken != null) {
            MimeMessage processedMessage = guardApi.processMimeMessage(message, GuardApis.mapFor("action", "process_message", "auth", authToken.getValue(), "guardsession", authToken.getGuardSessionId(), "user", Integer.toString(session.getUserId()), "context", Integer.toString(session.getContextId())));
            return new GuardResult(processedMessage, Reply.ACCEPT);
        } else {
            throw GuardExceptionCodes.MISSING_AUTH.create();
        }
    }

    private String getHeaderSafe(MimeMessage message) {
        try {
            return message.getHeader(HEADER_GUARD, null);
        } catch (Exception e) {
            return null;
        }
    }

    @Override
    public void onAfterMessageTransport(MimeMessage message, Exception exception, Session session) throws OXException {
        // Nothing to do
    }

    // -----------------------------------------------------------------------------------------------------------------------------

    private static class GuardResult implements Result {

        private final MimeMessage mimeMessage;
        private final Reply reply;

        GuardResult(MimeMessage mimeMessage, Reply reply) {
            super();
            this.mimeMessage = mimeMessage;
            this.reply = reply;
        }

        @Override
        public Reply getReply() {
            return reply;
        }

        @Override
        public MimeMessage getMimeMessage() {
            return mimeMessage;
        }
    }

}
