/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.json.actions;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.guard.internal.authentication.AuthenticationToken;
import com.openexchange.guard.internal.authentication.AuthenticationTokenHandler;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link AuthTokenAction} attaches or detaches an {@link AuthenticationToken} to or from a groupware session
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.2
 */
public class AuthTokenAction extends AbstractGuardAction {

    private static final String FIELD_MINUTES_VALID = "minutesValid";
    private static final String FIELD_GUARD_SESSION = "guardSession";
    private static final String FIELD_AUTH = "auth";
    private static final String JSESSIONID = "JSESSIONID";
    private static final String JSON = "json";

    /**
     * Initializes a new {@link AuthTokenAction}.
     *
     * @param services
     */
    public AuthTokenAction(ServiceLookup services) {
        super(services);
    }

    /**
     * Internal method to extract a cookie from the incoming request
     *
     * @param cookieName The name of the cookie
     * @param request The request
     * @return The cookie with the given name, or null if no such cookie was found
     */
    private Cookie extractCookieFrom(String cookieName, HttpServletRequest request) {
        Cookie foundCookie = null;
        Cookie[] cookies = request.getCookies();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals(cookieName)) {
                    foundCookie = cookie;
                }
            }
        }
        return foundCookie;
    }

    /**
     * Creates an {@link AuthenticationToken} from the given request
     *
     * @param requestData The request
     * @return The authentication token parsed from the given JSON
     * @throws JSONException
     * @throws OXException
     */
    private AuthenticationToken toAuthenticationToken(AJAXRequestData requestData) throws JSONException, OXException {
        JSONObject json = (JSONObject) requestData.requireData();

        if (!json.has(FIELD_AUTH)) {
            throw AjaxExceptionCodes.MISSING_FIELD.create(FIELD_AUTH);
        }
        String authToken = json.getString(FIELD_AUTH);

        String guardSessionId = null;
        if (json.has(FIELD_GUARD_SESSION)) {
            guardSessionId = json.getString(FIELD_GUARD_SESSION);
        }
        else {
            //Using the JSESSIONID cookie value as a fallback
            HttpServletRequest request = requestData.optHttpServletRequest();
            if (request != null) {
                Cookie jsessionID = extractCookieFrom(JSESSIONID, request);
                if (jsessionID != null) {
                    guardSessionId = jsessionID.getValue();
                }
            }
        }
        if (guardSessionId == null) {
            throw AjaxExceptionCodes.MISSING_FIELD.create(FIELD_GUARD_SESSION);
        }

        int minutesValid = -1;
        if (json.has(FIELD_MINUTES_VALID)) {
            minutesValid = json.getInt(FIELD_MINUTES_VALID);
        }
        return new AuthenticationToken(guardSessionId, authToken, minutesValid);
    }

    /**
     * Creates a JSON response for a given authentication token
     *
     * @param token The token
     * @return the token as JSONObject
     * @throws JSONException
     */
    private JSONObject asJsonResult(AuthenticationToken token) throws JSONException {
        if (token != null) {
            JSONObject result = new JSONObject();
            result.put("createdOn", token.getCreatedOn().getTime());
            result.put("minutesValid", token.getMinutesValid());
            return result;
        }
        return null;
    }

    /**
     * Gets if a Guard authentication token has been attached to a session
     *
     * @param requestData The request data
     * @param session The session
     * @return The result describing whether an authentication token has been attached to a session or not
     * @throws OXException
     * @throws JSONException
     */
    private AJAXRequestResult doPerformGet(AJAXRequestData requestData, ServerSession session) throws OXException, JSONException {
        AuthenticationToken tokenForSession = new AuthenticationTokenHandler().getForSession(session, false);
        requestData.setResponseHeader("content-type", "application/json; charset=UTF-8");
        return new AJAXRequestResult(asJsonResult(tokenForSession), "json");
    }

    /**
     * Attache/detaches a Guard authentication token to/from a given session
     *
     * @param requestData The request data
     * @param session The session
     * @return an empty result
     * @throws OXException
     * @throws JSONException
     */
    private AJAXRequestResult doPerformSet(AJAXRequestData requestData, ServerSession session) throws OXException, JSONException {
        AuthenticationToken authenticationToken = toAuthenticationToken(requestData);
        if (authenticationToken.getValue() == null || authenticationToken.getValue().isEmpty()) {
            //Detach existing token
            new AuthenticationTokenHandler().removeForSession(session);
        }
        else {
            //Attach a new  token
            new AuthenticationTokenHandler().setForSession(session, authenticationToken);
        }
        return new AJAXRequestResult(null, "json");
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.json.actions.AbstractGuardAction#doPerform(com.openexchange.ajax.requesthandler.AJAXRequestData, com.openexchange.tools.session.ServerSession)
     */
    @Override
    protected AJAXRequestResult doPerform(AJAXRequestData requestData, ServerSession session) throws OXException, JSONException {
        if (requestData.optHttpServletRequest().getMethod().equals("GET")) {
            //Query existing token
            return doPerformGet(requestData, session);
        }
        else {
            //Attach/Dettach authentication token
            return doPerformSet(requestData, session);
        }
    }
}
