/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.internal.authentication;

import java.util.concurrent.locks.Lock;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.api.GuardApis;
import com.openexchange.guard.exceptions.GuardExceptionCodes;
import com.openexchange.guard.internal.AbstractGuardAccess;
import com.openexchange.guard.internal.GuardApiImpl;
import com.openexchange.session.Session;

/**
 * {@link AuthenticationTokenHandler} handles the Guard authentication token
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.2
 */
public class AuthenticationTokenHandler extends AbstractGuardAccess {

    private static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(AuthenticationTokenHandler.class);
    private static final String AUTH_TOKEN_PARAMETER = "GUARD_AUTH_TOKEN";

    /**
     * Internal method to destroy the authentication token on the remote Guard service.
     * 
     * @param authToken The authentication token to destroy
     * @throws OXException
     * @throws JSONException
     */
    private void destroyAuthToken(AuthenticationToken authToken) throws OXException {

        GuardApiImpl guardApi = getGuardApi(PGPCORE_ENDPOINT);
        if (guardApi != null) {
            JSONObject json = new JSONObject();
            json.putSafe("session", authToken.getGuardSessionId());
            guardApi.doCallPut(GuardApis.mapFor("action", "deleteauthtoken"), json, Void.class);
        } else {
            logger.error("Endpoint not available " + PGPCORE_ENDPOINT);
        }
    }

    /**
     * Gets the authentication token for a given session
     * 
     * @param session The session to get the authentication token for
     * @param markAsUsed Whether to mark the token as used or not
     * @return An authentication token for the given session, or null if no authentication token was attached to the session
     * @throws OXException
     */
    public AuthenticationToken getForSession(Session session, boolean markAsUsed) throws OXException {
        final Lock lock = (Lock) session.getParameter(Session.PARAM_LOCK);
        lock.lock();
        try {
            if (session.containsParameter(AUTH_TOKEN_PARAMETER)) {
                AuthenticationToken authToken = (AuthenticationToken) session.getParameter(AUTH_TOKEN_PARAMETER);
                if (markAsUsed) {
                    authToken.setUsed();
                }
                return authToken;
            }
            return null;
        } finally {
            lock.unlock();
        }
    }

    /**
     * Gets the authentication token for a given session and mark the token as "used"
     * 
     * @param session The session to get he authentication token for
     * @return The authentication token for the given session, or null if no authentication token was attached to the session
     * @throws OXException
     */
    public AuthenticationToken getForSession(Session session) throws OXException {
        return getForSession(session, true);
    }

    /**
     * Gets the authentication token for a given session and mark the token as "used"
     * 
     * @param session The session to get the authentication token for
     * @return The authentication token for the given session
     * @throws OXException If no such token was found for the given session
     */
    public AuthenticationToken requireForSession(Session session) throws OXException {
        AuthenticationToken authToken = getForSession(session, true);
        if (authToken != null) {
            return authToken;
        }
        throw GuardExceptionCodes.MISSING_AUTH.create();
    }

    /**
     * Attaches the given authentication token for the associated session
     * 
     * @param session The session
     * @param authToken The authentication token
     * @return
     * @throws OXException
     */
    public void setForSession(Session session, AuthenticationToken authToken) throws OXException {
        final Lock lock = (Lock) session.getParameter(Session.PARAM_LOCK);
        lock.lock();
        try {
            session.setParameter(AUTH_TOKEN_PARAMETER, authToken);
        } finally {
            lock.unlock();
        }
    }

    /**
     * Removes an authentication token for the given session.
     * <p>
     * This will invalidate an authentication token on the Guard server as well.
     * 
     * @param session The session
     * @throws OXException
     */
    public void removeForSession(Session session) throws OXException {
        AuthenticationToken authToken = null;

        final Lock lock = (Lock) session.getParameter(Session.PARAM_LOCK);
        lock.lock();
        try {
            authToken = getForSession(session, false);
            if (authToken != null) {
                session.setParameter(AUTH_TOKEN_PARAMETER, null);
            }
        } finally {
            lock.unlock();
        }

        if (authToken != null) {
            destroyAuthToken(authToken);
        }
    }
}
