/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.auth.AuthScheme;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.AuthState;
import org.apache.http.auth.AuthenticationException;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CookieStore;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.client.params.CookiePolicy;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.InputStreamBody;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.ExecutionContext;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONValue;
import org.slf4j.Logger;
import com.openexchange.ajax.AJAXServlet;
import com.openexchange.ajax.container.ThresholdFileHolder;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.guard.api.Cookie;
import com.openexchange.guard.api.GuardApi;
import com.openexchange.guard.api.GuardApiExceptionCodes;
import com.openexchange.java.Charsets;
import com.openexchange.java.Streams;
import com.openexchange.java.Strings;
import com.openexchange.java.util.UUIDs;
import com.openexchange.mail.mime.MimeDefaultSession;
import com.openexchange.mail.mime.MimeMailException;
import com.openexchange.mail.mime.converters.FileBackedMimeMessage;
import com.openexchange.mail.mime.utils.MimeMessageUtility;
import com.openexchange.rest.client.httpclient.HttpClients;
import com.openexchange.rest.client.httpclient.HttpClients.ClientConfig;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.session.Session;
import com.openexchange.version.Version;

/**
 * {@link GuardApiImpl}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach></a>
 * @since v7.8.0
 */
public class GuardApiImpl implements GuardApi {

    private static final Logger LOGGER = org.slf4j.LoggerFactory.getLogger(GuardApiImpl.class);

    // -------------------------------------------------------------------------------------------------------------- //

    /** The status code policy to obey */
    public static interface StatusCodePolicy {

        /**
         * Examines given status line
         *
         * @param httpResponse The HTTP response
         * @throws OXException If an Open-Xchange error is yielded from status
         * @throws HttpResponseException If status is interpreted as an error
         */
        void handleStatusCode(HttpResponse httpResponse) throws OXException, HttpResponseException;
    }

    /** The default status code policy; accepting greater than/equal to <code>200</code> and lower than <code>300</code> */
    public static final StatusCodePolicy STATUS_CODE_POLICY_DEFAULT = new StatusCodePolicy() {

        @Override
        public void handleStatusCode(HttpResponse httpResponse) throws OXException, HttpResponseException {
            final StatusLine statusLine = httpResponse.getStatusLine();
            final int statusCode = statusLine.getStatusCode();
            if (statusCode < 200 || statusCode >= 300) {
                if (404 == statusCode) {
                    throw GuardApiExceptionCodes.NOT_FOUND_SIMPLE.create();
                }
                String reason;
                try {
                    JSONObject jsonObject = null;
                    String jsonBody = EntityUtils.toString(httpResponse.getEntity(), Charsets.UTF_8);
                    if (jsonBody.trim().startsWith("[")) {
                        JSONArray jsonArray = new JSONArray(jsonBody);
                        if (jsonArray.length() > 0) {
                            jsonObject = jsonArray.getJSONObject(0);
                            reason = jsonObject.getString("error_desc");
                        }
                        else {
                            reason = statusLine.getReasonPhrase();
                        }
                    }
                    else {
                        //Old style of error response? remove this?
                        jsonObject = new JSONObject(new InputStreamReader(httpResponse.getEntity().getContent(), Charsets.UTF_8));
                        reason = jsonObject.getJSONObject("error").getString("message");
                    }

                } catch (final Exception e) {
                    reason = statusLine.getReasonPhrase();
                }
                throw new HttpResponseException(statusCode, reason);
            }
        }
    };

    /** The status code policy; accepting greater than/equal to <code>200</code> and lower than <code>300</code> while ignoring <code>404</code> */
    public static final StatusCodePolicy STATUS_CODE_POLICY_IGNORE_NOT_FOUND = new StatusCodePolicy() {

        @Override
        public void handleStatusCode(HttpResponse httpResponse) throws HttpResponseException {
            final StatusLine statusLine = httpResponse.getStatusLine();
            final int statusCode = statusLine.getStatusCode();
            if ((statusCode < 200 || statusCode >= 300) && statusCode != 404) {
                String reason;
                try {
                    JSONObject jsonObject = null;
                    String jsonBody = EntityUtils.toString(httpResponse.getEntity(), Charsets.UTF_8);
                    if (jsonBody.trim().startsWith("[")) {
                        JSONArray jsonArray = new JSONArray(jsonBody);
                        if (jsonArray.length() > 0) {
                            jsonObject = jsonArray.getJSONObject(0);
                            reason = jsonObject.getString("error_desc");
                        }
                        else {
                            reason = statusLine.getReasonPhrase();
                        }
                    }
                    else {
                        //Old style of error response? remove this?
                        jsonObject = new JSONObject(new InputStreamReader(httpResponse.getEntity().getContent(), Charsets.UTF_8));
                        reason = jsonObject.getJSONObject("error").getString("message");
                    }

                } catch (final Exception e) {
                    reason = statusLine.getReasonPhrase();
                }
                throw new HttpResponseException(statusCode, reason);
            }
        }
    };

    // -------------------------------------------------------------------------------------------------------------- //

    private final String authLogin;
    private final String authPassword;
    private final URI uri;
    private final HttpHost targetHost;
    private final BasicHttpContext localcontext;
    private volatile DefaultHttpClient httpClient;

    /**
     * Initializes a new {@link GuardApiImpl}.
     *
     * @param endPoint The end-point
     * @throws OXException If initialization fails
     */
    public GuardApiImpl(String endPoint, ConfigurationService service) throws OXException {
        super();
        final String authLogin = service.getProperty("com.openexchange.rest.services.basic-auth.login");
        final String authPassword = service.getProperty("com.openexchange.rest.services.basic-auth.password");
        if (Strings.isEmpty(authLogin) || Strings.isEmpty(authPassword)) {
            LOGGER.error("Denied initialization due to unset Basic-Auth configuration. Please set properties 'com.openexchange.rest.services.basic-auth.login' and 'com.openexchange.rest.services.basic-auth.password' appropriately.");
            throw ServiceExceptionCode.absentService(ConfigurationService.class);
        }
        this.authLogin = authLogin.trim();
        this.authPassword = authPassword.trim();

        final String sUrl = endPoint;
        try {
            uri = new URI(sUrl);
            final HttpHost targetHost = new HttpHost(uri.getHost(), uri.getPort(), uri.getScheme());
            this.targetHost = targetHost;
        } catch (final URISyntaxException e) {
            throw GuardApiExceptionCodes.INVALID_GUARD_URL.create(null == sUrl ? "<empty>" : sUrl);
        }

        // Generate BASIC scheme object and stick it to the local execution context
        final BasicHttpContext context = new BasicHttpContext();
        final BasicScheme basicAuth = new BasicScheme();
        context.setAttribute("preemptive-auth", basicAuth);
        this.localcontext = context;
    }

    private DefaultHttpClient getHttpClient() {
        DefaultHttpClient tmp = httpClient;
        if (null == tmp) {
            synchronized (this) {
                tmp = httpClient;
                if (null == tmp) {
                    tmp = newHttpClient(null);
                    httpClient = tmp;
                }
            }
        }
        return tmp;
    }

    private DefaultHttpClient newHttpClient(String userAgent) {
        ClientConfig clientConfig = ClientConfig.newInstance()
            .setUserAgent(null == userAgent ? "OX Guard Http Client v" + Version.getInstance().getVersionString() : userAgent)
            .setMaxTotalConnections(100)
            .setMaxConnectionsPerRoute(100)
            .setConnectionTimeout(5000)
            .setSocketReadTimeout(15000);

        DefaultHttpClient tmp = HttpClients.getHttpClient(clientConfig);

        final Credentials credentials = new UsernamePasswordCredentials(authLogin, authPassword);
        tmp.getCredentialsProvider().setCredentials(AuthScope.ANY, credentials);

        // Add "preemptive-auth" as the first request interceptor
        tmp.addRequestInterceptor(new PreemptiveAuth(), 0);
        return tmp;
    }

    /**
     * Shuts-down this Guard API instance.
     */
    public void shutDown() {
        final DefaultHttpClient tmp = httpClient;
        if (null != tmp) {
            HttpClients.shutDown(tmp);
            httpClient = null;
        }
    }

    /**
     * Processes the specified MIME message.
     *
     * @param mimeMessage The MIME message to process
     * @param parameters The request parameters
     * @return The processed MIME message
     * @throws OXException If processing the MIME message fails
     */
    public MimeMessage processMimeMessage(final MimeMessage mimeMessage, Map<String, String> parameters) throws OXException {
        HttpPost request = null;
        InputStream msgSrc = null;
        try {
            request = new HttpPost(buildUri(toQueryString(parameters)));
            final InputStream data = MimeMessageUtility.getStreamFromPart(mimeMessage);

            final String boundary = UUIDs.getUnformattedStringFromRandom();
            final MultipartEntity multipartEntity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, boundary, Charsets.UTF_8) {

                @Override
                public Header getContentType() {
                    return new BasicHeader(HTTP.CONTENT_TYPE, generateContentType(boundary, null));
                }
            };
            multipartEntity.addPart("file", new InputStreamBody(data, "message/rfc882", "mail.eml"));
            request.setEntity(multipartEntity);

            msgSrc = handleHttpResponse(execute(request, getHttpClient()), InputStream.class);

            final ThresholdFileHolder sink = new ThresholdFileHolder();
            boolean closeSink = true;
            try {
                sink.write(msgSrc);
                msgSrc = null;

                final File tempFile = sink.getTempFile();
                MimeMessage tmp;
                if (null == tempFile) {
                    tmp = new MimeMessage(MimeDefaultSession.getDefaultSession(), sink.getStream()) {

                        @Override
                        public Date getReceivedDate() throws MessagingException {
                            Date receivedDate = mimeMessage.getReceivedDate();
                            return null == receivedDate ? super.getReceivedDate() : receivedDate;
                        }
                    };
                } else {
                    final FileBackedMimeMessage fbm = new FileBackedMimeMessage(MimeDefaultSession.getDefaultSession(), tempFile, mimeMessage.getReceivedDate());
                    tmp = fbm;
                }
                closeSink = false;
                return tmp;
            } catch (final MessagingException e) {
                throw MimeMailException.handleMessagingException(e);
            } finally {
                if (closeSink) {
                    sink.close();
                }
            }
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            Streams.close(msgSrc);
            reset(request);
        }
    }

    @Override
    public <R> R doCallGet(Map<String, String> parameters, Class<? extends R> clazz) throws OXException {
        HttpGet request = null;
        try {
            final URI uri = buildUri(toQueryString(parameters));
            request = new HttpGet(uri);

            LOGGER.debug("Executing GET using \"{}\"", uri);
            return handleHttpResponse(execute(request, getHttpClient()), clazz);
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(request);
        }
    }

//  Prepared for US GUARD-56
//    @Override
//    public <R> R doCallPost(Map<String, String> parameters, Map<String, String> bodyParameters, Class<? extends R> clazz) throws OXException {
//        HttpPost request = null;
//        try {
//            final URI uri = buildUri(toQueryString(parameters));
//            request = new HttpPost(uri);
//
//            List<NameValuePair> urlEncodedParameters = toQueryString(bodyParameters);
//            request.setEntity(new UrlEncodedFormEntity(urlEncodedParameters));
//
//            LOGGER.debug("Executing POST using \"{}\"", uri);
//            return handleHttpResponse(execute(request, getHttpClient()), clazz);
//        } catch (final HttpResponseException e) {
//            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
//                // Authentication failed -- recreate token
//                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
//            }
//            throw handleHttpResponseError(null, e);
//        } catch (final IOException e) {
//            throw handleIOError(e);
//        } catch (final RuntimeException e) {
//            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
//        } finally {
//            reset(request);
//        }
//    }

    @Override
    public <R> R doCallPut(Map<String, String> parameters, JSONValue jsonBody, Class<? extends R> clazz) throws OXException {
        HttpPut request = null;
        try {
            final URI uri = buildUri(toQueryString(parameters));
            request = new HttpPut(uri);
            request.setEntity(asHttpEntity(jsonBody));

            LOGGER.debug("Executing PUT using \"{}\" with body \"{}\"", uri, jsonBody);
            return handleHttpResponse(execute(request, getHttpClient()), clazz);
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final JSONException e) {
            throw GuardApiExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(request);
        }
    }

    @Override
    public InputStream requestResource(Map<String, String> parameters) throws OXException {
        HttpGet request = null;
        try {
            final URI uri = buildUri(toQueryString(parameters));
            request = new HttpGet(uri);

            LOGGER.debug("Executing GET using \"{}\"", uri);
            final InputStream data = handleHttpResponse(execute(request, getHttpClient()), InputStream.class);
            final ResourceReleasingInputStream in = new ResourceReleasingInputStream(data, request, null);
            request = null; // Avoid preliminary reset
            return in;
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(request);
        }
    }
    
    @Override
    public InputStream processResource(Map<String, String> parameters, InputStream resource, String contentType, String name) throws OXException {
        HttpPost request = null;
        InputStream processedResource = null;
        try {
            request = new HttpPost(buildUri(toQueryString(parameters)));

            final String boundary = UUIDs.getUnformattedStringFromRandom();
            MultipartEntity multipartEntity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, boundary, Charsets.UTF_8) {

                @Override
                public Header getContentType() {
                    return new BasicHeader(HTTP.CONTENT_TYPE, generateContentType(boundary, null));
                }
            };
            String mimeType = null == contentType ? "application/octet-stream" : contentType;
            multipartEntity.addPart("file", new InputStreamBody(resource, mimeType, null == name ? "file.dat" : name));
            request.setEntity(multipartEntity);

            LOGGER.debug("Executing POST using \"{}\" with binary content of type \"{}\"", uri, mimeType);
            processedResource = handleHttpResponse(execute(request, getHttpClient()), InputStream.class);
            ResourceReleasingInputStream in = new ResourceReleasingInputStream(processedResource, request, null);
            processedResource = null; // Avoid preliminary close
            request = null; // Avoid preliminary reset
            return in;
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            Streams.close(resource, processedResource);
            reset(request);
        }
    }

//    Prepared for US GUARD-56
//    @Override
//    public InputStream processResources(Map<String, String> parameters, Map<String, InputStream> resources, String contentType) throws OXException {
//        HttpPost request = null;
//        InputStream processedResource = null;
//        try {
//            request = new HttpPost(buildUri(toQueryString(parameters)));
//
//            final String boundary = UUIDs.getUnformattedStringFromRandom();
//            MultipartEntity multipartEntity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, boundary, Charsets.UTF_8) {
//
//                @Override
//                public Header getContentType() {
//                    return new BasicHeader(HTTP.CONTENT_TYPE, generateContentType(boundary, null));
//                }
//            };
//            String mimeType = null == contentType ? "application/octet-stream" : contentType;
//            for (Map.Entry<String, InputStream> resource : resources.entrySet()) {
//                multipartEntity.addPart(resource.getKey(), new InputStreamBody(resource.getValue(), mimeType, resource.getKey()));
//            }
//            request.setEntity(multipartEntity);
//
//            LOGGER.debug("Executing POST using \"{}\" with binary content of type \"{}\"", uri, mimeType);
//            processedResource = handleHttpResponse(execute(request, getHttpClient()), InputStream.class);
//            ResourceReleasingInputStream in = new ResourceReleasingInputStream(processedResource, request, null);
//            processedResource = null; // Avoid preliminary close
//            request = null; // Avoid preliminary reset
//            return in;
//        } catch (final HttpResponseException e) {
//            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
//                // Authentication failed -- recreate token
//                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
//            }
//            throw handleHttpResponseError(null, e);
//        } catch (final IOException e) {
//            throw handleIOError(e);
//        } catch (final RuntimeException e) {
//            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
//        } finally {
//            Streams.close(resources.values());
//            Streams.close(processedResource);
//            reset(request);
//        }
//    }

    @Override
    public <R> R doCallSessionSensitiveGet(Map<String, String> parameters, Class<? extends R> clazz, Session session, List<Cookie> cookies, List<com.openexchange.guard.api.Header> headers) throws OXException {
        DefaultHttpClient httpClient = null;
        HttpGet request = null;
        try {
            List<NameValuePair> queryString = toQueryString(parameters);
            if (null == queryString) {
                queryString = new ArrayList<NameValuePair>(1);
            }
            queryString.add(new BasicNameValuePair(AJAXServlet.PARAMETER_SESSION, session.getSessionID()));

            final URI uri = buildUri(queryString);
            request = new HttpGet(uri);

            if (null != headers && !headers.isEmpty()) {
                for (final com.openexchange.guard.api.Header header : headers) {
                    request.setHeader(header.getName(), header.getValue());
                }
            }

            // Build HTTP client
            httpClient = newHttpClient(checkForUserAgent(headers));

            CookieStore cookieStore = null;
            if (null != cookies && !cookies.isEmpty()) {
                cookieStore = new BasicCookieStore();

                final StringBuilder sCookies = new StringBuilder(64);
                boolean first = true;
                for (final Cookie cookie : cookies) {
                    final BasicClientCookie clientCookie = new BasicClientCookie(cookie.getName(), cookie.getValue());
                    clientCookie.setPath("/");
                    cookieStore.addCookie(clientCookie);

                    if (first) {
                        first = false;
                    } else {
                        sCookies.append("; ");
                    }
                    sCookies.append(cookie.getName()).append('=').append(cookie.getValue());
                }
                httpClient.setCookieStore(cookieStore);

                request.getParams().setParameter(ClientPNames.COOKIE_POLICY, CookiePolicy.IGNORE_COOKIES);
                request.setHeader("Cookie", sCookies.toString());
            }

            final BasicHttpContext context = new BasicHttpContext();
            final BasicScheme basicAuth = new BasicScheme();
            context.setAttribute("preemptive-auth", basicAuth);
            if (null != cookieStore) {
                context.setAttribute(ClientContext.COOKIE_STORE, cookieStore);
            }

            LOGGER.debug("Executing GET using \"{}\" including cookies \"{}\" and headers \"{}\"", uri, prettPrint(cookies), prettyPrint(headers));
            final R retval = handleHttpResponse(execute(request, httpClient, context), clazz);
            if (retval instanceof InputStream) {
                final ResourceReleasingInputStream in = new ResourceReleasingInputStream((InputStream) retval, request, httpClient);
                request = null; // Avoid preliminary reset
                httpClient = null; // Avoid preliminary closing
                return (R) in;
            }
            return retval;
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(request);
            HttpClients.shutDown(httpClient);
        }
    }

    @Override
    public InputStream processSessionSensitiveResource(Map<String, String> parameters, InputStream resource, String contentType, String name, Session session, List<Cookie> cookies, List<com.openexchange.guard.api.Header> headers) throws OXException {
        DefaultHttpClient httpClient = null;
        HttpPost request = null;
        InputStream processedResource = null;
        try {
            List<NameValuePair> queryString = toQueryString(parameters);
            if (null == queryString) {
                queryString = new ArrayList<NameValuePair>(1);
            }
            queryString.add(new BasicNameValuePair(AJAXServlet.PARAMETER_SESSION, session.getSessionID()));

            URI uri = buildUri(queryString);
            request = new HttpPost(uri);

            if (null != headers && !headers.isEmpty()) {
                for (final com.openexchange.guard.api.Header header : headers) {
                    request.setHeader(header.getName(), header.getValue());
                }
            }

            final String boundary = UUIDs.getUnformattedStringFromRandom();
            MultipartEntity multipartEntity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, boundary, Charsets.UTF_8) {

                @Override
                public Header getContentType() {
                    return new BasicHeader(HTTP.CONTENT_TYPE, generateContentType(boundary, null));
                }
            };
            String mimeType = null == contentType ? "application/octet-stream" : contentType;
            multipartEntity.addPart("file", new InputStreamBody(resource, mimeType, null == name ? "file.dat" : name));
            request.setEntity(multipartEntity);

            // Build HTTP client
            httpClient = newHttpClient(checkForUserAgent(headers));

            CookieStore cookieStore = null;
            if (null != cookies && !cookies.isEmpty()) {
                cookieStore = new BasicCookieStore();

                final StringBuilder sCookies = new StringBuilder(64);
                boolean first = true;
                for (final Cookie cookie : cookies) {
                    final BasicClientCookie clientCookie = new BasicClientCookie(cookie.getName(), cookie.getValue());
                    clientCookie.setPath("/");
                    cookieStore.addCookie(clientCookie);

                    if (first) {
                        first = false;
                    } else {
                        sCookies.append("; ");
                    }
                    sCookies.append(cookie.getName()).append('=').append(cookie.getValue());
                }
                httpClient.setCookieStore(cookieStore);

                request.getParams().setParameter(ClientPNames.COOKIE_POLICY, CookiePolicy.IGNORE_COOKIES);
                request.setHeader("Cookie", sCookies.toString());
            }

            final BasicHttpContext context = new BasicHttpContext();
            final BasicScheme basicAuth = new BasicScheme();
            context.setAttribute("preemptive-auth", basicAuth);
            if (null != cookieStore) {
                context.setAttribute(ClientContext.COOKIE_STORE, cookieStore);
            }

            LOGGER.debug("Executing POST using \"{}\" with binary content of type \"{}\" including cookies \"{}\" and headers \"{}\"", uri, mimeType, prettPrint(cookies), prettyPrint(headers));
            processedResource = handleHttpResponse(execute(request, httpClient), InputStream.class);
            ResourceReleasingInputStream in = new ResourceReleasingInputStream(processedResource, request, httpClient);
            processedResource = null; // Avoid preliminary close
            request = null; // Avoid preliminary reset
            httpClient = null; // Avoid preliminary shut-down
            return in;
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            Streams.close(resource, processedResource);
            reset(request);
            HttpClients.shutDown(httpClient);
        }
    }

    @Override
    public <R> R doCallSessionSensitivePut(Map<String, String> parameters, JSONValue jsonBody, Class<? extends R> clazz, Session session, List<Cookie> cookies, List<com.openexchange.guard.api.Header> headers) throws OXException {
        DefaultHttpClient httpClient = null;
        HttpPut request = null;
        try {
            List<NameValuePair> queryString = toQueryString(parameters);
            if (null == queryString) {
                queryString = new ArrayList<NameValuePair>(1);
            }
            queryString.add(new BasicNameValuePair(AJAXServlet.PARAMETER_SESSION, session.getSessionID()));

            URI uri = buildUri(queryString);
            request = new HttpPut(uri);

            if (null != headers && !headers.isEmpty()) {
                for (final com.openexchange.guard.api.Header header : headers) {
                    request.setHeader(header.getName(), header.getValue());
                }
            }

            // Set request entity
            request.setEntity(asHttpEntity(jsonBody));

            // Build HTTP client
            httpClient = newHttpClient(checkForUserAgent(headers));

            CookieStore cookieStore = null;
            if (null != cookies && !cookies.isEmpty()) {
                cookieStore = new BasicCookieStore();

                final StringBuilder sCookies = new StringBuilder(64);
                boolean first = true;
                for (final Cookie cookie : cookies) {
                    final BasicClientCookie clientCookie = new BasicClientCookie(cookie.getName(), cookie.getValue());
                    clientCookie.setPath("/");
                    cookieStore.addCookie(clientCookie);

                    if (first) {
                        first = false;
                    } else {
                        sCookies.append("; ");
                    }
                    sCookies.append(cookie.getName()).append('=').append(cookie.getValue());
                }
                httpClient.setCookieStore(cookieStore);

                request.getParams().setParameter(ClientPNames.COOKIE_POLICY, CookiePolicy.IGNORE_COOKIES);
                request.setHeader("Cookie", sCookies.toString());
            }

            final BasicHttpContext context = new BasicHttpContext();
            final BasicScheme basicAuth = new BasicScheme();
            context.setAttribute("preemptive-auth", basicAuth);
            if (null != cookieStore) {
                context.setAttribute(ClientContext.COOKIE_STORE, cookieStore);
            }

            LOGGER.debug("Executing PUT using \"{}\" with body \"{}\" including cookies \"{}\" and headers \"{}\"", uri, jsonBody, prettPrint(cookies), prettyPrint(headers));
            final R retval = handleHttpResponse(execute(request, httpClient, context), clazz);
            if (retval instanceof InputStream) {
                final ResourceReleasingInputStream in = new ResourceReleasingInputStream((InputStream) retval, request, httpClient);
                request = null; // Avoid preliminary reset
                httpClient = null; // Avoid preliminary closing
                return (R) in;
            }
            return retval;
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final JSONException e) {
            throw GuardApiExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(request);
            HttpClients.shutDown(httpClient);
        }
    }

    @Override
    public InputStream requestSessionSensitiveResource(Map<String, String> parameters, Session session, List<Cookie> cookies, List<com.openexchange.guard.api.Header> headers) throws OXException {
        DefaultHttpClient httpClient = null;
        HttpGet request = null;
        try {
            List<NameValuePair> queryString = toQueryString(parameters);
            if (null == queryString) {
                queryString = new ArrayList<NameValuePair>(1);
            }
            queryString.add(new BasicNameValuePair(AJAXServlet.PARAMETER_SESSION, session.getSessionID()));

            final URI uri = buildUri(queryString);
            //request = new HttpGet(uri.toASCIIString().replace("+", "%2b"));
            request = new HttpGet(uri);

            if (null != headers && !headers.isEmpty()) {
                for (final com.openexchange.guard.api.Header header : headers) {
                    request.setHeader(header.getName(), header.getValue());
                }
            }

            // Build HTTP client
            httpClient = newHttpClient(checkForUserAgent(headers));

            CookieStore cookieStore = null;
            if (null != cookies && !cookies.isEmpty()) {
                cookieStore = new BasicCookieStore();

                final StringBuilder sCookies = new StringBuilder(64);
                boolean first = true;
                for (final Cookie cookie : cookies) {
                    final BasicClientCookie clientCookie = new BasicClientCookie(cookie.getName(), cookie.getValue());
                    clientCookie.setPath("/");
                    cookieStore.addCookie(clientCookie);

                    if (first) {
                        first = false;
                    } else {
                        sCookies.append("; ");
                    }
                    sCookies.append(cookie.getName()).append('=').append(cookie.getValue());
                }
                httpClient.setCookieStore(cookieStore);

                request.getParams().setParameter(ClientPNames.COOKIE_POLICY, CookiePolicy.IGNORE_COOKIES);
                request.setHeader("Cookie", sCookies.toString());
            }

            final BasicHttpContext context = new BasicHttpContext();
            final BasicScheme basicAuth = new BasicScheme();
            context.setAttribute("preemptive-auth", basicAuth);
            if (null != cookieStore) {
                context.setAttribute(ClientContext.COOKIE_STORE, cookieStore);
            }

            LOGGER.debug("Executing GET using \"{}\" including cookies \"{}\" and headers \"{}\"", uri, prettPrint(cookies), prettyPrint(headers));
            final InputStream data = handleHttpResponse(execute(request, httpClient, context), InputStream.class);
            final ResourceReleasingInputStream in = new ResourceReleasingInputStream(data, request, httpClient);
            request = null; // Avoid preliminary reset
            httpClient = null; // Avoid preliminary closing
            return in;
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed -- recreate token
                throw GuardApiExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e);
        } catch (final RuntimeException e) {
            throw GuardApiExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(request);
            HttpClients.shutDown(httpClient);
        }
    }

    // ----------------------------------------------------------------------------------------------------------- //

    /**
     * Builds the URI from given arguments
     *
     * @param queryString The query string parameters
     * @return The built URI string
     * @throws IllegalArgumentException If the given string violates RFC 2396
     */
    protected URI buildUri(List<NameValuePair> queryString) {
        try {
            final URIBuilder builder = new URIBuilder();
            builder.setScheme(uri.getScheme()).
                setHost(uri.getHost()).
                setPort(uri.getPort()).
                setPath(uri.getPath()).
                setQuery(null == queryString ? null : URLEncodedUtils.format(queryString, "UTF-8"));
            return builder.build();
        } catch (final URISyntaxException x) {
            throw new IllegalArgumentException("Failed to build URI", x);
        }
    }

    /**
     * Turns specified JSON value into an appropriate HTTP entity.
     *
     * @param jValue The JSON value
     * @return The HTTP entity
     * @throws JSONException If a JSON error occurs
     * @throws IOException If an I/O error occurs
     */
    protected InputStreamEntity asHttpEntity(JSONValue jValue) throws JSONException, IOException {
        if (null == jValue) {
            return null;
        }

        ThresholdFileHolder sink = null;
        boolean error = true;
        try {
            sink = new ThresholdFileHolder();
            final OutputStreamWriter osw = new OutputStreamWriter(sink.asOutputStream(), Charsets.UTF_8);
            jValue.write(osw);
            osw.flush();
            final InputStreamEntity entity = new InputStreamEntity(sink.getStream(), sink.getLength(), ContentType.APPLICATION_JSON);
            error = false;
            return entity;
        } catch (final OXException e) {
            final Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw new IOException(null == cause ? e : cause);
        } finally {
            if (error && null != sink) {
                Streams.close(sink);
            }
        }
    }

    /**
     * Gets the appropriate query string for given parameters
     *
     * @param parameters The parameters
     * @return The query string
     */
    protected List<NameValuePair> toQueryString(Map<String, String> parameters) {
        if (null == parameters || parameters.isEmpty()) {
            return null;
        }
        final List<NameValuePair> l = new LinkedList<NameValuePair>();
        for (final Map.Entry<String, String> e : parameters.entrySet()) {
            l.add(new BasicNameValuePair(e.getKey(), e.getValue()));
        }
        return l;
    }

    /**
     * Executes specified HTTP method/request using given HTTP client instance.
     *
     * @param method The method/request to execute
     * @param httpClient The HTTP client to use
     * @return The HTTP response
     * @throws ClientProtocolException If client protocol error occurs
     * @throws IOException If an I/O error occurs
     */
    protected HttpResponse execute(HttpRequestBase method, DefaultHttpClient httpClient) throws ClientProtocolException, IOException {
        return execute(method, httpClient, localcontext);
    }

    /**
     * Executes specified HTTP method/request using given HTTP client instance.
     *
     * @param method The method/request to execute
     * @param httpClient The HTTP client to use
     * @param context The context
     * @return The HTTP response
     * @throws ClientProtocolException If client protocol error occurs
     * @throws IOException If an I/O error occurs
     */
    protected HttpResponse execute(HttpRequestBase method, DefaultHttpClient httpClient, BasicHttpContext context) throws ClientProtocolException, IOException {
        return httpClient.execute(targetHost, method, context);
    }

    /**
     * Resets given HTTP request
     *
     * @param request The HTTP request
     */
    protected static void reset(HttpRequestBase request) {
        if (null != request) {
            try {
                request.reset();
            } catch (Exception e) {
                // Ignore
            }
        }
    }

    /**
     * Closes the supplied response resources silently.
     *
     * @param response The HTTP response to consume and close
     */
    protected static void close(HttpResponse response) {
        if (null != response) {
            HttpEntity entity = response.getEntity();
            if (null != entity) {
                try {
                    EntityUtils.consumeQuietly(entity);
                } catch (Exception e) {
                    // Ignore
                }
            }
        }
    }

    /**
     * Closes the supplied HTTP request / response resources silently.
     *
     * @param request The HTTP request to reset
     * @param response The HTTP response to consume and close
     */
    protected static void closeAndReset(HttpRequestBase request, HttpResponse response) {
        close(response);
        reset(request);
    }

    /**
     * Handles given HTTP response while expecting <code>200 (Ok)</code> status code.
     *
     * @param httpResponse The HTTP response
     * @param clazz The class of the result object
     * @return The result object
     * @throws OXException If an Open-Xchange error occurs
     * @throws ClientProtocolException If a client protocol error occurs
     * @throws IOException If an I/O error occurs
     */
    protected <R> R handleHttpResponse(HttpResponse httpResponse, Class<R> clazz) throws OXException, ClientProtocolException, IOException {
        return handleHttpResponse(httpResponse, STATUS_CODE_POLICY_DEFAULT, clazz);
    }

    /**
     * Handles given HTTP response while expecting given status code.
     *
     * @param httpResponse The HTTP response
     * @param policy The status code policy to obey
     * @param clazz The class of the result object
     * @return The result object
     * @throws OXException If an Open-Xchange error occurs
     * @throws ClientProtocolException If a client protocol error occurs
     * @throws IOException If an I/O error occurs
     * @throws IllegalStateException If content stream cannot be created
     */
    protected <R> R handleHttpResponse(HttpResponse httpResponse, StatusCodePolicy policy, Class<R> clazz) throws OXException, ClientProtocolException, IOException {
        boolean close = true;
        try {
            policy.handleStatusCode(httpResponse);
            // OK, continue
            if (Void.class.equals(clazz)) {
                return null;
            }
            if (InputStream.class.equals(clazz)) {
                R retval = (R) httpResponse.getEntity().getContent();
                close = false;
                return retval;
            }
            try {
                InputStream in = httpResponse.getEntity().getContent();
                R retval = (R) new JSONObject(new InputStreamReader(in, Charsets.UTF_8));
                close = false;
                return retval;
            } catch (final JSONException e) {
                throw GuardApiExceptionCodes.JSON_ERROR.create(e, e.getMessage());
            }
        } finally {
            if (close) {
                close(httpResponse);
            }
        }
    }

    /**
     * Handles given I/O error.
     *
     * @param e The I/O error
     * @return The resulting exception
     */
    protected OXException handleIOError(IOException e) {
        LOGGER.warn("An IOError occurred while communicating with the guard server: {}", e.getMessage());
        final Throwable cause = e.getCause();
        if (cause instanceof AuthenticationException) {
            return GuardApiExceptionCodes.AUTH_ERROR.create(cause, cause.getMessage());
        }
        return GuardApiExceptionCodes.IO_ERROR.create(e, e.getMessage());
    }

    /** Status code (401) indicating that the request requires HTTP authentication. */
    private static final int SC_UNAUTHORIZED = 401;

    /** Status code (404) indicating that the requested resource is not available. */
    private static final int SC_NOT_FOUND = 404;

    /**
     * Handles given HTTP response error.
     *
     * @param identifier The optional identifier for associated Microsoft OneDrive resource
     * @param e The HTTP error
     * @return The resulting exception
     */
    protected OXException handleHttpResponseError(String identifier, HttpResponseException e) {
        LOGGER.warn("An error occurred while communicating with the guard server: {}", e.getMessage());
        if (null != identifier && SC_NOT_FOUND == e.getStatusCode()) {
            return GuardApiExceptionCodes.NOT_FOUND.create(e, identifier);
        }
        if (SC_UNAUTHORIZED == e.getStatusCode()) {
            return GuardApiExceptionCodes.AUTH_ERROR.create();
        }
        return GuardApiExceptionCodes.GUARD_SERVER_ERROR.create(e, Integer.valueOf(e.getStatusCode()), e.getMessage());
    }

    static class PreemptiveAuth implements HttpRequestInterceptor {

        @Override
        public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {

            final AuthState authState = (AuthState) context.getAttribute(
                ClientContext.TARGET_AUTH_STATE);

            // If no auth scheme avaialble yet, try to initialize it preemptively
            if (authState.getAuthScheme() == null) {
                final AuthScheme authScheme = (AuthScheme) context.getAttribute(
                    "preemptive-auth");
                final CredentialsProvider credsProvider = (CredentialsProvider) context.getAttribute(
                    ClientContext.CREDS_PROVIDER);
                final HttpHost targetHost = (HttpHost) context.getAttribute(
                    ExecutionContext.HTTP_TARGET_HOST);
                if (authScheme != null) {
                    final Credentials creds = credsProvider.getCredentials(
                        new AuthScope(
                            targetHost.getHostName(),
                            targetHost.getPort()));
                    if (creds == null) {
                        throw new HttpException("No credentials for preemptive authentication");
                    }
                    authState.setAuthScheme(authScheme);
                    authState.setCredentials(creds);
                }
            }
        }
    }

    private static String checkForUserAgent(List<com.openexchange.guard.api.Header> headers) {
        if (null == headers || headers.isEmpty()) {
            return null;
        }
        for (final com.openexchange.guard.api.Header header : headers) {
            if ("User-Agent".equalsIgnoreCase(header.getName())) {
                return header.getValue();
            }
        }
        return null;
    }

    /**
     * An <code>InputStream</code> that takes care of shutting-down associated <code>HttpClient</code> instance when closed.
     */
    private static class ResourceReleasingInputStream extends InputStream {

        private final InputStream in;
        private final DefaultHttpClient httpClient;
        private final HttpRequestBase request;

        ResourceReleasingInputStream(InputStream in, HttpRequestBase request, DefaultHttpClient httpClient) {
            super();
            this.in = in;
            this.request = request;
            this.httpClient = httpClient;
        }

        @Override
        public int read() throws IOException {
            return in.read();
        }

        @Override
        public int read(byte[] b) throws IOException {
            return in.read(b);
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return in.read(b, off, len);
        }

        @Override
        public long skip(long n) throws IOException {
            return in.skip(n);
        }

        @Override
        public String toString() {
            return in.toString();
        }

        @Override
        public int available() throws IOException {
            return in.available();
        }

        @Override
        public void close() throws IOException {
            try {
                in.close();
            } finally {
                GuardApiImpl.reset(request);
                HttpClients.shutDown(httpClient);
            }
        }

        @Override
        public void mark(int readlimit) {
            in.mark(readlimit);
        }

        @Override
        public void reset() throws IOException {
            in.reset();
        }

        @Override
        public boolean markSupported() {
            return in.markSupported();
        }
    } // End of class ClientReleasingInputStream

    private static Object prettyPrint(final List<com.openexchange.guard.api.Header> headers) {
        return new Object() {

            @Override
            public String toString() {
                if (null == headers || headers.isEmpty()) {
                    return "none";
                }

                final Iterator<com.openexchange.guard.api.Header> iter = headers.iterator();
                final StringBuilder sb = new StringBuilder(headers.size() << 2);

                com.openexchange.guard.api.Header hdr = iter.next();
                sb.append(hdr.getName()).append('=').append(hdr.getName());
                while (iter.hasNext()) {
                    hdr = iter.next();
                    sb.append(", ").append(hdr.getName()).append('=').append(hdr.getName());
                }
                return sb.toString();
            }
        };
    }

    private static Object prettPrint(final List<Cookie> cookies) {
        return new Object() {

            @Override
            public String toString() {
                if (null == cookies || cookies.isEmpty()) {
                    return "none";
                }

                final Iterator<Cookie> iter = cookies.iterator();
                final StringBuilder sb = new StringBuilder(cookies.size() << 2);

                Cookie cookie = iter.next();
                sb.append(cookie.getName()).append('=').append(cookie.getName());
                while (iter.hasNext()) {
                    cookie = iter.next();
                    sb.append(", ").append(cookie.getName()).append('=').append(cookie.getName());
                }
                return sb.toString();
            }
        };
    }

}
