/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

/**
 * {@link AbstractGuardAccess} - Access to OX Guard API.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 */
public abstract class AbstractGuardAccess {

    public static final String GUARDADMIN_ENDPOINT = "guardadmin";
    public static final String PGPCORE_ENDPOINT = "oxguard/pgpcore";

    private static final AtomicReference<Map<String, GuardApiImpl>> API_REF = new AtomicReference<Map<String, GuardApiImpl>>();

    static {
        API_REF.set(new HashMap<String, GuardApiImpl>());
    }

    /**
     * Adds a {@link GuardApiImpl} reference with a given name
     *
     * @param endpointName The name
     * @param guardApi The instance to add
     */
    public static void addGuardApi(String endpointName, GuardApiImpl guardApi) {
        API_REF.get().put(endpointName, guardApi);
    }

    /**
     * Unsets the {@link GuardApiImpl} reference
     *
     * @param guardApi The instance to all guardApis
     */
    public static Map<String, GuardApiImpl> unsetGuardApi() {
        Map<String, GuardApiImpl> guardApis;
        do {
            guardApis = API_REF.get();
            if (null == guardApis) {
                return null;
            }
        } while (!API_REF.compareAndSet(guardApis, null));
        return guardApis;
    }

    /**
     * Initializes a new {@link AbstractGuardAccess}.
     */
    protected AbstractGuardAccess() {
        super();
    }

    /**
     * Gets the Guard API access for the given endpoint name.
     *
     * @return The Guard API access
     */
    protected static GuardApiImpl getGuardApi(String endpointName) {
        return API_REF.get().get(endpointName);
    }
}
