/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.file.storage;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.api.GuardApi;
import com.openexchange.guard.internal.authentication.AuthenticationToken;

/**
 * {@link GuardDecryptedInputStream} decrypts data using OX Guard
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.2
 */
public class GuardDecryptedInputStream extends InputStream {

    private final GuardApi guardApi;
    private final AuthenticationToken authToken;
    private final InputStream encrypted;
    private InputStream guardStream = null;

    /**
     * Initializes a new {@link GuardDecryptedInputStream}.
     * 
     * @param guardApi The guard enpoint to use
     * @param authToken The authentication token for accessing Guard
     * @param encrypted The data to decrypt
     */
    public GuardDecryptedInputStream(GuardApi guardApi, AuthenticationToken authToken, InputStream encrypted) {
        this.guardApi = guardApi;
        this.authToken = authToken;
        this.encrypted = encrypted;
    }

    /**
     * Creates the InputStream which reads the decrypted data from Guard
     * 
     * @return An InputStream containing the decrypted data obtained from Guard
     * @throws OXException
     */
    private InputStream getGuardStream() throws OXException {
        ByteArrayInputStream jsonStream = null;
        JSONObject json = new JSONObject();
        JSONObject user = new JSONObject();
        user.putSafe("session", authToken.getGuardSessionId());
        user.putSafe("auth", authToken.getValue());
        json.putSafe("user", user);
        jsonStream = new ByteArrayInputStream(json.toString().getBytes());

        LinkedHashMap<String, InputStream> multipartParameters = new LinkedHashMap<String, InputStream>();
        multipartParameters.put("json", jsonStream);
        multipartParameters.put("data", encrypted);
        
        throw new RuntimeException("Not implemented yet");
//        return guardApi.processResources(GuardApis.mapFor("action", "decrypt", "respondWithJSON", "true" /* forcing guard to render errors in JSON */),
//            multipartParameters, null);
    }

    private synchronized InputStream guardStreamAccess() throws OXException {
        if (guardStream == null) {
            guardStream = getGuardStream();
        }
        return guardStream;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.io.InputStream#available()
     */
    @Override
    public int available() throws IOException {
        return guardStream != null ? guardStream.available() : 0;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.io.InputStream#close()
     */
    @Override
    public void close() throws IOException {
        if (guardStream != null) {
            guardStream.close();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.io.InputStream#markSupported()
     */
    @Override
    public boolean markSupported() {
        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.io.InputStream#read()
     */
    @Override
    public int read() throws IOException {
        try {
            return guardStreamAccess().read();
        } catch (OXException e) {
            throw new IOException(e.getMessage(), e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.io.InputStream#read(byte[])
     */
    @Override
    public int read(byte[] b) throws IOException {
        try {
            return guardStreamAccess().read(b);
        } catch (OXException e) {
            throw new IOException(e.getMessage(), e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.io.InputStream#read(byte[], int, int)
     */
    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        try {
            return guardStreamAccess().read(b, off, len);
        } catch (OXException e) {
            throw new IOException(e.getMessage(), e);
        }
    }
}
