
package com.openexchange.guard.support;

import java.net.URI;
import java.net.URISyntaxException;
import org.apache.http.client.utils.URIBuilder;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keys.dao.DeletedKey;
import com.openexchange.guard.keys.storage.DeletedKeysStorage;
import com.openexchange.guard.osgi.Services;

/**
 * Exposes a private key of a deleted user for downloading
 *
 * When user's get deleted we backup their keys for a certain amount of time.
 * In that time-span an administrator can enable download functionality for
 * a user to ensure the user can get his key, even if his account was already
 * deleted.
 */
public class DeletedKeyExposer {

    /**
     * Internal method to create an URI for accessing an exposed key
     *
     * @param deletedKey The deleted key to create an URI for
     * @return An URI from which a user can download an exposed key
     * @throws OXException
     */
    private URI buildGetExposedKeyUri(DeletedKey deletedKey) throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        //Reading the whole reader path from the configuration
        String fullReaderPath = configService.getProperty(GuardProperty.externalReaderPath);
        if (!fullReaderPath.contains("/")) {
            throw GuardCoreExceptionCodes.CONFIGURATION_ERROR.create("externalReaderPath", "Missing \"/\"");
        }

        //extracting the host name
        int firstSlashIndex = fullReaderPath.indexOf("/");
        String hostName = fullReaderPath.substring(0, firstSlashIndex);
        if (hostName.isEmpty()) {
            throw GuardCoreExceptionCodes.CONFIGURATION_ERROR.create("externalReaderPath", "Missing hostname");
        }

        //extracting the path
        String readerPath = fullReaderPath.substring(firstSlashIndex);
        if (hostName.isEmpty()) {
            throw GuardCoreExceptionCodes.CONFIGURATION_ERROR.create("externalReaderPath", "Missing path");
        }

        String scheme = "http" + (configService.getBooleanProperty(GuardProperty.backendSSL) ? "s" : "");
        URIBuilder uri = new URIBuilder().setScheme(scheme).setHost(hostName).setPath(readerPath).setParameter("action", "getKey").setParameter("userid", String.valueOf(deletedKey.getUserId())).setParameter("cid", String.valueOf(deletedKey.getCid())).setParameter("token", String.valueOf(deletedKey.getSalt()));
        try {
            return uri.build();
        } catch (URISyntaxException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, "Could not construct redirect location for URI '" + uri.toString() + "'");
        }
    }

    /**
     * Exposes all deleted keys for the given user
     *
     * @param email the user's email
     * @param cid the context id
     * @throws OXException On error during exposing the key
     * @return an URI for accessing deleted but exposed keys for the user
     */
    public URI exposeKey(String email, int cid) throws OXException {
        DeletedKeysStorage deletedKeyStorage = Services.getService(DeletedKeysStorage.class);

        //Checking if the user has deleted keys and get the salt of the first found key
        DeletedKey firstDeletedKey = deletedKeyStorage.getFirstForEmail(email, cid);
        if (firstDeletedKey == null) {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_MAIL_ERROR.create(email, cid);
        }

        //Creating URI for accessing deleted,exposed keys
        // example:
        //http://192.168.0.69:8337/appsuite/api/oxguard/reader/reader.html?action=getkey&userid=4&cid=1&token=mdl%2B1UEEycWnub%2BHb8yLxmhc3Yg%3D
        URI uri = buildGetExposedKeyUri(firstDeletedKey);

        //Exposing the keys of the given email in order to be available using the URI
        deletedKeyStorage.setExposedForEmail(email, cid);

        return uri;
    }
}
