/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.nio.charset.Charset;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.io.IOUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.ContentBody;
import com.google.gson.JsonObject;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.decryption.GuardEMailDecryptionService;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.pgp.decryption.impl.GuardEMailDecryptionServiceImpl;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.AuthenticationDataExtractor;
import com.openexchange.guard.servlets.authentication.GuardUserSession;

/**
 * Imports vcard attachment into contacts
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class ImportVCardAction extends GuardServletAction {

    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        OxCookie cookie = new OxCookie(request.getCookies());
        int userid = userSession.getUserId();
        int cid = userSession.getContextId();
        String emailid = ServletUtils.getStringParameter(request, "emailid", true);
        String attachid = ServletUtils.getStringParameter(request, "attach", true);
        String folder = ServletUtils.getStringParameter(request, "folder");
        String filename = ServletUtils.getStringParameter(request, "filename", true);
        String contactFolder = ServletUtils.getStringParameter(request, "contacts", true);
        boolean inline = ServletUtils.getBooleanParameter(request, "inline", false);
        String session = request.getParameter("session");
        String userAgent = request.getHeader("User-Agent");
        String password = new AuthenticationDataExtractor().getPassword(request, cookie);
        if (password == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("Password or auth missing.");
        }

        //Decrypting Vcard attachment
        GuardEMailDecryptionService decryptionService = new GuardEMailDecryptionServiceImpl();
        byte[] vcard = null;
        if (inline) {
            PGPDecryptionResult decryptionResult = decryptionService.decryptInlineAttachment(cookie, session, userAgent, folder, emailid, Integer.parseInt(attachid), userid, cid, password);
            vcard = IOUtils.toByteArray(decryptionResult.getInputStream());
        } else {
            PGPDecryptionResult decryptionResult = decryptionService.decryptMimeAttachmentByName(cookie, session, userAgent, folder, emailid, filename, userid, cid, password);
            vcard = IOUtils.toByteArray(decryptionResult.getInputStream());
        }

        MultipartEntity entity = new MultipartEntity(HttpMultipartMode.BROWSER_COMPATIBLE, null, Charset.forName("UTF-8"));
        ContentType type = ContentType.create("text/vcard" + "\r\nX-Forced-MIME-Type:true");
        ContentBody file = new ByteArrayBody(vcard, type.getMimeType(), filename);
        entity.addPart("file_0", file);

        JsonObject vcardResponse = new Api(cookie, request).importVcard(entity, contactFolder);
        if (vcardResponse != null) {
            ServletUtils.sendHtmlOK(response, vcardResponse.toString());
        } else {
            ServletUtils.sendHtmlOK(response, "OK");
        }
    }
}
