
package com.openexchange.guard.servlets;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.codec.net.URLCodec;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.Category;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.translation.GuardTranslationService;

public class GuardErrorResponseRenderer {

    private static final String REQUEST_PARAM_INCLUDE_STACKTRACE = "includeStackTraceOnError";
    private static final String MULTIPART = "multipart/";
    private static final String CALLBACK = "callback";
    private static final String JSON_ERROR = "error";
    private static final String JSON_ERROR_PARAMS = "error_params";
    private static final String JSON_CATEGORIES = "categories";
    private static final String JSON_CATEGORY = "category";
    private static final String JSON_CODE = "code";
    private static final String JSON_ERROR_ID = "error_id";
    private static final String JSON_ERROR_DESC = "error_desc";
    private static final String JSON_ERROR_STACK = "error_stack";
    private static final Object JSON_STACK_NATIVE_METHOD = "(Native Method)";
    private static final Object JSON_STACK_UNKNOWN_SOURCE = "(Unknown Source)";

    private static final char[] JS_FRAGMENT_PART1 = ("<!DOCTYPE html><html><head>" + "<META http-equiv=\"Content-Type\" " + "content=\"text/html; charset=UTF-8\">" + "<script type=\"text/javascript\">" + "(parent[\"callback_").toCharArray();

    private static final char[] JS_FRAGMENT_PART2 = "\"] || window.opener && window.opener[\"callback_".toCharArray();

    private static final char[] JS_FRAGMENT_PART3 = ")</script></head></html>".toCharArray();

    private static final Pattern PATTERN_QUOTE = Pattern.compile("(^|[^\\\\])\"");
    private static final Pattern PATTERN_CONTROL = Pattern.compile("[\\x00-\\x1F\\x7F]");
    private static final Pattern PATTERN_DSLASH = Pattern.compile("(?://+)");

    /**
     * Internal method to check whether the request contains multipart content or not
     *
     * @param request The request to be evaluated.
     * @return true if the request is multipart, false otherwise
     */
    private final boolean isMultipartContent(HttpServletRequest request) {
        final String contentType = request.getContentType();
        if (contentType == null) {
            return false;
        }
        if (contentType.toLowerCase().startsWith(MULTIPART)) {
            return true;
        }
        return false;
    }

    /**
     * Internal method to check if the client requested HTML response
     *
     * @param request the client's request
     * @return true, if the client requested a HTML response, false otherwise
     */
    private boolean isRespondWithHTML(final HttpServletRequest request) {
        return Boolean.parseBoolean(request.getParameter("respondWithHTML"));
    }

    /**
     * Checks whether the client expects the error object wrapped in HTML/JS
     *
     * @param request The HTTP request to check
     * @return <code>true</code> if a JavaScript call-back is expected; otherwise <code>false</code>
     */
    private boolean expectsJsCallback(HttpServletRequest request) {
        return (isMultipartContent(request) || isRespondWithHTML(request) || request.getParameter(CALLBACK) != null);
    }

    /**
     * Internal method to create a JSON error object
     *
     * @param e the exception to create the error object for
     * @param locale the request's locale
     * @param includeStackTrace true, if a stack trace should be included, false otherwise
     * @return The json error object for the given exception
     */
    protected JsonObject createErrorJson(OXException e, Locale locale, boolean includeStackTrace) {
        JsonObject errorObject = new JsonObject();
        String translatedDisplayMessage = getTranslatedDisplayMessage(e, locale);
        errorObject.addProperty(JSON_ERROR, translatedDisplayMessage);

        JsonArray errorParams = new JsonArray();
        Object[] logArgsArray = e.getLogArgs();
        if (logArgsArray != null) {
            for (Object logArgs : logArgsArray) {
                JsonPrimitive displayArgParam = new JsonPrimitive(logArgs.toString());
                errorParams.add(displayArgParam);
            }
        }
        errorObject.add(JSON_ERROR_PARAMS, errorParams);

        if ((e.getCategories() != null) && (e.getCategories().size() == 1)) {
            errorObject.addProperty(JSON_CATEGORIES, e.getCategory().toString());
        } else {
            JsonArray categoryParams = new JsonArray();
            List<Category> categories = e.getCategories();
            for (Category c : categories) {
                JsonPrimitive categorieParam = new JsonPrimitive(c.toString());
                categoryParams.add(categorieParam);
            }
            errorObject.add(JSON_CATEGORIES, errorParams);
        }

        errorObject.addProperty(JSON_CATEGORY, e.getCategory().toString());
        errorObject.addProperty(JSON_CODE, e.getErrorCode());
        errorObject.addProperty(JSON_ERROR_ID, e.getExceptionId());
        errorObject.addProperty(JSON_ERROR_DESC, e.getSoleMessage());

        JsonArray errorStack = new JsonArray();
        if (includeStackTrace) {
            StackTraceElement[] traceElements = e.getStackTrace();
            if (traceElements != null && traceElements.length > 0) {
                final StringBuilder tmp = new StringBuilder(64);
                for (final StackTraceElement stackTraceElement : traceElements) {
                    tmp.setLength(0);
                    writeElementTo(stackTraceElement, tmp);
                    JsonPrimitive error = new JsonPrimitive(tmp.toString());
                    errorStack.add(error);
                }
            }
        }
        errorObject.add(JSON_ERROR_STACK, errorStack);
        return errorObject;
    }

    private String getTranslatedDisplayMessage(OXException e, Locale locale) {
        String displayMessage = e.getDisplayMessage(locale); // try to get translation from middleware (e. g. for OXExceptionStrings.MESSAGE)
        if (Strings.isEmpty(displayMessage) || displayMessage.equalsIgnoreCase(e.getDisplayMessage(null))) { // try guard translation
            GuardTranslationService translationService = Services.optService(GuardTranslationService.class);
            if (translationService != null) {
                return translationService.getTranslation(displayMessage, locale.getLanguage());
            }
        }
        return displayMessage;
    }

    private static void writeElementTo(final StackTraceElement element, final StringBuilder sb) {
        sb.append(element.getClassName()).append('.').append(element.getMethodName());
        if (element.isNativeMethod()) {
            sb.append(JSON_STACK_NATIVE_METHOD);
        } else {
            final String fileName = element.getFileName();
            if (null == fileName) {
                sb.append(JSON_STACK_UNKNOWN_SOURCE);
            } else {
                sb.append('(').append(fileName);
                final int lineNumber = element.getLineNumber();
                if (lineNumber >= 0) {
                    sb.append(':').append(lineNumber);
                }
                sb.append(')');
            }
        }
    }

    /**
     * Sanitizes specified String input.
     * <ul>
     * <li>Do URL decoding until fully decoded
     * <li>Drop ASCII control characters
     * <li>Escape using HTML entities
     * <li>Replace double slashes with single one
     * </ul>
     *
     * @param sInput The input to sanitize, can be <code>null</code> or empty
     * @return The sanitized input or the original value if <code>null</code> or empty
     */
    private String sanitizeParam(String sInput) {
        if (sInput.isEmpty()) {
            return sInput;
        }

        String s = sInput;

        // Do URL decoding until fully decoded
        {
            int pos;
            while ((pos = s.indexOf('%')) >= 0 && pos < s.length() - 1) {
                try {
                    s = new URLCodec("UTF-8").decode(s);
                } catch (org.apache.commons.codec.DecoderException e) {
                    break;
                }
            }
        }

        // Drop ASCII control characters
        s = PATTERN_CONTROL.matcher(s).replaceAll("");

        // Escape using HTML entities
        s = org.apache.commons.lang.StringEscapeUtils.escapeHtml(s);

        // Replace double slashes with single one
        {
            final Pattern patternDslash = PATTERN_DSLASH;
            Matcher matcher = patternDslash.matcher(s);
            while (matcher.find()) {
                s = matcher.replaceAll("/");
                matcher = patternDslash.matcher(s);
            }
        }

        // Return result
        return s;
    }

    /**
     * Renders the given exception to a response
     *
     * @param request the request
     * @param response the response
     * @param actionName
     * @param e the error to render
     * @throws IOException
     */
    public void renderError(HttpServletRequest request, HttpServletResponse response, String actionName, OXException e) throws IOException {
        boolean includeStackTrace = ServletUtils.getBooleanParameter(request, REQUEST_PARAM_INCLUDE_STACKTRACE);
        JsonObject errorObject = createErrorJson(e, request.getLocale(), includeStackTrace);
        if (expectsJsCallback(request)) {
            String callback = ServletUtils.getStringParameter(request, CALLBACK);
            if (callback != null) {
                if (callback.indexOf('"') >= 0) {
                    callback = PATTERN_QUOTE.matcher(callback).replaceAll("$1\\\\\"");
                    //sanitize callback name, because it is injected dynamically
                    callback = sanitizeParam(callback);
                }
            } else {
                callback = actionName;
            }

            final PrintWriter writer = response.getWriter();
            writer.write(JS_FRAGMENT_PART1);
            writer.write(callback);
            writer.write(JS_FRAGMENT_PART2);
            writer.write(callback);
            writer.write("\"])(");
            writer.write(errorObject.toString());
            writer.write(JS_FRAGMENT_PART3);
        } else {
            JsonArray errorObjects = new JsonArray();
            errorObjects.add(errorObject);
            ServletUtils.sendJsonOK(response, errorObjects);
        }
    }
}
