/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp.keymanagment;

import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgp.PGPPublicHandler;
import com.openexchange.guard.pgp.attributes.PGPAttributeExtractor;
import com.openexchange.guard.ratifier.GuardRatifierService;

/**
 * Provides functionality for accessing user keys
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuardUserKeyServiceImpl implements GuardUserKeyService {

    /**
     * Internal method to get the target shard ID for guests and normal users
     * 
     * @param userId the userId
     * @param contextId the contextId
     * @return The shard ID if the user is a guest or 0 if it a regular user
     * @throws OXException
     */
    public int getShardId(int userId, int contextId) throws OXException {
        int shard = 0;
        if (contextId < 0) {
            GuardShardingService sharding = Services.getService(GuardShardingService.class);
            shard = sharding.getShard(userId, contextId);
        }
        return shard;
    }

    @Override
    public String getAsciiArmoredCurrentPrivateKey(int userId, int contextId, String password) throws OXException {
        int shard = getShardId(userId, contextId);
        GuardKeys keyPair = Services.getService(KeyTableStorage.class).getCurrentKeyForUser(userId, contextId, shard);

        if (keyPair != null) {
            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            return pgpKeyService.exportPGPPrivateKey(keyPair, password);
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_IDS_ERROR.create(userId, contextId);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgp.keymanagment.GuardUserKeyService#getAsciiArmoredCurrentPublicKey(int, int)
     */
    @Override
    public String getAsciiArmoredCurrentPublicKey(int userId, int contextId) throws OXException {
        int shard = getShardId(userId, contextId);
        GuardKeys keyPair = Services.getService(KeyTableStorage.class).getCurrentKeyForUser(userId, contextId, shard);

        if (keyPair != null) {
            return keyPair.exportPGPPublicKeyRing();
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_IDS_ERROR.create(userId, contextId);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgp.keymanagment.GuardUserKeyService#getAsciiArmoredPublicKeyByEmail(java.lang.String)
     */
    @Override
    public String getAsciiArmoredCurrentPublicKeyByEmail(String email) throws OXException {
        GuardRatifierService guardRatifierService = Services.getService(GuardRatifierService.class);
        guardRatifierService.validate(email);

        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        Email ogEmail = ogEmailStorage.getByEmail(email);
        if (ogEmail != null) {
            return getAsciiArmoredCurrentPublicKey(ogEmail.getUserId(), ogEmail.getContextId());
        }

        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgp.keymanagment.GuardUserKeyService#getAsciiArmoredPrivateKeyById()
     */
    @Override
    public String getAsciiArmoredPrivateKeyById(int userId, int contextId, long keyId, String password) throws OXException {
        int shard = getShardId(userId, contextId);
        GuardKeys keyPair = Services.getService(KeyTableStorage.class).getKeyForUserById(userId, contextId, shard, Long.toString(keyId));
        if (keyPair != null) {
            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            return pgpKeyService.exportPGPPrivateKey(keyPair, password);
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_KEY_ID_ERROR.create(keyId);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgp.keymanagment.GuardUserKeyService#getAsciiArmoredPublicKeyById(int, int, long)
     */
    @Override
    public String getAsciiArmoredPublicKeyById(int userId, int contextId, long keyId) throws OXException {
        int shard = getShardId(userId, contextId);
        GuardKeys keyPair = Services.getService(KeyTableStorage.class).getKeyForUserById(userId, contextId, shard, Long.toString(keyId));
        if (keyPair != null) {
            return keyPair.exportPGPPublicKeyRing();
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_KEY_ID_ERROR.create(keyId);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.pgp.keymanagment.GuardUserKeyService#getKeyImage(int, int, long)
     */
    @Override
    public byte[] getKeyImage(int userId, int contextId, long keyId, int attributeId) throws OXException {

        //Searching the user's own key for the fiven keyId
        int shard = getShardId(userId, contextId);
        GuardKeys keyPair = Services.getService(KeyTableStorage.class).getKeyForUserById(userId, contextId, shard, Long.toString(keyId));
        if (keyPair == null) {
            //If keyId does not belong to the user's own keys, we search for public/external uploaded keys
            keyPair = PGPPublicHandler.getUploadedPublic(userId, contextId, LongUtil.longToHexStringTruncated(keyId));
        }

        if (keyPair != null) {
            PGPUserAttributeSubpacketVector imageAttribute = new PGPAttributeExtractor().getAttributeById(keyPair.getPublicKeyRing(), attributeId);
            if (imageAttribute != null) {
                return imageAttribute.getImageAttribute().getImageData();
            }
            return null;
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_KEY_ID_ERROR.create(keyId);
        }
    }
}
