/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp.decryption;

import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.oxapi.OxCookie;

/**
 * Service for decrypting E-Mails
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public interface GuardEMailDecryptionService {

    enum AttachmentSaveMode{

        /**
         * Storing attachment as plain text
         */
        PLAIN,

        /**
         * Storing attachment encrypted
         */
        ENCRYPTED
    };

    /**
     * Decrypts an MIME E-mail
     * @param cookie The UI cookie used for accessing the OX backend
     * @param session The session
     * @param userAgent The user agent
     * @param emailFolder The folder containing the email
     * @param emailId The ID of the mail
     * @param userId The ID of the user
     * @param contextId The context ID
     * @param password The password used for decryption
     * @return The decryption result
     * @throws Exception
     */
    PGPDecryptionResult decryptMimeEmail(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int userId, int contextId, String password) throws OXException;

    /**
     * Decrypts an Inline E-mail
     * @param cookie The UI cookie used for accessing the OX backend
     * @param session The session
     * @param userAgent the user agent
     * @param emailFolder the folder containing the E-Mail
     * @param emailId The ID of the email
     * @param attachmentId The ID of the attachment representing the Mail body
     * @param userId the ID of the user
     * @param contextId The context ID
     * @param password The password used for decryption
     * @return The decryption result
     * @throws Exception
     */
    PGPDecryptionResult decryptInlineEmail(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int userId, int contextId, String password) throws OXException;

    /**
     * Decrypts an E-Mail MIME attachment specified by it's name
     * @param cookie The UI cookie used for authentication
     * @param session The session
     * @param userAgent the user agent
     * @param emailFolder The folder containing the email
     * @param emailId The ID of the mail containing the attachment
     * @param attachmentName The name of the attachment to decrypt
     * @param userId The ID of the user
     * @param contextId The context ID
     * @param password The password used for decrypting
     * @return The decryption result
     * @throws Exception
     */
    PGPDecryptionResult decryptMimeAttachmentByName(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, String attachmentName, int userId, int contextId, String password) throws OXException;

    /**
     * Decrypts an E-Mail MIME attachment specified by it's ID
     * @param cookie The UI cookie used for authentication
     * @param session The session
     * @param userAgent the user agent
     * @param emailFolder The folder containing the email
     * @param emailId The ID of the mail containing the attachment
     * @param attachmentId The ID of the attachment to decrypt
     * @param userId The ID of the user
     * @param contextId The context ID
     * @param password The password used for decrypting
     * @return The decryption result
     * @throws Exception
     */
    PGPDecryptionResult decryptMimeAttachmentById(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, String attachmentId, int userId, int contextId, String password) throws OXException;

    /**
     * Decrypts an E-Mail inline attachment
     * @param cookie The UI cookie used for authentication
     * @param session The session
     * @param userAgent the user agent
     * @param emailFolder The folder containing the email
     * @param emailId The ID of the mail containing the attachment
     * @param attachmentName The name of the attachment to decrypt
     * @param userId The ID of the user
     * @param contextId The context ID
     * @param password The password used for decrypting
     * @return The decryption result
     * @throws Exception
     */
    PGPDecryptionResult decryptInlineAttachment(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int userId, int contextId, String password) throws OXException;

    /**
     * Decrypts an E-Mail MIME attachment and saves it to OX Drive
     * @param cookie The UI cookie used for authentication
     * @param session The session
     * @param userAgent The user agent
     * @param emailFolder The folder containing the email
     * @param emailId The ID of the mail containing the attachment
     * @param attachmentId The ID of the attachment
     * @param folderId The id of the target folder
     * @param fileName The filename
     * @param description The file description
     * @param userId the ID of the user
     * @param contextId The context ID
     * @param password The password used for decryption
     * @param saveMode the mode controlls the way hot to store the attachment
     * @return The JsonObject representing the new file created
     * @throws Exception
     */
    JsonObject decryptMimeAttachmentToDrive(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, String attachmentName, int folderId, String fileName, String description, int userId, int contextId, String password,AttachmentSaveMode saveMode) throws OXException;

    /**
     * Decrypts an E-Mail inline attachment and saves it to OX Drive
     * @param cookie The UI cookie used for authentication
     * @param session The session
     * @param userAgent The user agent
     * @param emailFolder The folder containing the email
     * @param emailId The ID of the mail containing the attachment
     * @param attachmentId The ID of the attachment
     * @param folderId The id of the target folder
     * @param fileName The filename
     * @param description The file description
     * @param userId the ID of the user
     * @param contextId The context ID
     * @param password The password used for decryption
     * @param saveMode the mode controls the way hot to store the attachment
     * @return The JsonObject representing the new file created
     * @throws Exception
     */
    JsonObject decryptInlineAttachmentToDrive(OxCookie cookie, String session, String userAgent, String emailFolder, String emailId, int attachmentId, int folderId, String fileName, String description, int userId, int contextId, String password,AttachmentSaveMode saveMode) throws OXException;
}