
package com.openexchange.guard.pgp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.translation.GuardTranslationService;

public class PGPMailCreator {

    private static final Logger LOG = LoggerFactory.getLogger(PGPMailCreator.class);

    public MimeMultipart getPGPMime(final byte[] data, final List<String> languages) throws Exception {
        final MimeMultipart pgp = new MimeMultipart("encrypted; protocol=\"application/pgp-encrypted\"");
        final BodyPart ver = new MimeBodyPart();
        ver.setDescription("PGP/MIME version identification");
        ver.setContent("Version: 1\r\n", "application/pgp-encrypted");
        ver.setHeader("Content-Transfer-Encoding", "7bit");
        pgp.addBodyPart(ver);
        final BodyPart pt = new MimeBodyPart();
        pt.setFileName("encrypted.asc");
        pt.setDescription("OpenPGP encrypted message");
        pt.setDisposition("inline");
        pt.setContent(new String(data, "UTF-8"), "application/octet-stream");
        pt.setHeader("Content-Transfer-Encoding", "7bit");

        pgp.addBodyPart(pt);

        final List<HelpAttach> attachments = getHelp(languages);
        for (int i = 0; i < attachments.size(); i++) {
            final BodyPart help = new MimeBodyPart();
            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
            help.setFileName(translationService.getTranslation(GuardPGPMessages.HELP, attachments.get(i).getLanguage()) + ".txt");
            help.setContent(attachments.get(i).getTranslated(), "text/plain; charset=utf-8");
            help.setDisposition("inline");
            pgp.addBodyPart(help);
        }

        return (pgp);

    }

    /**
     * Get the mime section of a message (remove header)
     * Returns starting at content-type header
     * 
     * @param msg
     * @return
     * @throws IOException
     * @throws MessagingException
     */
    public String getBody(final Message msg) throws IOException, MessagingException {
        final ByteArrayOutputStream output = new ByteArrayOutputStream();
        msg.writeTo(output);
        final byte[] data = output.toByteArray();
        String mime = new String(data, "UTF-8");
        if (mime.contains("Content-Type")) {
            mime = mime.substring(mime.indexOf("Content-Type"));
        }
        output.close();
        return (mime);
    }

    private List<HelpAttach> getHelp(final List<String> languages) {
        final List<HelpAttach> helpfiles = new ArrayList<HelpAttach>();
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String externalReaderPath = guardConfigService.getProperty(GuardProperty.externalReaderPath);
            String oxURL = guardConfigService.getProperty(GuardProperty.externalEmailURL);

            for (int i = 0; i < languages.size(); i++) {
                GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                final String lang = translationService.getAvailableCode(languages.get(i));
                String data = translationService.getHelpFile(lang);
                if (data != null) {
                	data = data.replace("%guest%", "https://" + externalReaderPath);
                	data = data.replace("%webmail%", "https://" + oxURL);
                    if (!contains(helpfiles, lang)) {
                        final HelpAttach att = new HelpAttach(lang, data);
                        helpfiles.add(att);
                    }
                }
            }
        } catch (final Exception e) {
            LOG.error("Problem loading help file ", e);
        }
        return (helpfiles);
    }

    private boolean contains(final List<HelpAttach> list, final String lang) {
        for (final HelpAttach att : list) {
            if (att.getLanguage().equals(lang)) {
                return (true);
            }
        }
        return (false);
    }
}
