/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.osgi;

import org.osgi.service.http.HttpService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.GuardMaintenanceService;
import com.openexchange.guard.bad.GuardBadStorage;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.cron.Jobs;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.encryption.EncryptedItemsStorage;
import com.openexchange.guard.guestupgrade.GuestUpgradeService;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.handler.impl.ResponseHandlerImpl;
import com.openexchange.guard.internal.GuardMaintenanceServiceImpl;
import com.openexchange.guard.internal.GuardVersion;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.GuardMasterKeyService;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.PublicKeyService;
import com.openexchange.guard.keys.storage.DeletedKeysStorage;
import com.openexchange.guard.keys.storage.KeyCacheStorage;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.OGPGPKeysStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.mailvelope.MailvelopeStorage;
import com.openexchange.guard.mailvelope.internal.MailvelopeStorageImpl;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.oxapi.pooling.HttpConnectionPoolService;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.servlets.GuardServlets;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.settings.GlobalSettingsStorage;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.update.GuardUpdateService;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.startup.SignalHttpApiAvailabilityService;
import com.openexchange.timer.TimerService;

/**
 * The activator for the OX Guard bundle
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuardServiceActivator extends HousekeepingActivator {

    private static final Logger LOG = LoggerFactory.getLogger(GuardServiceActivator.class);

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { HttpService.class, SignalHttpApiAvailabilityService.class,
            GuardConfigurationService.class, GuardMasterKeyService.class, TimerService.class, GuardTranslationService.class,
            GuardRatifierService.class, GuardCipherFactoryService.class, GuardDatabaseService.class, GuardSessionService.class,
            KeyCacheStorage.class, PublicKeyService.class, MailCreatorService.class, DNSService.class, PGPKeyService.class, GlobalSettingsStorage.class,
            GuardNotificationService.class, GuardBadStorage.class, GuardKeyService.class, EmailStorage.class,
            GuardShardingService.class, KeyTableStorage.class, PGPKeysStorage.class, EncryptedItemsStorage.class, OGPGPKeysStorage.class,
            DeletedKeysStorage.class, Storage.class, GuardUpdateService.class, GuestUpgradeService.class, HKPClientService.class };
    }


    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        LOG.info("Starting bundle {}", context.getBundle().getSymbolicName());
        Services.setServiceLookup(this);
        
        //Loading the Guard version information
        GuardVersion.loadGuardVersion();

        // Register the maintenance service
        GuardMaintenanceService maintenanceService = new GuardMaintenanceServiceImpl();
        registerService(GuardMaintenanceService.class, maintenanceService);
        trackService(GuardMaintenanceService.class);

        trackService(HttpConnectionPoolService.class);

        registerService(ResponseHandler.class, new ResponseHandlerImpl());
        trackService(ResponseHandler.class);

        // Maybe stuff Mailvelope storage and services to a different bundle?
        registerService(MailvelopeStorage.class, new MailvelopeStorageImpl());
        trackService(MailvelopeStorage.class);

        openTrackers();

        Services.getService(GuardUpdateService.class).createMasterKeysIfNecessary();

        new Jobs(); // Initialise the cron jobs

        //Registering the OX Guard servlets
        GuardServlets.registerServlets(getService(HttpService.class));
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#stopBundle()
     */
    @Override
    protected void stopBundle() throws Exception {
        LOG.info("Stopping bundle {}", context.getBundle().getSymbolicName());

        HttpConnectionPoolService poolService = Services.optService(HttpConnectionPoolService.class);
        if (poolService != null) {
            poolService.closeAll();
        }

        unregisterService(GuardMasterKeyService.class);

        //Unregistering the OX Guard servlets
        GuardServlets.unregisterServlets(getService(HttpService.class));
    }
}
